/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdbgui.widgets;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.eclipse.swt.SWT;
import org.eclipse.swt.events.FocusEvent;
import org.eclipse.swt.events.FocusListener;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.Text;

import alma.acs.logging.AcsLogLevel;
import alma.acs.tmcdb.BACIPropArchMech;
import alma.acs.tmcdb.BACIProperty;
import alma.acs.tmcdb.FaultFamily;
import alma.acs.tmcdb.FaultMember;
import alma.obops.dam.utils.ConversationTokenProvider.ConversationToken;
import alma.obops.tmcdbgui.TmcdbGui;
import alma.obops.tmcdbgui.editors.TmcdbObjectEditor;
import alma.obops.tmcdbgui.utils.conversation.AlarmConversationUtils;

/**
 * Composite used to display & edit baci properties.
 * @author sharring, rtobar
 */
public class BACIPropertyEditingComposite extends Composite implements SelectionListener {

	private static final String NOTIFICATION_CHANNEL = BACIPropArchMech.NOTIFICATION_CHANNEL.toString(); 
	private static final String MONITOR_COLLECTOR = BACIPropArchMech.MONITOR_COLLECTOR.toString();
	public static final String WIDGET_ENABLER = "widgetEnabler";

	private static class WidgetAssociation {
		public Button checkBox;
		public Control widget;
	}

	// Label texts
	private static final String MIN_TIMER_TRIGGER_TEXT = "Min timer trigger";
	private static final String PROPERTY_NAME_TEXT = "Property name";
	private static final String DESCRIPTION_TEXT = "Description";
	private static final String MAX_VALUE_TEXT = "Max value";
	private static final String ALARM_HIGH_ON_TEXT = "Alarm high on";
	private static final String MIN_VALUE_TEXT = "Min value";
	private static final String ALARM_HIGH_OFF_TEXT = "Alarm high off";
	private static final String DEFAULT_VALUE_TEXT = "Default value";
	private static final String ALARM_FAULT_FAMILY_TEXT = "Alarm fault family";
	private static final String UNITS_TEXT = "Units";
	private static final String ALARM_FAULT_MEMBER_TEXT = "Alarm fault member";
	private static final String FORMAT_TEXT = "Format";
	private static final String ALARM_LEVEL_TEXT = "Alarm level";
	private static final String RESOLUTION_TEXT = "Resolution";
	private static final String ALARM_LOW_ON_TEXT = "Alarm low on";
	private static final String STATES_DESCRIPTION_TEXT = "States description";
	private static final String ALARM_LOW_OFF_TEXT = "Alarm low off";
	private static final String ALARM_ON_TEXT = "Alarm on";
	private static final String WHEN_SET_TEXT = "When set";
	private static final String WHEN_CLEARED_TEXT = "When cleared";
	private static final String ALARM_OFF_TEXT = "Alarm off";
	private static final String ARCHIVE_DELTA_TEXT = "Archive delta";
	private static final String ARCHIVE_DELTA_PERCENT_TEXT = "Archive delta percent";
	private static final String ALARM_TIMER_TRIGGER_TEXT = "Alarm timer trigger";
	private static final String ARCHIVE_MAX_INT_TEXT = "Archive max int";
	private static final String ARCHIVE_MIN_INT_TEXT = "Archive min int";
	private static final String ARCHIVE_SUPPRESS_TEXT = "Archive suppress";
	private static final String ARCHIVE_MECHANISM_TEXT = "Archive mechanism";
	private static final String DEFAULT_TIMER_TRIGGER_TEXT = "Default timer trigger";
	private static final String ARCHIVE_PRIORITY_TEXT = "Archive priority";
	private static final String MIN_DELTA_TRIGGER_TEXT = "Min delta trigger";
	private static final String BIT_DESCRIPTION_TEXT = "Bit description";
	private static final String GRAPH_MIN_TEXT = "Graph min";
	private static final String CONDITION_TEXT = "Condition";
	private static final String GRAPH_MAX_TEXT = "Graph max";
	private static final String DATA_TEXT = "Data";
	private static final String INITIALIZE_DEVIO_TEXT = "Initialize devio";
	private static final String MIN_STEP_TEXT = "Min step";

	private static final int TEXT_WIDTH = 150;

	// Object property names
	private static final String WHEN_SET = "whenSet";
	private static final String WHEN_CLEARED = "whenCleared";
	private static final String UNITS = "units";
	private static final String STATES_DESCRIPTION = "statesDescription";
	private static final String RESOLUTION = "resolution";
	public  static final String PROPERTY_NAME = "propertyName";
	private static final String MIN_VALUE = "min_value";
	private static final String MIN_TIMER_TRIG = "min_timer_trig";
	private static final String MIN_STEP = "min_step";
	private static final String MIN_DELTA_TRIG = "min_delta_trig";
	private static final String MAX_VALUE = "max_value";
	private static final String INITIALIZE_DEVIO = "initialize_devio";
	private static final String GRAPH_MIN = "graph_min";
	private static final String GRAPH_MAX = "graph_max";
	private static final String FORMAT = "format";
	private static final String DESCRIPTION = "description";
	private static final String DEFAULT_VALUE = "default_value";
	private static final String DEFAULT_TIMER_TRIG = "default_timer_trig";
	private static final String DATA = "data";
	private static final String CONDITION = "condition";
	private static final String BIT_DESCRIPTION = "bitDescription";
	private static final String ARCHIVE_PRIORITY = "archive_priority";
	private static final String ARCHIVE_MIN_INT = "archive_min_int";
	private static final String ARCHIVE_MAX_INT = "archive_max_int";
	private static final String ARCHIVE_MECHANISM = "archive_mechanism";
	private static final String ARCHIVE_SUPPRESS = "archive_suppress";
	private static final String ARCHIVE_DELTA = "archive_delta";
	private static final String ARCHIVE_DELTA_PERCENT = "archive_delta_percent";
	private static final String ALARM_TIMER_TRIG = "alarm_timer_trig";
	private static final String ALARM_ON = "alarm_on";
	private static final String ALARM_OFF = "alarm_off";
	private static final String ALARM_LOW_ON = "alarm_low_on";
	private static final String ALARM_LOW_OFF = "alarm_low_off";
	private static final String ALARM_LEVEL = "alarm_level";
	private static final String ALARM_HIGH_ON = "alarm_high_on";
	private static final String ALARM_HIGH_OFF = "alarm_high_off";
	private static final String ALARM_FAULT_MEMBER = "alarm_fault_member";
	private static final String ALARM_FAULT_FAMILY = "alarm_fault_family";

	private BACIProperty baciProp;
	private TmcdbObjectEditor editor;
	private Listener listener;
	private boolean includeCheckboxes;
	private Map<String, FaultFamily> faultFamilyMap = new HashMap<String, FaultFamily>();
	private Map<String, WidgetAssociation> associationMap = new HashMap<String, WidgetAssociation>();
	
	/**
	 * Constructor.
	 * 
	 * @param parent parent composite.
	 * @param style swt style.
	 * @param baciProperty the baci property that we're editing. 
	 * @param listener the listener that property changes will 
	 *        be published to when fields of the baci property 
	 *        are changed.
	 */
	public BACIPropertyEditingComposite(TmcdbObjectEditor editor, Composite parent, int style,
			BACIProperty baciProperty, Listener listener) 
	{
		super(parent, style);
		this.editor = editor;
		this.listener = listener;
		this.includeCheckboxes = editor == null;
		this.associationMap = new HashMap<String, WidgetAssociation>();
		initialize(baciProperty);
		if( editor != null )
			bindDataToGuiWidgets();
	}
	
	@Override
	public boolean setFocus() {
		return widgetForProperty(PROPERTY_NAME).setFocus();
	}
	
	private void bindDataToGuiWidgets()
	{
		// combo widgets
		editor.bind( ALARM_FAULT_FAMILY, widgetForProperty(ALARM_FAULT_FAMILY) );
		editor.bind( ALARM_FAULT_MEMBER, widgetForProperty(ALARM_FAULT_MEMBER) );
		
		// text widgets
		editor.bind( PROPERTY_NAME, widgetForProperty(PROPERTY_NAME) );
		editor.bind( DESCRIPTION, widgetForProperty(DESCRIPTION) );
		editor.bind( ALARM_HIGH_ON, widgetForProperty(ALARM_HIGH_ON) );
		editor.bind( ALARM_HIGH_OFF, widgetForProperty(ALARM_HIGH_OFF) );
		editor.bind( INITIALIZE_DEVIO, widgetForProperty(INITIALIZE_DEVIO) );
		editor.bind( ALARM_LEVEL, widgetForProperty(ALARM_LEVEL) );
		editor.bind( ALARM_LOW_ON, widgetForProperty(ALARM_LOW_ON) );
		editor.bind( ALARM_LOW_OFF, widgetForProperty(ALARM_LOW_OFF) );
		editor.bind( ALARM_ON, widgetForProperty(ALARM_ON) );
		editor.bind( ALARM_OFF, widgetForProperty(ALARM_OFF) );
		editor.bind( ALARM_TIMER_TRIG, widgetForProperty(ALARM_TIMER_TRIG) );
		editor.bind( ARCHIVE_DELTA, widgetForProperty(ARCHIVE_DELTA) );
		editor.bind( ARCHIVE_DELTA_PERCENT, widgetForProperty(ARCHIVE_DELTA_PERCENT) );
		editor.bind( ARCHIVE_MAX_INT, widgetForProperty(ARCHIVE_MAX_INT) );
		editor.bindBACIPropArchMechCombo( ARCHIVE_MECHANISM, (Combo)(widgetForProperty(ARCHIVE_MECHANISM)) );
		editor.bind( ARCHIVE_SUPPRESS, widgetForProperty(ARCHIVE_SUPPRESS) );
		editor.bind( ARCHIVE_MIN_INT, widgetForProperty(ARCHIVE_MIN_INT) );
		editor.bind( ARCHIVE_PRIORITY, widgetForProperty(ARCHIVE_PRIORITY) );
		editor.bind( BIT_DESCRIPTION, widgetForProperty(BIT_DESCRIPTION) );
		editor.bind( CONDITION, widgetForProperty(CONDITION) );
		editor.bind( DATA, widgetForProperty(DATA) );
		editor.bind( MAX_VALUE, widgetForProperty(MAX_VALUE) );
		editor.bind( MIN_VALUE, widgetForProperty(MIN_VALUE) );
		editor.bind( DEFAULT_VALUE, widgetForProperty(DEFAULT_VALUE));
		editor.bind( MIN_STEP, widgetForProperty(MIN_STEP));
		editor.bind( MIN_TIMER_TRIG, widgetForProperty(MIN_TIMER_TRIG) );
		editor.bind( DEFAULT_TIMER_TRIG, widgetForProperty(DEFAULT_TIMER_TRIG));
		editor.bind( MIN_DELTA_TRIG, widgetForProperty(MIN_DELTA_TRIG));
		editor.bind( WHEN_SET, widgetForProperty(WHEN_SET) );
		editor.bind( WHEN_CLEARED, widgetForProperty(WHEN_CLEARED));
		editor.bind( GRAPH_MAX, widgetForProperty(GRAPH_MAX));
		editor.bind( GRAPH_MIN, widgetForProperty(GRAPH_MIN) );
		editor.bind( UNITS, widgetForProperty(UNITS));
		editor.bind( FORMAT, widgetForProperty(FORMAT));
		editor.bind( RESOLUTION, widgetForProperty(RESOLUTION) );
		editor.bind( STATES_DESCRIPTION, widgetForProperty(STATES_DESCRIPTION));
	}

	private void createWidget( String propertyName, String labelText, int style, Class<?> clazz) {

		GridData gd;
		Button checkBox = null;

		// Initial checkbox, if needed
		if( includeCheckboxes ) {
			checkBox = new Button(this, SWT.CHECK);
			checkBox.setData(WIDGET_ENABLER, true);
		}

		// Create the label
		Label label = new Label(this, SWT.NONE);
		label.setText(labelText);

		// Create the widget
		gd = new GridData(); gd.widthHint = TEXT_WIDTH;
		Control widget = null;
		if( clazz.equals(Text.class) ) {
			widget = new Text(this, style);
			if( listener != null )
				widget.addListener(SWT.KeyUp, listener);
		}
		else if( clazz.equals(Combo.class) && !propertyName.equals(ARCHIVE_MECHANISM)) {
			widget = new Combo(this, style);
			if( listener != null )
				widget.addListener(SWT.Selection, listener);
		}
		else if( clazz.equals(Combo.class) && propertyName.equals(ARCHIVE_MECHANISM)) {
			widget = new Combo(this, style);
			((Combo)widget).setData("type", ARCHIVE_MECHANISM);
			if( listener != null )
				widget.addListener(SWT.Selection, listener);
		}
		else if( clazz.equals(Button.class) ) {
			widget = new Button(this, style);
			if( listener != null )
				widget.addListener(SWT.Selection, listener);
		}
		else {
			throw new RuntimeException("Unsupported class: " + clazz);
		}
		widget.setLayoutData(gd);

		// Index the widget with it's property name (also the checkbox, if needed)
		WidgetAssociation association = new WidgetAssociation();
		if( includeCheckboxes && null != checkBox) {
			checkBox.addSelectionListener(this);
			if( listener != null )
				checkBox.addListener(SWT.Selection, listener);
			association.checkBox = checkBox;
		}
		association.widget = widget;
		associationMap.put(propertyName, association);

	}

	private void initialize(BACIProperty baciProperty) 
	{
		createWidget( PROPERTY_NAME, PROPERTY_NAME_TEXT, SWT.BORDER, Text.class);
		createWidget( DESCRIPTION, DESCRIPTION_TEXT, SWT.BORDER, Text.class);
		createWidget( MAX_VALUE, MAX_VALUE_TEXT, SWT.BORDER, Text.class);
		createWidget( ALARM_HIGH_ON, ALARM_HIGH_ON_TEXT, SWT.BORDER, Text.class);
		createWidget( MIN_VALUE, MIN_VALUE_TEXT, SWT.BORDER, Text.class);
		createWidget( ALARM_HIGH_OFF, ALARM_HIGH_OFF_TEXT, SWT.BORDER, Text.class);
		createWidget( DEFAULT_VALUE, DEFAULT_VALUE_TEXT, SWT.BORDER, Text.class);
		createWidget( ALARM_FAULT_FAMILY, ALARM_FAULT_FAMILY_TEXT, SWT.READ_ONLY | SWT.NONE, Combo.class);
		createWidget( UNITS, UNITS_TEXT, SWT.BORDER, Text.class);
		createWidget( ALARM_FAULT_MEMBER, ALARM_FAULT_MEMBER_TEXT, SWT.READ_ONLY | SWT.NONE, Combo.class);
		createWidget( FORMAT, FORMAT_TEXT, SWT.BORDER, Text.class);
		createWidget( ALARM_LEVEL, ALARM_LEVEL_TEXT, SWT.BORDER, Text.class);
		createWidget( RESOLUTION, RESOLUTION_TEXT, SWT.BORDER, Text.class);
		createWidget( ALARM_LOW_ON, ALARM_LOW_ON_TEXT, SWT.BORDER, Text.class);
		createWidget( STATES_DESCRIPTION, STATES_DESCRIPTION_TEXT, SWT.BORDER, Text.class);
		createWidget( ALARM_LOW_OFF, ALARM_LOW_OFF_TEXT, SWT.BORDER, Text.class);
		createWidget( WHEN_SET, WHEN_SET_TEXT, SWT.BORDER, Text.class);
		createWidget( ALARM_ON, ALARM_ON_TEXT, SWT.BORDER, Text.class);
		createWidget( WHEN_CLEARED, WHEN_CLEARED_TEXT, SWT.BORDER, Text.class);
		createWidget( ALARM_OFF, ALARM_OFF_TEXT, SWT.BORDER, Text.class);
		createWidget( ARCHIVE_DELTA, ARCHIVE_DELTA_TEXT, SWT.BORDER, Text.class);
		createWidget( ARCHIVE_DELTA_PERCENT, ARCHIVE_DELTA_PERCENT_TEXT, SWT.BORDER, Text.class);
		createWidget( ALARM_TIMER_TRIG, ALARM_TIMER_TRIGGER_TEXT, SWT.BORDER, Text.class);
		createWidget( ARCHIVE_MAX_INT, ARCHIVE_MAX_INT_TEXT, SWT.BORDER, Text.class);
		createWidget( MIN_TIMER_TRIG, MIN_TIMER_TRIGGER_TEXT, SWT.BORDER, Text.class);
		createWidget( ARCHIVE_MIN_INT, ARCHIVE_MIN_INT_TEXT, SWT.BORDER, Text.class);
		createWidget( DEFAULT_TIMER_TRIG, DEFAULT_TIMER_TRIGGER_TEXT, SWT.BORDER, Text.class);
		createWidget( ARCHIVE_PRIORITY, ARCHIVE_PRIORITY_TEXT, SWT.BORDER, Text.class);
		createWidget( MIN_DELTA_TRIG, MIN_DELTA_TRIGGER_TEXT, SWT.BORDER, Text.class);
		createWidget( ARCHIVE_MECHANISM, ARCHIVE_MECHANISM_TEXT, SWT.READ_ONLY | SWT.NONE, Combo.class);
		createWidget( MIN_STEP, MIN_STEP_TEXT, SWT.BORDER, Text.class);
		createWidget( ARCHIVE_SUPPRESS, ARCHIVE_SUPPRESS_TEXT, SWT.CHECK, Button.class);
		createWidget( GRAPH_MIN, GRAPH_MIN_TEXT, SWT.BORDER, Text.class);
		createWidget( BIT_DESCRIPTION, BIT_DESCRIPTION_TEXT, SWT.BORDER, Text.class);
		createWidget( GRAPH_MAX, GRAPH_MAX_TEXT, SWT.BORDER, Text.class);
		createWidget( CONDITION, CONDITION_TEXT, SWT.BORDER, Text.class);
		createWidget( DATA, DATA_TEXT, SWT.BORDER, Text.class);
		createWidget( INITIALIZE_DEVIO, INITIALIZE_DEVIO_TEXT, SWT.CHECK, Button.class);

		// If including checkboxes, set everything to disabled at the beginning
		if( includeCheckboxes ) {
			for (WidgetAssociation association: associationMap.values()) {
				association.checkBox.setSelection(false);
				association.widget.setEnabled(false);
			}
		}

		GridLayout gridLayout = new GridLayout();
		gridLayout.numColumns = ( includeCheckboxes ? 6 : 4);
		gridLayout.horizontalSpacing = 10;
		this.setLayout(gridLayout);
 
		setBACIProperty(baciProperty);
		populateArchiveMechanismCombo();
		createAlarmFaultMemberCombo();
		createAlarmLevelCombo();

		this.pack();
	}

	/**
	 * Populates the faultFamily combo with all the fault family names available for the Configuration in which the BACI property resides.
	 * It also selects the current value of the BACI property's fault family (empty string if current value is not valid).
	 * 
	 * This method should be called whenever the underlying BACI property is changed (e.g., when initializing the editor)
	 */
	private void populateFaultFamilyCombo() 
	{
		if( baciProp == null )
			return;

		FaultFamily[] alarmFaultFamilies = null;
		try {
			alarmFaultFamilies = AlarmConversationUtils.getInstance().getFaultFamilies(baciProp.getComponent().getConfiguration(), ConversationToken.CONVERSATION_COMPLETED);
		} catch (Exception e) {
			TmcdbGui.getLogger().log(AcsLogLevel.WARNING, "Couldn't populate fault families combobox, editor won't be complete: " + e.getMessage());
			alarmFaultFamilies = new FaultFamily[0];
		}

		// Save in the map the current available Fault Families
		faultFamilyMap.clear();
		List<String> itemList = new ArrayList<String>();
		itemList.add("");
		for(FaultFamily faultFamily : alarmFaultFamilies) {
			itemList.add(faultFamily.getFamilyName());
			faultFamilyMap.put(faultFamily.getFamilyName(), faultFamily);
		}

		// Set the selection to the current value of the BACI property's fault family
		String[] items = itemList.toArray(new String[0]);
		FaultFamily currentSelection = faultFamilyMap.get(baciProp.getAlarm_fault_family());
		String currentSelectionString = (currentSelection == null) ? "" : currentSelection.getFamilyName();

		Combo alarmFaultFamilyCombo = (Combo)widgetForProperty(ALARM_FAULT_FAMILY);
		alarmFaultFamilyCombo.setItems(items);
		alarmFaultFamilyCombo.select(alarmFaultFamilyCombo.indexOf(currentSelectionString));
	}

	private void populateArchiveMechanismCombo()
	{
		Combo archiveMechanismCombo = (Combo)widgetForProperty(ARCHIVE_MECHANISM);
		archiveMechanismCombo.setItems(new String[] {MONITOR_COLLECTOR, NOTIFICATION_CHANNEL});

		// Not necessarily enabled if we're using this composite in a bulk-changes wizard page
		// archiveMechanismCombo.setEnabled(true);
	}
	
	/**
	 * This method initializes alarmFaultMemberCombo	
	 *
	 */
	private void createAlarmFaultMemberCombo() 
	{

		Combo alarmFaultFamilyCombo = (Combo)widgetForProperty(ALARM_FAULT_FAMILY);
		alarmFaultFamilyCombo.addSelectionListener(new SelectionListener() {

			@Override
			public void widgetDefaultSelected(SelectionEvent arg0) {
				widgetSelected(arg0);
			}

			@Override
			public void widgetSelected(SelectionEvent arg0) {
				Combo combo = (Combo)widgetForProperty(ALARM_FAULT_FAMILY);
				populateFaultMemberCombo();	
				baciProp.setAlarm_fault_family(combo.getText());
			}
			
		});
		
		alarmFaultFamilyCombo.addFocusListener(new FocusListener() {

			@Override
			public void focusGained(FocusEvent arg0) {
				// TODO Auto-generated method stub
			}

			@Override
			public void focusLost(FocusEvent arg0) {
				populateFaultMemberCombo();
			}
		});
	}

	private void populateFaultMemberCombo()
	{
		Combo alarmFaultFamilyCombo = (Combo)widgetForProperty(ALARM_FAULT_FAMILY);
		Combo alarmFaultMemberCombo = (Combo)widgetForProperty(ALARM_FAULT_MEMBER);

		String faultFamilyString = alarmFaultFamilyCombo.getText();
		FaultFamily faultFamily = faultFamilyMap.get(faultFamilyString);

		alarmFaultMemberCombo.removeAll();
		if(null != faultFamily)
		{
			try {
				faultFamilyMap.put(faultFamilyString, AlarmConversationUtils.getInstance().hydrateFaultFamily(faultFamily));
			} catch (Exception e) {
				// TODO Error handling
				e.printStackTrace();
				return;
			}
			Set<FaultMember> alarmFaultMembers = faultFamily.getFaultMembers();
			
			List<String> itemList = new ArrayList<String>();
			itemList.add("");
			for(FaultMember faultMember : alarmFaultMembers)
				itemList.add(faultMember.getMemberName());
			
			String[] items = itemList.toArray(new String[0]);
			alarmFaultMemberCombo.setItems(items);
			alarmFaultMemberCombo.setEnabled(true);
		} 
		else {
			alarmFaultMemberCombo.setItems(new String[] {""});
			alarmFaultMemberCombo.select(alarmFaultMemberCombo.indexOf(""));
			alarmFaultMemberCombo.setEnabled(false);
		}

		alarmFaultMemberCombo.addSelectionListener(new SelectionListener() {

			@Override
			public void widgetDefaultSelected(SelectionEvent e) {
				widgetSelected(e);
			}

			@Override
			public void widgetSelected(SelectionEvent e) {
				Combo combo = (Combo)widgetForProperty(ALARM_FAULT_MEMBER);
				baciProp.setAlarm_fault_member(combo.getText());
			}
		});
	}
	
	/**
	 * This method initializes alarmLevelCombo	
	 */
	private void createAlarmLevelCombo() {
				
		// TODO: determine if we really even need the fault level item in baci property; discuss with Heiko!
		// for now, it is just permanently disabled in the GUI.
		
//		alarmFaultFamilyCombo.addSelectionListener(new SelectionListener() 
//		{
//
//			@Override
//			public void widgetDefaultSelected(SelectionEvent arg0) {
//				// TODO Auto-generated method stub
//			}
//
//			@Override
//			public void widgetSelected(SelectionEvent arg0) {
//				alarmLevelCombo.setEnabled(true);
//			}
//			
//		});
	}

	public void setBACIProperty(BACIProperty property) {

		baciProp = property;
		populateFaultFamilyCombo();
		populateFaultMemberCombo();

	}

	@Override
	public void widgetDefaultSelected(SelectionEvent e) {
		widgetSelected(e);
	}

	@Override
	public void widgetSelected(SelectionEvent e) {

		Control w = null;
		for(WidgetAssociation association: associationMap.values()) {
			if( association.checkBox == e.widget )
				w = association.widget;
			
		}
		if( w == null )
			return;
		w.setEnabled(!w.getEnabled());
		
	}

	public Control widgetForProperty(String propertyName) {
		return associationMap.get(propertyName).widget;
	}

	public String[] getEnabledProperties() {

		List<String> result = new ArrayList<String>();
		for (String propName : associationMap.keySet()) {
			if( associationMap.get(propName).widget.isEnabled() )
				result.add(propName);
		}

		return result.toArray(new String[0]);
	}

	public Object[] getValuesForEnabledProperties() {

		List<Object> result = new ArrayList<Object>();
		for (String propName : associationMap.keySet()) {
			Control widget = associationMap.get(propName).widget;
			if( widget.isEnabled() ) {
				if( widget instanceof Text )
					result.add(getObjectSafeValueForProperty(propName, ((Text)widget).getText()));
				else if( widget instanceof Button )
					result.add(((Button)widget).getSelection());
				else if( widget instanceof Combo ) {
					Combo c = (Combo)widget;
					result.add( getObjectSafeValueForProperty(propName, c.getItems()[c.getSelectionIndex()]) );
				}
			}
		}

		return result.toArray();
	}

	private Object getObjectSafeValueForProperty(String propName, String value) {

		Object o = null;

		// Booleans
		if( propName.equals(ARCHIVE_SUPPRESS) ||
		    propName.equals(INITIALIZE_DEVIO) )
			return Boolean.parseBoolean(value);

		// Integers
		if( propName.equals(ARCHIVE_PRIORITY) ||
		    propName.equals(ALARM_LEVEL) )
			return Integer.parseInt(value);

		// Doubles
		if( propName.equals(ARCHIVE_MIN_INT) ||
		    propName.equals(ARCHIVE_MAX_INT) ||
		    propName.equals(DEFAULT_TIMER_TRIG) ||
		    propName.equals(MIN_TIMER_TRIG) ||
		    propName.equals(MIN_DELTA_TRIG) ||
		    propName.equals(GRAPH_MIN) ||
		    propName.equals(GRAPH_MAX) ||
		    propName.equals(MIN_STEP) ||
		    propName.equals(ARCHIVE_DELTA) ||
		    propName.equals(ARCHIVE_DELTA_PERCENT) ||
		    propName.equals(ALARM_LOW_ON) ||
		    propName.equals(ALARM_LOW_OFF) ||
		    propName.equals(ALARM_HIGH_ON) ||
		    propName.equals(ALARM_HIGH_OFF) ||
		    propName.equals(ALARM_TIMER_TRIG) ||
		    propName.equals(MIN_VALUE) ||
		    propName.equals(MAX_VALUE) )
			return Double.parseDouble(value);


		if( propName.equals(PROPERTY_NAME) ||
		    propName.equals(DESCRIPTION) ||
		    propName.equals(FORMAT) ||
		    propName.equals(UNITS) ||
		    propName.equals(RESOLUTION) ||
		    propName.equals(DEFAULT_VALUE) ||
		    propName.equals(BIT_DESCRIPTION) ||
		    propName.equals(WHEN_SET) ||
		    propName.equals(WHEN_CLEARED) ||
		    propName.equals(STATES_DESCRIPTION) ||
		    propName.equals(CONDITION) ||
		    propName.equals(ALARM_ON) ||
		    propName.equals(ALARM_OFF) ||
		    propName.equals(ALARM_FAULT_FAMILY) ||
		    propName.equals(ALARM_FAULT_MEMBER) ||
		    propName.equals(DATA) )
			return value;

		// archive mechanism
		if(propName.equals(ARCHIVE_MECHANISM)) {
			return BACIPropArchMech.valueOfForEnum(value);
		}


		return o;
	}
}
