/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdbgui.widgets;

import java.text.DecimalFormat;
import java.util.Calendar;
import java.util.Date;
import java.util.Set;

import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.CLabel;
import org.eclipse.swt.events.KeyEvent;
import org.eclipse.swt.events.KeyListener;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Font;
import org.eclipse.swt.graphics.FontData;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.ImageData;
import org.eclipse.swt.graphics.RGB;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.DateTime;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;

import alma.obops.tmcdb.alarms.ui.utils.RcpUtils;
import alma.obops.tmcdbgui.utils.GuiUtils;
import alma.obops.tmcdbgui.utils.TmcdbConstants;
import alma.obops.tmcdbgui.views.providers.CoordinateRow;
import alma.obops.tmcdbgui.widgets.support.DirtyListener;
import alma.obops.tmcdbgui.widgets.support.StatusListener;
import alma.obops.tmcdbgui.wizards.support.IntegerStringVerifyListener;
import alma.obops.tmcdbgui.wizards.support.VerifyDecimalListener;
import alma.tmcdb.domain.Antenna;
import alma.tmcdb.domain.AntennaType;
import alma.tmcdb.domain.BaseElement;
import alma.tmcdb.domain.BaseElementType;
import alma.tmcdb.domain.Coordinate;
import alma.tmcdb.domain.HwConfiguration;

/**
 * Widget for antenna attributes; can be used in different contexts such as the add antenna wizard or the antenna editor.
 * @author sharring
 */
public class AntennaAttributesComposite extends StatusPublishingComposite 
{
	private static Image BLANK_IMAGE;
	private static final int TEXT_WIDTH_SMALL = 50;
	private static final int TEXT_WIDTH_COORDINATES = PadAttributesComposite.COORDINATES_TEXT_WIDTH;
	public static final String ANTENNA_ALREADY_EXISTS = "Antenna already exists: prefix + number must be unique";
	private static final String BASELINE_CORRELATOR_INPUT = "Baseline correlator input:";
	private static final String ACA_CORRELATOR_INPUT = "ACA correlator input:";
	private static final String POSITION = "Position (m)";
	private static final String OFFSET = "Offset (m):";
	private static final String LO_OFFSETTING =  "LO offsetting:";
	private static final String WALSH_SEQUENCE = "Walsh sequence:";
	public static final String OFFSET_FORMAT = "##0.######E0"; 
	public static final String COORDINATE_FORMAT = "#######0.########";
	
	private String errorMessage;
    private Set<BaseElement> baseElements;
    private Text coordX, coordY, coordZ, offsetX, loOffsetting, walshSequence, baselineCorrelatorInput, acaCorrelatorInput;
	private Combo typeCombo, diameterCombo, namePrefixCombo, nameNumberCombo;
    private DateTime commissionDate;
    private CLabel coordXLabel, coordYLabel, coordZLabel, acaCorrLabel, baselineCorrLabel, loOffsettingLabel, offsetXLabel, walshSequenceLabel, commissionDateLabel;
    private HwConfiguration configuration;
    private AntennaAttributesSelectionListener updateCombosBasedOnSelectionsSL;
    private KeyListener kListener;
    private Antenna antenna;

    /**
	 * Constructor.
	 * @param parent the parent composite.
	 * @param style the style for the widget.
	 * @param antenna the antenna that is being "dealt" with.
	 */
	public AntennaAttributesComposite(Composite parent, int style, Antenna antenna, StatusListener statusListener, DirtyListener dirtyListener)
	{
		super(parent, style);
		this.setAntenna(antenna);
		this.addStatusListener(statusListener);
		this.addDirtyListener(dirtyListener);
		createControl();
	}
	
	/**
	 * Constructor.
	 * @param parent the parent composite.
	 * @param style the style for the widget.
	 */
	public AntennaAttributesComposite(Composite parent, int style, StatusListener statusListener, HwConfiguration config) 
	{
		this(parent, style, null, statusListener, null);
		this.setConfiguration(config);
	}
	
	/**
	 * Constructor.
	 * @param parent the parent composite.
	 * @param style the style for the widget.
	 */
	public AntennaAttributesComposite(Composite parent, int style, StatusListener statusListener) 
	{
		this(parent, style, null, statusListener, null);
	}
	
	/**
	 * Constructor.
	 * @param parent the parent composite.
	 * @param style the style for the widget.
	 */
	public AntennaAttributesComposite(Composite parent, int style, DirtyListener dirtyListener) 
	{
		this(parent, style, null, null, dirtyListener);
	}
	
	public AntennaAttributesComposite(Composite parent, int style) 
	{
		this(parent, style, (StatusListener)null);
	}

	/**
     * Setter for the antenna antenna being edited.
     * @param the antenna begin edited.
     */
	public void setAntenna(Antenna antenna) 
	{
		if(null == antenna || this.antenna == antenna) 
		{
			return;
		}
		
		this.antenna = antenna;
		if(antenna.getAntennaType().equals(AntennaType.ACA)) {
			updateFieldsForACAType();
		} else if(antenna.getAntennaType().equals(AntennaType.AEC)) {
			updateFieldsForAECType();
		} else if(antenna.getAntennaType().equals(AntennaType.VA)) {
			updateFieldsForVAorLAType();
		}
		
		DecimalFormat formatter = new DecimalFormat(COORDINATE_FORMAT);

		// position
		
		if(null != antenna.getPosition()) {
			String formattedX = formatter.format(antenna.getPosition().getX());
			String formattedY = formatter.format(antenna.getPosition().getY());
			String formattedZ = formatter.format(antenna.getPosition().getZ());
			this.coordX.setText(formattedX);
			this.coordY.setText(formattedY);
			this.coordZ.setText(formattedZ);
		} else {
			this.coordX.setText("");
			this.coordY.setText("");
			this.coordZ.setText("");
		}
		
		// offset
		if(null != antenna.getOffset()) {
			formatter = new DecimalFormat(OFFSET_FORMAT);
			String formattedOffset = formatter.format(antenna.getOffset().getX());
			this.offsetX.setText(formattedOffset);
		} else {
			this.offsetX.setText("");
		}
		
		// commissionDate
		Calendar cal = Calendar.getInstance();
		cal.setTimeInMillis(antenna.getCommissionDate());
		this.commissionDate.setDate(cal.get(Calendar.YEAR), cal.get(Calendar.MONTH), cal.get(Calendar.DAY_OF_MONTH));

		// name
		String prefix = antenna.getName().substring(0, 2).toUpperCase();
		String suffix = antenna.getName().substring(2);
		this.namePrefixCombo.select(this.namePrefixCombo.indexOf(prefix));
		this.nameNumberCombo.select(this.nameNumberCombo.indexOf(suffix));
		
		// diameter
		if(antenna.getDiameter().equals(new Double(12.0))) {
			this.diameterCombo.select(this.diameterCombo.indexOf(TmcdbConstants._12M));			
		} else {
			this.diameterCombo.select(this.diameterCombo.indexOf(TmcdbConstants._7M));
		}

		// type
		this.typeCombo.select(this.typeCombo.indexOf(antenna.getAntennaType().name()));
		
		// looffsetting index
		this.loOffsetting.setText(Integer.toString(antenna.getLoOffsettingIndex()));
		
		// walsh sequence
		this.walshSequence.setText(Integer.toString(antenna.getWalshSeq()));
		
		// baseline CAI
		if(null != antenna.getCaiBaseline()) {
			this.baselineCorrelatorInput.setText(antenna.getCaiBaseline().toString());
		} else {
			this.baselineCorrelatorInput.setText("");
		}
		
		// ACA CAI
		if(null != antenna.getCaiAca()) {
			this.acaCorrelatorInput.setText(antenna.getCaiAca().toString());
		} else {
			this.acaCorrelatorInput.setText("");
		}
		
		this.configuration = antenna.getConfiguration();
	}
	
	 /**
     * Setter for the configuration associated with the antenna being edited.
     * @param config the owning configuration for the antenna begin edited.
     */
	public void setConfiguration(HwConfiguration config) {
		this.configuration = config;
	}

	/**
	 * Getter for the new antenna's type.
	 * @return the type of the new antenna.
	 */
	public AntennaType getAntennaType() {
		AntennaType retVal = null;
		String currentAntennaTypeSelected = (typeCombo.getSelectionIndex() == -1) ? 
				null : typeCombo.getItem(typeCombo.getSelectionIndex());
		if(null != currentAntennaTypeSelected) {
		   retVal = AntennaType.valueOf(currentAntennaTypeSelected);
		}
		return retVal;
	}

	/**
	 * Getter for the new antenna's commission date.
	 * @return the new antenna's commission date.
	 */
	public Date getCommissionDate() 
	{
		Date retVal = null;
		
		Calendar cal = Calendar.getInstance();
		cal.set(Calendar.YEAR, this.commissionDate.getYear());
		cal.set(Calendar.MONTH, this.commissionDate.getMonth());
		cal.set(Calendar.DAY_OF_MONTH, this.commissionDate.getDay());
		cal.set(Calendar.HOUR_OF_DAY, this.commissionDate.getHours());
		cal.set(Calendar.MINUTE, this.commissionDate.getMinutes());
		cal.set(Calendar.SECOND, this.commissionDate.getSeconds());
		retVal = cal.getTime();
		
		return retVal;
	}

	/**
	 * Getter for the new antenna's diameter.
	 * @return the new antenna's diameter.
	 */
	public Double getDiameter() 
	{
		Double retVal;
		
		String value = diameterCombo.getItem(diameterCombo.getSelectionIndex());
		retVal = Double.valueOf((value.replace("m", "")).trim());
		
		return retVal;
	}

	/**
	 * Getter for the new antenna's name.
	 * @return the new antenna's name.
	 */
	public String getAntennaName() 
	{
		String retVal = null;
		
		String prefix = namePrefixCombo.getItem(namePrefixCombo.getSelectionIndex());
		String number = nameNumberCombo.getItem(nameNumberCombo.getSelectionIndex());
		
		retVal = prefix + number;
		
		return retVal;
	}
	
	/**
	 * Getter for the antenna number.
	 * @return the antenna number.
	 */
	public String getAntennaNumber()
	{
		return (nameNumberCombo.getSelectionIndex() == -1) ? null : nameNumberCombo.getItem(nameNumberCombo.getSelectionIndex());
	}

	/**
	 * Getter for the error message, if any.
	 * @return the error message, or null if none.
	 */
	public String getErrorMessage() 
	{
		return this.errorMessage;
	}
	
	/**
	 * Getter for the new antenna's offset (x offset; y, z are ignored).
	 * @return the new antenna's offset.
	 */
	public Coordinate getOffset() 
	{
		Coordinate retVal = new Coordinate();
		
		// offset only means an X value, despite being stored as X,Y,Z coordinate
		retVal.setX(Double.valueOf(offsetX.getText()));
		
		// thus, we set y & z to 0.0
		retVal.setY(0.0);
		retVal.setZ(0.0);
		
		return retVal;
	}
	
	/**
	 * Getter for the new antenna's position.
	 * @return the position of the new antenna.
	 */
	public Coordinate getPosition() {
		Coordinate retVal = new Coordinate();
		if(coordX.getText() != null && coordX.getText().trim().length() > 0) {
		   retVal.setX(Double.valueOf(coordX.getText()));
		}
		if(coordY.getText() != null && coordY.getText().trim().length() > 0) {
			retVal.setY(Double.valueOf(coordY.getText()));			
		}
		if((coordZ.getText() != null && coordZ.getText().trim().length() > 0)) {
		   retVal.setZ(Double.valueOf(coordZ.getText()));
		}
		return retVal;
	}
	
	private void addKeyListeners()
	{
		kListener = new KeyListener() 
		{
			@Override
			public void keyPressed(KeyEvent arg0) {
				setDirty(true);
			}

			@Override
			public void keyReleased(KeyEvent arg0) {
				isComplete();
			}	
		};
		coordX.addKeyListener(kListener); 
		coordY.addKeyListener(kListener); 
		coordZ.addKeyListener(kListener); 
		offsetX.addKeyListener(kListener); 
		loOffsetting.addKeyListener(kListener);
		walshSequence.addKeyListener(kListener);
		baselineCorrelatorInput.addKeyListener(kListener);
		acaCorrelatorInput.addKeyListener(kListener);
	}
	
	private void addSelectionListeners() {
		updateCombosBasedOnSelectionsSL = new AntennaAttributesSelectionListener();
		typeCombo.addSelectionListener( updateCombosBasedOnSelectionsSL );
        diameterCombo.addSelectionListener(updateCombosBasedOnSelectionsSL);
        namePrefixCombo.addSelectionListener(updateCombosBasedOnSelectionsSL);
		nameNumberCombo.addSelectionListener(updateCombosBasedOnSelectionsSL);
	}
	
	private void createControl() 
	{
		Image img = RcpUtils.getImage("icons/blank12x12.png");
		ImageData ideaData = img.getImageData();
		int whitePixel = ideaData.palette.getPixel(new RGB(255,255,255));
		ideaData.transparentPixel = whitePixel;
		BLANK_IMAGE = new Image(getDisplay(),ideaData);
		
		GridLayout layout = new GridLayout();
		layout.numColumns = 1;  // label, entry
		setLayout( layout );

		createAntennaPrefixAndNumberControl();
		createAntennaTypeAndDiameterControl();
		createAntennaPositionControl();
		createCommissionDateControl();
		
		addSelectionListeners();
		addKeyListeners();
	}

	private void createAntennaPositionControl() 
	{
		Composite positionAndOffset = new Composite(this, SWT.NONE);
		
		// Antenna position
		GridLayout gridLayout = new GridLayout();
		gridLayout.numColumns = 3;
		positionAndOffset.setLayout(gridLayout);
		GridData gridData = new GridData();
		gridData.horizontalSpan = 1;
		
		Group coordinates = new Group(positionAndOffset, SWT.NONE);
		GridLayout coordLayout = new GridLayout();
		coordLayout.numColumns = 2;
		coordinates.setLayout(coordLayout);
		coordinates.setText(POSITION);
		coordinates.setLayoutData(gridData);

		coordXLabel = new CLabel(coordinates, SWT.NONE);
		coordXLabel.setImage(BLANK_IMAGE);
		coordXLabel.setText("x:");
		coordX = new Text(coordinates, SWT.SINGLE | SWT.BORDER);
		GridData gridDataX = new GridData();
		gridDataX.horizontalSpan = 1;
		gridDataX.widthHint = TEXT_WIDTH_COORDINATES;
		gridDataX.grabExcessHorizontalSpace = true;
		coordX.setLayoutData(gridDataX);
		coordX.addVerifyListener(new VerifyDecimalListener());
		
		coordYLabel = new CLabel(coordinates, SWT.NONE);
		coordYLabel.setImage(BLANK_IMAGE);
		coordYLabel.setText("y:");
		coordY = new Text(coordinates, SWT.SINGLE | SWT.BORDER);
		coordY.setLayoutData(gridDataX);
		coordY.addVerifyListener(new VerifyDecimalListener());

		coordZLabel = new CLabel(coordinates, SWT.NONE);
		coordZLabel.setImage(BLANK_IMAGE);
		coordZLabel.setText("z:");
		coordZ = new Text(coordinates, SWT.SINGLE | SWT.BORDER);
		coordZ.setLayoutData(gridDataX);
		coordZ.addVerifyListener(new VerifyDecimalListener());
		
		createAdditionalControls(positionAndOffset);
	}
	
	private void createAdditionalControls(Composite parent)
	{
		Composite additionalComposite = new Composite(parent, SWT.NONE);
		GridLayout gridLayout = new GridLayout();
		gridLayout.numColumns = 4;
		additionalComposite.setLayout(gridLayout);
		GridData gd = new GridData();
		gd.widthHint = TEXT_WIDTH_SMALL;
		
		// Antenna offset
		offsetXLabel = new CLabel(additionalComposite, SWT.NONE);
		offsetXLabel.setImage(BLANK_IMAGE);
		offsetXLabel.setText(OFFSET);
		offsetX = new Text(additionalComposite, SWT.SINGLE | SWT.BORDER);
		offsetX.setLayoutData(gd);
		offsetX.addVerifyListener(new VerifyDecimalListener());
		offsetX.setEnabled(true);
		
		// LO offsetting
		loOffsettingLabel = new CLabel(additionalComposite, SWT.NONE);
		loOffsettingLabel.setImage(BLANK_IMAGE);
		loOffsettingLabel.setText(LO_OFFSETTING);
		loOffsetting = new Text(additionalComposite, SWT.SINGLE | SWT.BORDER);
		loOffsetting.setLayoutData(gd);
		loOffsetting.addVerifyListener(new IntegerStringVerifyListener(9));
		loOffsetting.setEnabled(true);
		
		// Baseline Correlator input
		baselineCorrLabel = new CLabel(additionalComposite, SWT.NONE);
		baselineCorrLabel.setImage(BLANK_IMAGE);
		baselineCorrLabel.setText(BASELINE_CORRELATOR_INPUT);
		baselineCorrLabel.setEnabled(GuiUtils.isGodUser());
        baselineCorrelatorInput = new Text(additionalComposite, SWT.SINGLE | SWT.BORDER);
        baselineCorrelatorInput.setLayoutData(gd);
        baselineCorrelatorInput.addVerifyListener(new IntegerStringVerifyListener(2));
        baselineCorrelatorInput.addSelectionListener(new SelectionListener() {

			@Override
			public void widgetDefaultSelected(SelectionEvent e) {
				setDirty(true);
			}

			@Override
			public void widgetSelected(SelectionEvent e) {
				setDirty(true);
			}
        	
        });
        baselineCorrelatorInput.setEnabled(GuiUtils.isGodUser());
        
		// ACA Correlator input
		acaCorrLabel = new CLabel(additionalComposite, SWT.NONE);
		acaCorrLabel.setImage(BLANK_IMAGE);
		acaCorrLabel.setText(ACA_CORRELATOR_INPUT);
		acaCorrLabel.setEnabled(GuiUtils.isGodUser());
        acaCorrelatorInput = new Text(additionalComposite, SWT.SINGLE | SWT.BORDER);
        acaCorrelatorInput.setLayoutData(gd);
        acaCorrelatorInput.addVerifyListener(new IntegerStringVerifyListener(2));
        acaCorrelatorInput.addSelectionListener(new SelectionListener() {

			@Override
			public void widgetDefaultSelected(SelectionEvent e) {
				setDirty(true);
			}

			@Override
			public void widgetSelected(SelectionEvent e) {
				setDirty(true);
			}
        	
        });
        acaCorrelatorInput.setEnabled(GuiUtils.isGodUser());
        
        // Walsh sequence
        walshSequenceLabel = new CLabel(additionalComposite, SWT.NONE);
        walshSequenceLabel.setImage(BLANK_IMAGE);
        walshSequenceLabel.setText(WALSH_SEQUENCE);
        walshSequence = new Text(additionalComposite, SWT.SINGLE | SWT.BORDER);
        walshSequence.setLayoutData(gd);
        walshSequence.addVerifyListener(new IntegerStringVerifyListener(3));	
        walshSequence.setEnabled(true);	

        new Label(additionalComposite, SWT.NONE).setText("");
        new Label(additionalComposite, SWT.NONE).setText("");

	}
	
	private void createAntennaPrefixAndNumberControl() 
	{
		Group namePrefixAndNumber = new Group(this, SWT.NONE);
		namePrefixAndNumber.setText("Name");
		GridLayout gridLayout = new GridLayout();
		gridLayout.numColumns = 4;
		namePrefixAndNumber.setLayout(gridLayout);
		GridData gridData = new GridData();
		gridData.horizontalSpan = 4;
		namePrefixAndNumber.setLayoutData(gridData);

		GridData gd;
		// Antenna name prefix
		Label lName = new Label( namePrefixAndNumber, SWT.NULL );
		lName.setText( "Prefix" );
		namePrefixCombo = new Combo( namePrefixAndNumber, SWT.READ_ONLY );
		gd = new GridData();
		namePrefixCombo.setItems(TmcdbConstants.NAME_PREFIX_ARRAY );
		namePrefixCombo.setLayoutData( gd );
		namePrefixCombo.setEnabled(GuiUtils.isGodUser());
		
		// Antenna number
		Label lNumber = new Label( namePrefixAndNumber, SWT.NULL );
		lNumber.setText( "Number" );
		nameNumberCombo = new Combo( namePrefixAndNumber, SWT.READ_ONLY );
		gd = new GridData();
		nameNumberCombo.setItems(TmcdbConstants.DA_NUMBERS);
		nameNumberCombo.setLayoutData( gd );
		nameNumberCombo.setEnabled(GuiUtils.isGodUser());
	}
	
	/** @return <code>true</code> when all required fields are populated */
	public boolean isComplete() 
	{	
		boolean complete = 
			(namePrefixCombo.getSelectionIndex() != -1) &&
			(nameNumberCombo.getSelectionIndex() != -1) &&
			!antennaExistsInConfig() &&
			(typeCombo.getSelectionIndex() != -1 ) && 
			(coordX.getText().length() > 0 && coordY.getText().length() > 0 && coordZ.getText().length() > 0) &&
			(offsetX.getText().length() > 0) &&
			(diameterCombo.getSelectionIndex() != -1) &&
			(loOffsetting.getText().length() > 0) &&
			(walshSequence.getText().length() > 0);

		notifyListenersOfCompletion(complete);
		return complete;
	}
	
	private boolean antennaExistsInConfig()
	{
		boolean retVal = false;

		if(null == baseElements) 
		{
			this.baseElements = configuration.getBaseElements();
		} 	

		try	{
			retVal = foundCorrespondingBaseElement();
		}
		catch(Exception e) {
			throw new RuntimeException("Unable to get the base elements for the configuration", e);
		}
		
		if(retVal == true) {
			this.setStatus(ANTENNA_ALREADY_EXISTS);
		} else {
			this.setStatus(null);
		}
		return retVal;
	}

	private boolean foundCorrespondingBaseElement() {
		boolean retVal = false;
		for(BaseElement be: baseElements) 
		{
			if(be.getType().equals(BaseElementType.Antenna) && be.getName().equals(getAntennaName())) 
			{
				retVal = true;
				break;
			}
		}
		return retVal;
	}

	private void createAntennaTypeAndDiameterControl() 
	{
		Composite typeAndDiameter = new Composite(this, SWT.NONE);
		GridLayout gridLayout = new GridLayout();
		gridLayout.numColumns = 4;
		typeAndDiameter.setLayout(gridLayout);
		GridData gridData = new GridData();
		gridData.horizontalSpan = 4;
		typeAndDiameter.setLayoutData(gridData);
		
		// Antenna type
		Label lType = new Label( typeAndDiameter, SWT.NULL );
		lType.setText( "Type" );
		typeCombo = new Combo( typeAndDiameter, SWT.READ_ONLY );
		GridData gd = new GridData();
		gd.grabExcessHorizontalSpace = false;
		typeCombo.setLayoutData(gd);
		String [] antennaValues = new String[AntennaType.values().length];
		int counter = 0;
		for(AntennaType aType : AntennaType.values() ) {
			antennaValues[counter++] = aType.name();
		}
		typeCombo.setItems( antennaValues );
		typeCombo.setEnabled(GuiUtils.isGodUser());
		
		// Antenna diameter
		Label lDiameter = new Label( typeAndDiameter, SWT.NULL );
		lDiameter.setImage(BLANK_IMAGE);
		lDiameter.setText( "Diameter" );
		diameterCombo = new Combo( typeAndDiameter, SWT.READ_ONLY );
		gd = new GridData();
		gd.grabExcessHorizontalSpace = false;
		diameterCombo.setLayoutData(gd);
		diameterCombo.setItems( TmcdbConstants.ANTENNA_DIAMETER_ACA_ARRAY );
		diameterCombo.setEnabled(GuiUtils.isGodUser());
	}

	private void createCommissionDateControl() 
	{
		Composite commissionDateComposite = new Composite(this, SWT.NONE);
		GridLayout gridLayout = new GridLayout();
		gridLayout.numColumns = 4;
		commissionDateComposite.setLayout(gridLayout);
		GridData gridData = new GridData();
		gridData.horizontalSpan = 4;
		commissionDateComposite.setLayoutData(gridData);
		
		commissionDateLabel = new CLabel(commissionDateComposite, SWT.NONE);
		commissionDateLabel.setImage(BLANK_IMAGE);
        commissionDateLabel.setText("Commission date");
        commissionDateLabel.setEnabled(true);
        commissionDate = new DateTime(commissionDateComposite, SWT.DATE | SWT.MEDIUM);
        commissionDate.addSelectionListener(new SelectionListener() {

			@Override
			public void widgetDefaultSelected(SelectionEvent e) {
				setDirty(true);
			}

			@Override
			public void widgetSelected(SelectionEvent e) {
				setDirty(true);
			}
        	
        });
        commissionDate.setEnabled(true);
	}
	
	private void updateFieldsForDAPrefix() {
		nameNumberCombo.setItems(TmcdbConstants.DA_NUMBERS);
		typeCombo.select(typeCombo.indexOf(TmcdbConstants.AEC));
		diameterCombo.setEnabled(false);
		diameterCombo.select(diameterCombo.indexOf(TmcdbConstants._12M));
	}

	private void updateFieldsForDVorLAPrefix() {
		nameNumberCombo.setItems(TmcdbConstants.DV_NUMBERS);
		typeCombo.select(typeCombo.indexOf(TmcdbConstants.VA));
		diameterCombo.setEnabled(false);
		diameterCombo.select(diameterCombo.indexOf(TmcdbConstants._12M));
	}

	private void updateFieldsForPMPrefix() {
		nameNumberCombo.setItems(TmcdbConstants.PM_NUMBERS);
		typeCombo.select(typeCombo.indexOf(TmcdbConstants.ACA));
		diameterCombo.select(diameterCombo.indexOf(TmcdbConstants._12M));
		diameterCombo.setEnabled(false);
	}

	private void updateFieldsForCMPrefix() {
		nameNumberCombo.setItems(TmcdbConstants.CM_NUMBERS);
		typeCombo.select(typeCombo.indexOf(TmcdbConstants.ACA));
		diameterCombo.select(diameterCombo.indexOf(TmcdbConstants._7M));
		diameterCombo.setEnabled(false);
	}

	private void updateFieldsForVAorLAType() {
		nameNumberCombo.setItems(TmcdbConstants.DV_NUMBERS);
		String prefix = parsePrefix(antenna.getName());
		namePrefixCombo.select(namePrefixCombo.indexOf(prefix));
		diameterCombo.select(diameterCombo.indexOf(TmcdbConstants._12M));
		diameterCombo.setEnabled(false);
	}

	private String parsePrefix(String name) {
		String retVal = null;
		retVal = name.substring(0, 2);
		return retVal;
	}

	private void updateFieldsForAECType() {
		nameNumberCombo.setItems(TmcdbConstants.DA_NUMBERS);
		namePrefixCombo.select(namePrefixCombo.indexOf(TmcdbConstants.DA));
		diameterCombo.select(diameterCombo.indexOf(TmcdbConstants._12M));
		diameterCombo.setEnabled(false);
	}

	private void updateFieldsForACAType() {
		diameterCombo.setItems(TmcdbConstants.ANTENNA_DIAMETER_ACA_ARRAY);
		diameterCombo.setEnabled(true);
		if(diameterCombo.getText().equals(TmcdbConstants._12M)) {
			namePrefixCombo.select(namePrefixCombo.indexOf(TmcdbConstants.PM));
		} else {
			namePrefixCombo.select(namePrefixCombo.indexOf(TmcdbConstants.CM));
		}
		if(namePrefixCombo.getSelectionIndex() != -1) 
		{
			if(namePrefixCombo.getItem(namePrefixCombo.getSelectionIndex()).equals(TmcdbConstants.PM)) {
				nameNumberCombo.setItems(TmcdbConstants.PM_NUMBERS);
				diameterCombo.select(diameterCombo.indexOf(TmcdbConstants._12M));
				diameterCombo.setEnabled(false);
			} 
			else if(namePrefixCombo.getItem(namePrefixCombo.getSelectionIndex()).equals(TmcdbConstants.CM)) 
			{
				nameNumberCombo.setItems(TmcdbConstants.CM_NUMBERS);
				diameterCombo.select(diameterCombo.indexOf(TmcdbConstants._7M));
				diameterCombo.setEnabled(false);
			}
			else 
			{
				namePrefixCombo.select(namePrefixCombo.indexOf(TmcdbConstants.PM));
			}
		}
	}
	
	/**
	 * Private class to handle the dependencies (interrelatedness) between the widgets, so that invalid
	 * choices are not possible.
	 * 
	 * @author sharring
	 */
	private class  AntennaAttributesSelectionListener implements SelectionListener
	{
		private String previousNamePrefix = "";
		private String previousType = "";
		
		public void widgetDefaultSelected( SelectionEvent e ) 
		{
			if(e.widget == namePrefixCombo && namePrefixCombo.getSelectionIndex() != -1) {
				updateForNamePrefixSelection();
			}
			if(e.widget == typeCombo && typeCombo.getSelectionIndex() != -1) {
				updateForTypeSelection();
			}
			isComplete();
			setDirty(true);
		}

		public void widgetSelected( SelectionEvent e ) 
		{
			widgetDefaultSelected(e);
		}
		
		private void updateForNamePrefixSelection() {
			if(namePrefixCombo.getItem(namePrefixCombo.getSelectionIndex()).equals(TmcdbConstants.DA) && !previousNamePrefix.equals(TmcdbConstants.DA)) {
				updateFieldsForDAPrefix();
			} else if(namePrefixCombo.getItem(namePrefixCombo.getSelectionIndex()).equals(TmcdbConstants.DV) && !previousNamePrefix.equals(TmcdbConstants.DV)) {
				updateFieldsForDVorLAPrefix();
			} else if(namePrefixCombo.getItem(namePrefixCombo.getSelectionIndex()).equals(TmcdbConstants.LA) && !previousNamePrefix.equals(TmcdbConstants.LA)) {
				updateFieldsForDVorLAPrefix();
			}
			else if(namePrefixCombo.getItem(namePrefixCombo.getSelectionIndex()).equals(TmcdbConstants.PM) && !previousNamePrefix.equals(TmcdbConstants.PM)) {
				updateFieldsForPMPrefix();
			} else if(namePrefixCombo.getItem(namePrefixCombo.getSelectionIndex()).equals(TmcdbConstants.CM) && !previousNamePrefix.equals(TmcdbConstants.CM)) {
				updateFieldsForCMPrefix();
			}
			updatePreviousNameAndType();
			setDirty(true);
		}

		private void updatePreviousNameAndType() {
			previousNamePrefix = namePrefixCombo.getItem(namePrefixCombo.getSelectionIndex());
			previousType = typeCombo.getItem(typeCombo.getSelectionIndex());
		}
		
		private void updateForTypeSelection() 
		{
			if(typeCombo.getItem(typeCombo.getSelectionIndex()).equals(TmcdbConstants.VA) && !previousType.equals(TmcdbConstants.VA)) 
			{
				updateFieldsForVAorLAType();
			} 
			else if(typeCombo.getItem(typeCombo.getSelectionIndex()).equals(TmcdbConstants.AEC) && !previousType.equals(TmcdbConstants.AEC)) 
			{
				updateFieldsForAECType();
			} 
			else if(typeCombo.getItem(typeCombo.getSelectionIndex()).equals(TmcdbConstants.ACA) && !previousType.equals(TmcdbConstants.ACA)) 
			{
				updateFieldsForACAType(); 
			}
			updatePreviousNameAndType();
			setDirty(true);
		}
	}

	public Integer getLoOffsetting() 
	{
		Integer retVal = null;

		if(null != loOffsetting.getText() && loOffsetting.getText().length() > 0) {
			retVal = Integer.valueOf(loOffsetting.getText());
		}

		return retVal;
	}

	public Integer getCaiBaseline() 
	{
		Integer retVal = null;

		if(null != baselineCorrelatorInput.getText() && baselineCorrelatorInput.getText().length() > 0) {
			retVal = Integer.valueOf(baselineCorrelatorInput.getText());
		}

		return retVal;
	}
	
	/**
	 * @return
	 */
	public Integer getCaiAca() {
		Integer retVal = null;
		if(null != acaCorrelatorInput.getText() && acaCorrelatorInput.getText().length() > 0) {
			retVal = Integer.valueOf(acaCorrelatorInput.getText());
		}
		return retVal;
	}

	public Integer getWalshSequence() 
	{
		Integer retVal = null;

		if(null != walshSequence.getText() && walshSequence.getText().length() > 0) {
			retVal = Integer.valueOf(walshSequence.getText());
		}

		return retVal;
	}

	/**
	 * Used to disable all of the controls (e.g. for use in read-only settings like history browsing). 
	 */
	public void makeReadOnly() {
		coordX.setEnabled(false);
		coordY.setEnabled(false);
		coordZ.setEnabled(false);
		offsetX.setEnabled(false);
		loOffsetting.setEnabled(false);
		walshSequence.setEnabled(false);
		baselineCorrelatorInput.setEnabled(false);
		acaCorrelatorInput.setEnabled(false);
		typeCombo.setEnabled(false);
		diameterCombo.setEnabled(false);
		namePrefixCombo.setEnabled(false);
		nameNumberCombo.setEnabled(false);
	    commissionDate.setEnabled(false);
	}

	public void emphasizeCaiAca() {
		FontData[] fD = acaCorrelatorInput.getFont().getFontData();
		fD[0].setStyle(SWT.BOLD);
		acaCorrelatorInput.setFont(new Font(getDisplay(),fD[0]));
		acaCorrLabel.setImage(CoordinateRow.CHANGED_IMAGE);
	}
	
	public void emphasizeCaiBaseline() {
		FontData[] fD = baselineCorrelatorInput.getFont().getFontData();
		fD[0].setStyle(SWT.BOLD);
		baselineCorrelatorInput.setFont(new Font(getDisplay(),fD[0]));
		baselineCorrLabel.setImage(CoordinateRow.CHANGED_IMAGE);
	}

	public void emphasizeCommissionDate() {
		FontData[] fD = commissionDate.getFont().getFontData();
		fD[0].setStyle(SWT.BOLD);
		commissionDate.setFont(new Font(getDisplay(),fD[0]));
		commissionDateLabel.setImage(CoordinateRow.CHANGED_IMAGE);
	}

	public void emphasizeOffset() {
		FontData[] fD = offsetX.getFont().getFontData();
		fD[0].setStyle(SWT.BOLD);
		offsetX.setFont(new Font(getDisplay(),fD[0]));
		offsetXLabel.setImage(CoordinateRow.CHANGED_IMAGE);
	}
	
	public void emphasizeWalshSequence() {
		FontData[] fD = walshSequence.getFont().getFontData();
		fD[0].setStyle(SWT.BOLD);
		walshSequence.setFont(new Font(getDisplay(),fD[0]));
		walshSequenceLabel.setImage(CoordinateRow.CHANGED_IMAGE);
	}
	
	public void emphasizeLoOffsetting() {
		FontData[] fD = offsetX.getFont().getFontData();
		fD[0].setStyle(SWT.BOLD);
		loOffsetting.setFont(new Font(getDisplay(),fD[0]));
		loOffsettingLabel.setImage(CoordinateRow.CHANGED_IMAGE);
	}

	public void emphasizePositionX() {
		FontData[] fD = coordX.getFont().getFontData();
		fD[0].setStyle(SWT.BOLD);
		coordX.setFont(new Font(getDisplay(),fD[0]));
		coordXLabel.setImage(CoordinateRow.CHANGED_IMAGE);
	}
	
	public void emphasizePositionY() {
		FontData[] fD = coordY.getFont().getFontData();
		fD[0].setStyle(SWT.BOLD);
		coordY.setFont(new Font(getDisplay(),fD[0]));
		coordYLabel.setImage(CoordinateRow.CHANGED_IMAGE);
	}
	
	public void emphasizePositionZ() {
		FontData[] fD = coordZ.getFont().getFontData();
		fD[0].setStyle(SWT.BOLD);
		coordZ.setFont(new Font(getDisplay(),fD[0]));
		coordZLabel.setImage(CoordinateRow.CHANGED_IMAGE);
	}
}
