/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdbgui.widgets;

import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;

import alma.acs.tmcdb.Component;
import alma.obops.tmcdbgui.utils.TmcdbConstants;
import alma.obops.tmcdbgui.utils.conversation.ComponentConversationUtils;
import alma.obops.tmcdbgui.widgets.support.DirtyListener;
import alma.tmcdb.domain.HwConfiguration;

/**
 * Common logic for all baseelements which may need component assignment capabilities.
 * @author sharring
 */
public abstract class AbstractComponentAssignmentComposite extends DirtyPublishingComposite 
{
	protected boolean isPopulated;
	protected static final String EMPTY_STRING = "";
	protected static final String UNABLE_TO_HYDRATE_COMPONENTS = "Unable to hydrate components";
	protected Map<String, Component> componentMap = new HashMap<String, Component>();
	protected Combo componentCombo;
	protected HwConfiguration configuration;
	protected String componentNamePlusPath;

	public AbstractComponentAssignmentComposite(Composite parent, int style, DirtyListener dirtyListener) 
	{
		super(parent, style);
		this.addDirtyListener(dirtyListener);
		createControl();
	}
	
	public void setConfiguration(HwConfiguration config)
	{
		if(this.configuration != config) 
		{
			this.configuration = config;
			populateCombo();
		}
	}
	
	/**
	 * Getter for the new antenna's component (ALMA/ACS/CORBA sw component).
	 * @return the new antenn's component.
	 */
	public Component getComponent() 
	{
		Component retVal = null;
		
		int selectionIndex = componentCombo.getSelectionIndex();
		if(selectionIndex != -1) {
			String selectedString = componentCombo.getItem(selectionIndex);
			retVal = componentMap.get(selectedString);
		} else {
			retVal = childGetComponent();
		}
		
		return retVal;
	}
	
	protected abstract Component childGetComponent();

	public boolean isComplete() {
		boolean retVal = (this.componentCombo.getSelectionIndex() != -1) ? true : false;
		return retVal;
	}
	
	public void addSelectionListener(SelectionListener listener) {
		this.componentCombo.addSelectionListener(listener);
	}
	
	public void removeSelectionListener(SelectionListener listener) {
		this.componentCombo.removeSelectionListener(listener);
	}
	
	private void createControl()
	{
		GridLayout layout = new GridLayout();
		layout.numColumns = 2;
		this.setLayout(layout);
		
		Label label = new Label(this, SWT.READ_ONLY);
		label.setText("Component");

		GridData gd = new GridData();
		gd.widthHint = 170;
		gd.heightHint = 25;
		this.componentCombo = new Combo(this, SWT.READ_ONLY);
		componentCombo.setLayoutData(gd);

		this.pack();
	}
	
	private void populateCombo()
	{
		if(null != configuration)
		{
			try {
				this.getShell().setCursor(this.getShell().getDisplay().getSystemCursor(SWT.CURSOR_WAIT));
				
				List<Component> filteredComponents = ComponentConversationUtils.getInstance().
				    findComponentByNamePrefixWithinConfiguration(getFilters(), configuration.getSwConfiguration());

				String[] componentChoices = new String[filteredComponents.size()];
				int counter = 0;
				for(Component component : filteredComponents) 
				{
					String pathSeparator = component.getPath().endsWith(TmcdbConstants.SLASH) ? EMPTY_STRING : TmcdbConstants.SLASH;
					String componentNameWithPath = component.getPath() + pathSeparator + component.getComponentName();
					componentChoices[counter++] = componentNameWithPath;
					componentMap.put(componentNameWithPath, component);
				}
				Arrays.sort(componentChoices);
				componentCombo.setItems( componentChoices );
				addSelectionListeners();
				setSelection();
			}
			catch (Exception ex) {
				throw new RuntimeException(UNABLE_TO_HYDRATE_COMPONENTS, ex);
			}
			finally {
				this.getShell().setCursor(null);
			}
		}
	}
	
	protected abstract String[] getFilters();

	private void addSelectionListeners()
	{
		SelectionListener dirtySelectionListener = new SelectionListener() {
			@Override
			public void widgetDefaultSelected(SelectionEvent e) {
				setDirty(true);
			}

			@Override
			public void widgetSelected(SelectionEvent e) {
				setDirty(true);
			}
		};
		
		this.addSelectionListener(dirtySelectionListener);
	}

	protected void setSelection() {
		if(null != componentNamePlusPath) {
			componentCombo.select(componentCombo.indexOf(componentNamePlusPath));
		}
	}

	public boolean isPopulated() {
		return this.isPopulated;
	}
}
