/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
/**
 * AbcdEditingSupport.java
 *
 * Copyright European Southern Observatory 2008
 */

package alma.obops.tmcdbgui.views.support;

import org.eclipse.jface.viewers.CellEditor;
import org.eclipse.jface.viewers.CheckboxCellEditor;
import org.eclipse.jface.viewers.EditingSupport;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Shell;

import alma.obops.tmcdb.alarms.ui.tree.helpers.ThreeColumnDomainObjectHelper;
import alma.obops.tmcdb.alarms.ui.utils.RcpUtils;
import alma.obops.tmcdbgui.views.providers.helpers.startup.BaseElementStartupHelper;
import alma.obops.tmcdbgui.views.providers.helpers.startup.StartupHelperFactory;
import alma.obops.tmcdbgui.views.providers.typedlists.LRUTypeRole;
import alma.tmcdb.domain.BaseElementStartup;

/**
 * Support editing of a TreeViewer column with BooleanCellEditors for
 * yes/no values.
 *
 * @author amchavan, Mar 11, 2009
 * 
 */



public class StartupScenarioEditingSupport extends EditingSupport {
    
    private CellEditor editor;
    private TreeViewer owningViewer;
    private FieldToEdit fieldToEdit;
    
    public enum FieldToEdit {
    	IS_STARTED,
    	IS_SIMULATED
    }
    
    /**
     * @param viewer
     */
    public StartupScenarioEditingSupport( TreeViewer viewer, FieldToEdit field ) {
        super( viewer );
        owningViewer = viewer;
        this.fieldToEdit = field;

//        editor = new BooleanCellEditor( viewer.getTree() );
        editor = new CheckboxCellEditor( viewer.getTree() );
    }

    /**
     * Decide if the Boolean editor should be displayed.
     * 
     * @return <code>true</code> if the input element is an LRUTypeRole,
     *         <code>false</code> for all other classes
     * 
     * @see org.eclipse.jface.viewers.EditingSupport#canEdit(java.lang.Object)
     */
    @Override
    protected boolean canEdit( Object element ) {
    	boolean retVal = ( (element instanceof LRUTypeRole) || (element instanceof BaseElementStartup)); 
        return retVal;
    }

    /**
     * @return A BooleanCellEditor
     * @see org.eclipse.jface.viewers.EditingSupport#getCellEditor(java.lang.Object)
     */
    @Override
    protected CellEditor getCellEditor( Object element ) {
        return editor;
    }

    /**
     * @return Whatever {@link LRUTypeRole#isStarted()} returns for the
     *         underlying LRUTypeRole.
     * 
     * @see org.eclipse.jface.viewers.EditingSupport#getValue(java.lang.Object)
     */
    @Override
    protected Object getValue( Object element ) 
    {
    	boolean retVal = false;
    	
    	if(element instanceof LRUTypeRole)
    	{
    		retVal = getValueForLruTypeRole((LRUTypeRole)element);
    	}
    	else if(element instanceof BaseElementStartup)
    	{
    		retVal = getValueForBaseElementStartup((BaseElementStartup)element);
    	}

        return retVal;
    }

	private boolean getValueForBaseElementStartup(BaseElementStartup beStartup) 
	{
		boolean retVal = false;
		ThreeColumnDomainObjectHelper helper = StartupHelperFactory.getInstance().getHelper(beStartup);
		BaseElementStartupHelper startupHelper = (BaseElementStartupHelper) helper;
		switch(fieldToEdit) 
		{
		case IS_STARTED:
			retVal = startupHelper.isStarted();
			break;
		case IS_SIMULATED:
			retVal = startupHelper.isSimulated();
			break;
		}
		return retVal;
	}

	private boolean getValueForLruTypeRole(LRUTypeRole lruTR) 
	{
		boolean retVal = false;

		switch(fieldToEdit) 
		{
		case IS_STARTED:
			retVal = lruTR.isStarted();
			break;
		case IS_SIMULATED:
			retVal = lruTR.isSimulated();
			break;
		}
		return retVal;
	}

    /**
     * Sets the underlying LRUTypeRole as "started" or "non-started".
     * @see LRUTypeRole#setStarted(boolean);
     * @see org.eclipse.jface.viewers.EditingSupport#setValue(java.lang.Object, java.lang.Object)
     */
    @Override
    protected void setValue( Object element, Object value ) 
    {
    	Boolean booleanValue = (Boolean) value;	
    	if(element instanceof LRUTypeRole)
    	{
    		setValueForLruTypeRole((LRUTypeRole)element, booleanValue, true);
    	}
    	else if(element instanceof BaseElementStartup)
    	{
    		setValueForBaseElementStartup((BaseElementStartup)element, booleanValue);
    	}
    }

	private void setValueForBaseElementStartup(BaseElementStartup beStartup,
			Boolean booleanValue) 
	{
		ThreeColumnDomainObjectHelper helper = StartupHelperFactory.getInstance().getHelper(beStartup);
		BaseElementStartupHelper startupHelper = (BaseElementStartupHelper) helper;
		
		try 
		{
			owningViewer.getTree().getShell().setCursor(owningViewer.getTree().getShell().getDisplay().getSystemCursor(SWT.CURSOR_WAIT));
			switch(fieldToEdit) 
			{
			case IS_STARTED:
				startupHelper.setStarted(booleanValue, false);
				break;
			case IS_SIMULATED:
				startupHelper.setSimulated(booleanValue);
				break;
			}
		}
		catch(Exception e) {
			e.printStackTrace();
			Shell shell = owningViewer.getControl().getShell();
			String title = "Persistence error";
			String message = e.getMessage();
			RcpUtils.errorMessage( e, shell, title, message );
		}
		finally {
			owningViewer.getTree().getShell().setCursor(null);
			owningViewer.refresh();
		}
	}

	private void setValueForLruTypeRole(LRUTypeRole lruTR, Boolean booleanValue, boolean commitChanges) {
		try {
			owningViewer.getTree().getShell().setCursor(owningViewer.getTree().getShell().getDisplay().getSystemCursor(SWT.CURSOR_WAIT));

			switch(fieldToEdit) 
			{
			case IS_STARTED:
				lruTR.setStarted( booleanValue, commitChanges );
				break;
			case IS_SIMULATED:
				lruTR.setSimulated( booleanValue, true );
				break;
			}

			getViewer().update( lruTR, null );
		}
		catch( Exception e ) {
			e.printStackTrace();
			Shell shell = owningViewer.getControl().getShell();
			String title = "Persistence error";
			String message = e.getMessage();
			RcpUtils.errorMessage( e, shell, title, message );
		} finally {
			owningViewer.getTree().getShell().setCursor(null);
		}
	}
}
