/*******************************************************************************
  * ALMA - Atacama Large Millimeter Array
  * Copyright (c) NRAO - National Radio Astronomy Observatory, 2012
  * (in the framework of the ALMA collaboration).
  * All rights reserved.
  *
  * This library is free software; you can redistribute it and/or
  * modify it under the terms of the GNU Lesser General Public
  * License as published by the Free Software Foundation; either
  * version 2.1 of the License, or (at your option) any later version.
  *
  * This library is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  * Lesser General Public License for more details.
  *
  * You should have received a copy of the GNU Lesser General Public
  * License along with this library; if not, write to the Free Software
  * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 
*******************************************************************************/
package alma.obops.tmcdbgui.views.support;

import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeSupport;

import alma.acs.tmcdb.AcsService;
import alma.acs.tmcdb.translator.TmcdbObject;

/**
 * Utility class to "wrap" an acs service so that the UI can treat it distinctly
 * from a regular AcsService object. This is used, e.g., to allow us to have two separate
 * ways to view the AcsService objects: 1) by browsing to the computer containing the
 * service, and 2) by viewing all the services as a top level list. See for example
 * COMP-6846. There are several ways in which we want to treat these wrapped objects
 * distinctly/differently from a non-wrapped ACS service object: 1) we don't want to show
 * children below the wrapped objects; 2) we want to display the labels for them differently;
 * 3) we will require that when a user creates a new object from the list of services in the 
 * main tree (as opposed to the list of services beneath a computer) that they select a computer;
 * 4) we want the services list in the main tree to get data from the configuration, not from the computer, 
 * and so forth. By creating a 'wrapper' object we are able to differentiate these cases more easily.
 * 
 * @author sharring
 */
public class AcsServiceWrapper implements TmcdbObjectWrapper, PropertyChangeListener 
{
	private static final String SERVICE_INSTANCE_NAME = "serviceInstanceName";
	private static final String SERVICE_TYPE = "serviceType";
	
	private AcsService service;
	
	public AcsServiceWrapper(AcsService service) 
	{ 
		this.service = service;
		service.addPropertyChangeListener(SERVICE_INSTANCE_NAME, this);
		service.addPropertyChangeListener(SERVICE_TYPE, this);
	}
	
	public AcsService getAcsService() { return service; }
	public void setAcsService(AcsService service) { this.service = service; }

	protected PropertyChangeSupport propertyChangeSupport = new PropertyChangeSupport(this);

	public void addPropertyChangeListener(String propertyName, PropertyChangeListener listener) {
		propertyChangeSupport.addPropertyChangeListener(propertyName, listener);
	}

	public void removePropertyChangeListener(String propertyName, PropertyChangeListener listener) {
		propertyChangeSupport.removePropertyChangeListener(propertyName, listener);
	}
	
	public TmcdbObject getWrappedObject() {
		return service;
	}

	@Override
	public void propertyChange(PropertyChangeEvent evt) {
		PropertyChangeEvent wrappedEvent = new PropertyChangeEvent(this, evt.getPropertyName(), evt.getOldValue(), evt.getNewValue());
		propertyChangeSupport.firePropertyChange(wrappedEvent);
	}
}
