/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
/**
 * LRUTypeRole.java
 *
 * Copyright European Southern Observatory 2008
 */

package alma.obops.tmcdbgui.views.providers.typedlists;

import java.lang.reflect.InvocationTargetException;
import java.util.List;
import java.util.Set;

import org.eclipse.jface.dialogs.MessageDialog;

import alma.acs.tmcdb.Component;
import alma.obops.dam.ServiceException;
import alma.obops.dam.utils.ConversationTokenProvider.ConversationToken;
import alma.obops.tmcdb.alarms.ui.utils.RcpUtils;
import alma.obops.tmcdbgui.utils.TmcdbConstants;
import alma.obops.tmcdbgui.utils.conversation.AssemblyTypeConversationUtils;
import alma.obops.tmcdbgui.utils.conversation.ComponentConversationUtils;
import alma.obops.tmcdbgui.utils.conversation.StartupScenarioConversationUtils;
import alma.obops.tmcdbgui.views.StartupScenariosView;
import alma.tmcdb.domain.Antenna;
import alma.tmcdb.domain.AssemblyRole;
import alma.tmcdb.domain.AssemblyStartup;
import alma.tmcdb.domain.AssemblyType;
import alma.tmcdb.domain.BaseElementStartup;
import alma.tmcdb.domain.BaseElementType;
import alma.tmcdb.domain.StartupScenario;

/**
 * A utility class to represent a child node of a BaseElementStartup parent
 * node. It links the parent node with an LRUType and one of its AssemblyRoles.
 * It may represent an actual AssemblyStartup instance or act as a placeholder.
 * A "placeholder" represents the possibility of an assembly startup (that the
 * user could - but has not yet - define to be started), whereas
 * an actual assemblystartup represents an assembly that should be started 
 * (i.e. the user did designate, at some point, that it should be started).
 * 
 * @author amchavan, Mar 6, 2009
 * 
 */



public class LRUTypeRole {
	
	private final static String MOUNT = "mount";
    
	private boolean simulated;
	private Component component;
	private AssemblyType assemblyTypeToUse;
	private BaseElementStartup rootOfTree;
    //private LruType LruType;
    private AssemblyRole assemblyRole;
    private BaseElementStartup baseElementStartup;
    private AssemblyStartup assemblyStartup;
    
    /**
     * @return <code>true</code> if this instance represents an actual
     *         AssemblyStartup; <code>false</code> if we are a placeholder.
     */
    public boolean isStarted() {
        return assemblyStartup != null;
    }

    public static String computePathForStartupHierarchy(BaseElementStartup bes) 
    {
    	String retVal = "";
    	
    	if(bes.getParent() != null) {
    		retVal +=  computePathForStartupHierarchy(bes.getParent()) + TmcdbConstants.SLASH;
    	}
    	
    	if(null == bes.getBaseElement()) {
    		retVal += bes.getType().name();
    	} 
    	else {
    		retVal += bes.getBaseElement().getName();
    	}
    	
    	return retVal;
	}

	public LRUTypeRole( BaseElementStartup baseElementStartup, 
                        AssemblyRole assemblyRole) 
    {
        this.baseElementStartup = baseElementStartup;
        this.assemblyRole = assemblyRole;
        
        // Find out whether we represent an AssemblyStartup or just the
        // possibility of one
        this.assemblyStartup = null;    // assume no AssemblyStartup
        for( AssemblyStartup as : baseElementStartup.getAssemblyStartups() ) {
            if( as.getAssemblyRole().equals( assemblyRole )) {
                this.assemblyStartup = as;
            }
        }
        
        rootOfTree = determineRootOfBaseElementTree();
        component = determineComponent();
        assemblyTypeToUse = determineAssemblyTypeToUse();
    }
    
    private AssemblyType determineAssemblyTypeToUse() 
    {
    	AssemblyType retVal = assemblyRole.getAssemblyType();
    	
    	// HACK: special case for mount - for which we need to differentiate the value used in the component.code field
    	// based on the type (VA, AEC, etc) of antenna
    	if(assemblyRole.getAssemblyType().getName().toLowerCase().equals((MOUNT)) && rootOfTree.getBaseElement().getType().equals(BaseElementType.Antenna)) 
    	{
    		Antenna ant = (Antenna) rootOfTree.getBaseElement(); 
    		try {
    			retVal = AssemblyTypeConversationUtils.getInstance().findAssemblyTypeForMountOfGivenAntennaType(ant.getAntennaType());
    		} catch(Exception e) {
    			retVal = assemblyRole.getAssemblyType();
    		}
    	}
    	
    	return retVal;
	}

	public AssemblyRole getAssemblyRole() {
        return assemblyRole;
    }
    
    public void setAssemblyRole( AssemblyRole assemblyRole ) {
        this.assemblyRole = assemblyRole;
    }

    /**
     * Determine whether this node represents an actual AssemblyStartup instance
     * or acts as a placeholder. Any changes to our BaseElementStartup will
     * be persisted (immediately).
     * 
     * @param start
     *            If <code>true</code>, it will make sure there is an
     *            AssemblyStartup associated with this node; if needed, a new
     *            instance will be created.
     * 
     *            If <code>false</code>, it will remove any existing ones,
     *            turning this node into a placeholder.
     *     
     * @param commitChangesImmediately boolean to indicate if changes should be written back to the persistent store or not.
     * 
     * @throws ServiceException 
     * @throws InvocationTargetException 
     * @throws IllegalAccessException 
     * @throws NoSuchMethodException 
     * @throws IllegalArgumentException 
     * @throws SecurityException 
     */
    public void setStarted( boolean start, boolean commitChangesImmediately ) throws Exception 
    {    
        // we're toggling the item to start (toggling on):
        if( start ) {
            // we need to make sure we have an AssemblyStartup
            if( isStarted() ) {         // do we have one?
                return;                 // YES, nothing to do
            }
            
            if(!isEnabled()) {
        		return;
        	}

            // we do not have an AssemblyStartup instance, let's create one
           assemblyStartup = new AssemblyStartup(baseElementStartup, assemblyRole);
           assemblyStartup.setSimulated(this.simulated);
           if(commitChangesImmediately) {
        	   StartupScenarioConversationUtils.getInstance().saveOrUpdateBaseElementStartup(baseElementStartup, ConversationToken.CONVERSATION_COMPLETED);
           }
        }
        
        // we're toggling the item to not start (toggling off)
        else {
            // we need to make sure we have no AssemblyStartups
            if( ! isStarted() ) {       // do we have one?
                return;                 // NO, nothing to do
            }
            
            // we do have an AssemblyStartup instance, let's get rid of it
            Set<AssemblyStartup> as = baseElementStartup.getAssemblyStartups();
            boolean removed = as.remove( assemblyStartup );
            assert(removed);
            assemblyStartup = null;
            if(commitChangesImmediately) {
            	StartupScenarioConversationUtils.getInstance().saveOrUpdateBaseElementStartup(baseElementStartup, ConversationToken.CONVERSATION_COMPLETED);
            }
        }
    }

	private BaseElementStartup determineRootOfBaseElementTree() 
	{
		// "generic" baseelementstartup objects have no associated startup reference;
		// instead, they have a parent, which eventually resolves to a startup reference. 
		// let's loop to find the startup, using the parent(s) hierarchy:
		rootOfTree = baseElementStartup;
		while(null == rootOfTree.getStartup()) {
			rootOfTree = rootOfTree.getParent();
		}

		// if the top of the hierarchy still didn't have a startup reference, then
		// there is a problem with the DB as this is not a valid state!
		if(null == rootOfTree.getStartup()) {
			throw new IllegalStateException("BaseElementStartup hierarchy does not belong to a startup; DB is corrupt.");
		}
		return rootOfTree;
	}
    
	public boolean isSimulated() 
	{
		boolean retVal = (this.assemblyStartup != null && this.assemblyStartup.getSimulated() != null) ? this.assemblyStartup.getSimulated() : this.simulated;
		return retVal;
	}

	/**
	 * Sets the item as simulated or not.
	 * @param issimulated the boolean indicating if the item should be simulated (true) or not (false).
	 * @param displayMessages boolean indicating if messages should be displayed.
	 * @return errMsg indicating if the code field of the associated component has been overridden, in which case
	 * toggling the simulated/not-simulated status of the item will have no effect when ACS starts the system.
	 * @throws Exception
	 */
    public String setSimulated(boolean issimulated, boolean displayMessages) throws Exception
    {
    	String retVal = null;
    	
    	if(!isEnabled() || null == this.assemblyStartup) {
    		this.simulated = issimulated;
    		return retVal;
    	}
    	this.assemblyStartup.setSimulated(issimulated);
    	
    	// check to see if the component's code field has been overridden (i.e. is not equal to *either*
    	// the production code or the simulation code); if so, then simulating/not simulating this assembly startup
    	// really will have no effect as ACS will simply adhere to the overridden value. In such cases, we wish to 
    	// notify the user with a popup.
    	if(component != null) 
    	{
    		if(!component.getCode().equals(assemblyTypeToUse.getProductionCode()) &&
    		   !component.getCode().equals(assemblyTypeToUse.getSimulatedCode())) 
    		{
    			String fullPath = component.getPath() + "/" + component.getComponentName();
    			retVal = fullPath;
    			if( displayMessages ) {
    				MessageDialog.openWarning(null, "Warning", "The 'code' field for component '" + fullPath + "'" 
    						+ " has been overridden (from the defaults defined in the Assembly Type library mappings). Therefore, toggling its simulated/not-simulated attribute will have no effect.");
    			}
    		}
    	}
    	
    	StartupScenarioConversationUtils.getInstance().saveOrUpdateAssemblyStartup(assemblyStartup);
    	StartupScenariosView view = (StartupScenariosView) RcpUtils.findView(StartupScenariosView.ID);
    	view.refresh();
    	
    	return retVal;
    }

	private Component determineComponent() 
	{
		String path = TmcdbConstants.CONTROL_PREFIX + TmcdbConstants.SLASH + computePathForStartupHierarchy(baseElementStartup);

		List<Component> comps = null;
		try 
		{
			comps = ComponentConversationUtils.getInstance().findComponentByPathAndNameWithinConfiguration(path, assemblyRole.getName(), rootOfTree.getBaseElement().getConfiguration().getSwConfiguration());
			if(null == comps || comps.size() == 0 || comps.size() > 1) 
			{
				comps = null;
			}
		} catch(Exception e) {
			comps = null;
		}
		
		Component retVal = null;
		if(comps != null) {
			retVal = comps.get(0);
		}
		
		return retVal;
	}
    
    public StartupScenario getStartup() 
    {
    	StartupScenario retVal = null;
    	retVal = baseElementStartup.getStartup();
    	
    	// hack / workaround for the fact that generic baseelementstartup objects
    	// don't have the startup field set; this should be corrected.
    	if(null == retVal) {
    		if( null != this.baseElementStartup.getParent())
    			retVal = this.baseElementStartup.getParent().getStartup();
    	}

    	return retVal;
    }

	public boolean isEnabled() 
	{
		boolean retVal = false;
		
		if(component != null) {
			retVal = true;
		}
		
		return retVal;
	}

	/**
	 * This method returns true if the code used for the component 
	 * is identical in both simulation and production mode. The value
	 * of said information is from the assemblytype.
	 * @return boolean indicating if the values are the same (true) or not the same (false).
	 */
	public boolean simAndProductionCodeAreSame() 
	{
		boolean retVal = false;
		
		if(null != assemblyTypeToUse)
		{
			if(null == assemblyTypeToUse.getProductionCode() && null != assemblyTypeToUse.getSimulatedCode())
			{
				retVal = false;
			}
			else if(null != assemblyTypeToUse.getProductionCode() && null == assemblyTypeToUse.getSimulatedCode())
			{
				retVal = false;
			}
			else if(null == assemblyTypeToUse.getProductionCode() && null == assemblyTypeToUse.getSimulatedCode())
			{
				retVal = true;
			}
			else if(assemblyTypeToUse.getProductionCode().equals(assemblyTypeToUse.getSimulatedCode()))
			{
				retVal = true;
			}
		}
		
		return retVal;
	}
}
