/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdbgui.views.providers.helpers.startup;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.Font;
import org.eclipse.swt.graphics.Image;

import alma.obops.tmcdb.alarms.ui.tree.helpers.ThreeColumnDomainObjectHelper;
import alma.obops.tmcdb.alarms.ui.utils.RcpUtils;
import alma.obops.tmcdbgui.utils.conversation.StartupScenarioConversationUtils;
import alma.obops.tmcdbgui.views.providers.typedlists.AntennaStartupList;
import alma.tmcdb.domain.BaseElementStartup;
import alma.tmcdb.domain.BaseElementType;
import alma.tmcdb.domain.StartupScenario;

/**
 * Helper class for StartupScenario domain object.
 * @author sharrington
 */
public class StartupScenarioHelper implements ThreeColumnDomainObjectHelper 
{
	private StartupScenario startup;
	private boolean hydrated;
	private static Map<StartupScenario, StartupScenarioHelper> instanceMap = new HashMap<StartupScenario, StartupScenarioHelper>(); 

	/**
	 * Constructor.
	 * @param startup the StartupScenario domain object for which
	 * this helper class provides info (e.g. images, text, 
	 * children, etc.)
	 */
	private StartupScenarioHelper(StartupScenario startup)
	{
		this.startup = startup;
	}
	
	public synchronized static StartupScenarioHelper getInstance(StartupScenario startup)
	{
		StartupScenarioHelper retVal = instanceMap.get(startup);
		
		if(null == retVal) {
			retVal = new StartupScenarioHelper(startup);
			instanceMap.put(startup, retVal);
		}
		
		return retVal;
	}
	
	public synchronized static void clearCache() 
	{
		instanceMap.clear();
	}
	
	@Override
	public Object[] getChildren() 
	{
		Object[] retVal = new Object[0];
		
		List<Object> children = new ArrayList<Object>();

		// Antennas
		AntennaStartupList antennas = new AntennaStartupList(startup);
		try {
			if(!hydrated)
			{
				StartupScenarioConversationUtils.getInstance().hydrateBaseElementStartups(startup);
				hydrated = true;
			}
		} catch (Exception ex) {
			throw new RuntimeException("Could not hydrate base element startups.", ex);
		}

		for( BaseElementStartup be : startup.getBaseElementStartups() ) 
		{
			if( null != be.getBaseElement() && be.getBaseElement().getType().equals( BaseElementType.Antenna ) ) 
			{
				antennas.add( be );
			}
		}
		children.add( antennas );

		// CentralRacks 
		CentralRackStartupList centralracks = new CentralRackStartupList(startup);
		for( BaseElementStartup be : startup.getBaseElementStartups() ) 
		{
			if( null != be.getBaseElement() && be.getBaseElement().getType().equals( BaseElementType.CentralLO ) ) 
			{
				centralracks.add( be );
			}
		}
		if(centralracks.size() > 0) {
			children.add( centralracks );
		}

		// MasterClocks
		MasterClockStartupList masterclocks = new MasterClockStartupList(startup);
		for( BaseElementStartup be : startup.getBaseElementStartups() ) 
		{
			if( null != be.getBaseElement() && be.getBaseElement().getType().equals( BaseElementType.AOSTiming ) ) 
			{
				masterclocks.add( be );
			}
		}
		if(masterclocks.size() > 0) {
			children.add( masterclocks );
		}
		
		// WeatherStations
		WeatherStationStartupList weatherStations = new WeatherStationStartupList(startup);
		for( BaseElementStartup be : startup.getBaseElementStartups() ) 
		{
			if( null != be.getBaseElement() && be.getBaseElement().getType().equals( BaseElementType.WeatherStationController ) ) 
			{
				weatherStations.add( be );
			}
		}
		if(weatherStations.size() > 0) {
			children.add( weatherStations );
		}
		
		retVal = children.toArray( retVal );
		
		return retVal;
	}

	@Override
	public Image getFirstColumnImage() {
		return RcpUtils.getImage( "icons/startup.png" );
	}

	@Override
	public String getFirstColumnText() {
		return startup.getName();
	}

	@Override
	public Image getSecondColumnImage() {
		return null;
	}

	@Override
	public String getSecondColumnText() {
		return null;
	}

	@Override
	public Image getThirdColumnImage() {
		return null;
	}

	@Override
	public String getThirdColumnText() {
		return null;
	}

	@Override
	public boolean hasChildren() {
		return true;
	}

	@Override
	public Font getFont() {
		return null;
	}

	@Override
	public Color getForeground() {
		return null;
	}

}
