/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdbgui.views.providers.helpers.startup;

import java.util.List;

import alma.obops.tmcdb.alarms.ui.tree.helpers.ListHelper;
import alma.obops.tmcdb.alarms.ui.tree.helpers.ThreeColumnDomainObjectHelper;
import alma.obops.tmcdb.alarms.ui.tree.helpers.factory.ThreeColumnDomainObjectHelperFactory;
import alma.obops.tmcdbgui.views.providers.helpers.config.AntennaListHelper;
import alma.obops.tmcdbgui.views.providers.helpers.config.FrontEndListHelper;
import alma.obops.tmcdbgui.views.providers.typedlists.AntennaStartupList;
import alma.obops.tmcdbgui.views.providers.typedlists.FrontEndStartupList;
import alma.obops.tmcdbgui.views.providers.typedlists.LRUTypeRole;
import alma.tmcdb.domain.Antenna;
import alma.tmcdb.domain.BaseElementStartup;
import alma.tmcdb.domain.BaseElementStartupType;
import alma.tmcdb.domain.FrontEnd;
import alma.tmcdb.domain.HwConfiguration;
import alma.tmcdb.domain.StartupScenario;

/**
 * Factory class for creating helper objects within the startup tree/view.
 * @author sharrington
 */
public class StartupHelperFactory implements ThreeColumnDomainObjectHelperFactory
{
	private static StartupHelperFactory singletonInstance = null;
	private BaseElementStartupHelper baseElementStartupHelper;
	private HwConfiguration owningConfiguration;
	
	/**
	 * Getter for the singleton instance.
	 * @return the singleton instance, after creating it if needed.
	 */
	public synchronized static StartupHelperFactory getInstance()
	{
		if(null == singletonInstance)
		{
			singletonInstance = new StartupHelperFactory();
		}
		
		return singletonInstance;
	}
	
	/**
	 * Clears all the helper caches (which are used for performance optimization); 
	 * this method should usually be called after a reload of configurations, 
	 * so that we don't get hibernate duplicate object id exceptions, and so forth.
	 */
	public synchronized void clearCaches()
	{
		ConfigurationStartupHelper.clearCache();
		if(null != baseElementStartupHelper)
		{
			baseElementStartupHelper.clearBaseElementStartupHelperCaches();
		}
		StartupScenarioHelper.clearCache();
		AntennaStartupHelper.clearCache();
		FrontEndStartupHelper.clearCache();
		MasterClockStartupHelper.clearCache();
		CentralRackStartupHelper.clearCache();
		PhotonicReferenceStartupHelper.clearCache();
		WeatherStationStartupHelper.clearCache();
	}
	
	@Override
	public ThreeColumnDomainObjectHelper getHelper(Object object)
	{
		ThreeColumnDomainObjectHelper retVal = null;
		
		if(object instanceof HwConfiguration)
		{
			retVal = getHelper((HwConfiguration)object);
		}
		
		else if(object instanceof StartupScenario)
		{
			retVal = getHelper((StartupScenario)object);
		}
		
		else if(object instanceof BaseElementStartup)
		{
			retVal = getHelper((BaseElementStartup)object);
		}

		else if(object instanceof AntennaStartupList)
		{
			retVal = getHelper((AntennaStartupList)object);
		}
		
		else if(object instanceof FrontEndStartupList)
		{
			retVal = getHelper((FrontEndStartupList)object);
		}
		
		else if(object instanceof MasterClockStartupList)
		{
			retVal = getHelper((MasterClockStartupList)object);
		}
		
		else if(object instanceof CentralRackStartupList)
		{
			retVal = getHelper((CentralRackStartupList)object);
		}
		
		else if(object instanceof WeatherStationStartupList)
		{
			retVal = getHelper((WeatherStationStartupList)object);
		}
		
		else if(object instanceof LRUTypeRole)
		{
			retVal = getHelper((LRUTypeRole)object);
		}
		
		else if(object instanceof List<?>)
		{
			retVal = getHelper((List<?>)object);
		}
		
		else 
		{
			 failUnsupported(object);
		}
		
		return retVal;
	}
	
	/**
	 * Constructor, private to enforce singleton pattern.
	 */
	private StartupHelperFactory()
	{
	}
	
	private ThreeColumnDomainObjectHelper getHelper(StartupScenario startup)
	{
		ThreeColumnDomainObjectHelper retVal = StartupScenarioHelper.getInstance(startup);
		return retVal;
	}
	
	private ThreeColumnDomainObjectHelper getHelper(HwConfiguration config)
	{
		ThreeColumnDomainObjectHelper retVal = ConfigurationStartupHelper.getInstance(config);
		return retVal;
	}
	
	private ThreeColumnDomainObjectHelper getHelper(BaseElementStartup baseElementStartup)
	{
		ThreeColumnDomainObjectHelper retVal = null;
		if(baseElementStartup.getBaseElement() instanceof Antenna) 
		{
			retVal = AntennaStartupHelper.getInstance( baseElementStartup, owningConfiguration );
		} 
		else if(baseElementStartup.getBaseElement() == null 
				&& baseElementStartup.getType().equals(BaseElementStartupType.FrontEnd) 
				|| baseElementStartup.getBaseElement() instanceof FrontEnd) 
		{
			retVal = FrontEndStartupHelper.getInstance( baseElementStartup, owningConfiguration );
		}
		else if(baseElementStartup.getType().equals(BaseElementStartupType.AOSTiming)) 
		{
			retVal = MasterClockStartupHelper.getInstance( baseElementStartup, owningConfiguration );
		}
		else if(baseElementStartup.getType().equals(BaseElementStartupType.CentralLO)) 
		{
			retVal = CentralRackStartupHelper.getInstance( baseElementStartup, owningConfiguration );
		}
		else if(baseElementStartup.getType().equals(BaseElementStartupType.PhotonicReference1) || 
				baseElementStartup.getType().equals(BaseElementStartupType.PhotonicReference2) ||
				baseElementStartup.getType().equals(BaseElementStartupType.PhotonicReference3) ||
				baseElementStartup.getType().equals(BaseElementStartupType.PhotonicReference4) ||
				baseElementStartup.getType().equals(BaseElementStartupType.PhotonicReference5) ||
				baseElementStartup.getType().equals(BaseElementStartupType.PhotonicReference6))
		{
			retVal = PhotonicReferenceStartupHelper.getInstance( baseElementStartup, baseElementStartup.getType(), owningConfiguration );
		}
		else if(baseElementStartup.getType().equals(BaseElementStartupType.WeatherStationController)) 
		{
			retVal = WeatherStationStartupHelper.getInstance( baseElementStartup, owningConfiguration );
		}
		else {
			synchronized(this) {
				if(null == baseElementStartupHelper) {
					baseElementStartupHelper = new BaseElementStartupHelper(baseElementStartup, owningConfiguration);
				}
				retVal = baseElementStartupHelper;
			}
		}
		return retVal;
	}
	
	private ThreeColumnDomainObjectHelper getHelper(List<?> list)
	{
		ThreeColumnDomainObjectHelper retVal = new ListHelper(list);
		return retVal;
	}
	
	private ThreeColumnDomainObjectHelper getHelper(AntennaStartupList list)
	{
		ThreeColumnDomainObjectHelper retVal = new AntennaListHelper(list);
		return retVal;
	}
	
	private ThreeColumnDomainObjectHelper getHelper(FrontEndStartupList list)
	{
		ThreeColumnDomainObjectHelper retVal = new FrontEndListHelper(list);
		return retVal;
	}
	
	private ThreeColumnDomainObjectHelper getHelper(MasterClockStartupList list)
	{
		ThreeColumnDomainObjectHelper retVal = new MasterClockListHelper(list);
		return retVal;
	}

	private ThreeColumnDomainObjectHelper getHelper(CentralRackStartupList list)
	{
		ThreeColumnDomainObjectHelper retVal = new CentralRackListHelper(list);
		return retVal;
	}
	
	private ThreeColumnDomainObjectHelper getHelper(WeatherStationStartupList list)
	{
		ThreeColumnDomainObjectHelper retVal = new WeatherStationStartupListHelper(list);
		return retVal;
	}
	
	private ThreeColumnDomainObjectHelper getHelper(LRUTypeRole role)
	{
		ThreeColumnDomainObjectHelper retVal = new LRUTypeRoleHelper(role);
		return retVal;
	}

	private void failUnsupported(Object element) {
		// Should never happen
		String msg = "Unsupported class: " + element.getClass().getName();
		IllegalArgumentException e = new IllegalArgumentException( msg );
		e.printStackTrace();
		throw e;
	}

	public void setConfiguration(HwConfiguration conf) {
		owningConfiguration = conf;
	}
}

