/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdbgui.views.providers.helpers.startup;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.Font;
import org.eclipse.swt.graphics.Image;

import alma.acs.tmcdb.Component;
import alma.obops.dam.utils.ConversationTokenProvider.ConversationToken;
import alma.obops.tmcdb.alarms.ui.tree.helpers.ThreeColumnDomainObjectHelper;
import alma.obops.tmcdb.alarms.ui.utils.RcpUtils;
import alma.obops.tmcdbgui.utils.DomainObjectUtils;
import alma.obops.tmcdbgui.utils.TmcdbConstants;
import alma.obops.tmcdbgui.utils.conversation.AssemblyConversationUtils;
import alma.obops.tmcdbgui.utils.conversation.ComponentConversationUtils;
import alma.obops.tmcdbgui.utils.conversation.StartupScenarioConversationUtils;
import alma.obops.tmcdbgui.views.providers.helpers.config.ConfigHelperFactory;
import alma.obops.tmcdbgui.views.providers.typedlists.LRUTypeRole;
import alma.tmcdb.domain.AssemblyRole;
import alma.tmcdb.domain.AssemblyType;
import alma.tmcdb.domain.BaseElementStartup;
import alma.tmcdb.domain.BaseElementStartupType;
import alma.tmcdb.domain.HwConfiguration;

/**
 * Helper class for generic BaseElementStartup domain objects 
 * with baseElementType other than Antenna or FrontEnd.
 * 
 * @author sharrington
 */
public class BaseElementStartupHelper implements ThreeColumnDomainObjectHelper 
{
	protected HwConfiguration owningConfiguration;
	private BaseElementStartup baseElementStartup;
	private BaseElementStartup rootOfTree;
	private boolean hydrated = false;
	private Component component;
	private Map<BaseElementStartupType, AssemblyType[]> assemblyTypesMap = new HashMap<BaseElementStartupType, AssemblyType[]>();
	private Map<BaseElementStartupType, Object[]> childrenMap = new HashMap<BaseElementStartupType, Object[]>();
	
	/**
	 * Constructor.
	 * @param baseElementStartup the BaseElementStartup domain object
	 * for which this helper class provides info (e.g. images, text,
	 * children, etc.)
	 */
	public BaseElementStartupHelper(BaseElementStartup baseElementStartup, HwConfiguration owningConfiguration)
	{
		this.baseElementStartup = baseElementStartup;
		this.owningConfiguration = owningConfiguration;
		rootOfTree = DomainObjectUtils.determineRootOfBaseElementTree(baseElementStartup);
		component = determineComponent();
	}
	
	private Component determineComponent()
	{
		if(rootOfTree == null) {
			rootOfTree = DomainObjectUtils.determineRootOfBaseElementTree(baseElementStartup);
		}
		String pathPlusName = TmcdbConstants.CONTROL_PREFIX + TmcdbConstants.SLASH + LRUTypeRole.computePathForStartupHierarchy(baseElementStartup);
		String[] pathParts = pathPlusName.split(TmcdbConstants.SLASH);
		String name = pathParts[pathParts.length - 1];
		int indexOfName = pathPlusName.indexOf(name);
		String path = pathPlusName.substring(0, indexOfName - 1);

		List<Component> comps = null;
		try
		{
			comps = ComponentConversationUtils.getInstance().findComponentByPathAndNameWithinConfiguration(path, name, rootOfTree.getStartup().getConfiguration().getSwConfiguration());
			if(null == comps || comps.size() == 0 || comps.size() > 1) 
			{
				comps = null;
			}
		} catch(Exception e) {
			comps = null;
			throw new RuntimeException("Could not get component for base element startup", e);
		}
		
		Component retVal = null;
		if(comps != null) {
			retVal = comps.get(0);
		}
		
		return retVal;
	}
	
	@Override
	public Object[] getChildren() 
	{
		Object[] retVal = new Object[0];
	
		if(!hydrated)
		{
			// hydrate baseelementstartup down to its children, to avoid hibernate lazy initialization exceptions
			try {
				StartupScenarioConversationUtils.getInstance().hydrateBaseElementStartupToChildren(baseElementStartup);
			} catch (Exception e) {
				e.printStackTrace();
				throw new RuntimeException("Unable to hydrate baseelementstartup", e);
			}
			hydrated = true;
		}
		
		BaseElementStartupType type = baseElementStartup.getType();
		retVal = getChildrenByType(type);
		
		if(baseElementStartup.getChildren() != null && baseElementStartup.getChildren().size() > 0) 
		{
			List<Object> children = new ArrayList<Object>();
			for(int i = 0; i < retVal.length; i++) {
				children.add(retVal[i]);
			}
			children.addAll(baseElementStartup.getChildren());
			retVal = new Object[retVal.length + baseElementStartup.getChildren().size()];
			retVal = children.toArray(retVal);
		}
        
        return retVal;
	}

	@Override
	public Image getFirstColumnImage() {
		return RcpUtils.getImage("icons/unknown.png");
	}

	@Override
	public String getFirstColumnText() {
		String retVal = null;
		if(null != baseElementStartup.getBaseElement()) {
		   retVal = baseElementStartup.getBaseElement().getName();
		} else {
			retVal = baseElementStartup.getType().name();
		}
		String configNameOfBaseElement = 
			(null != baseElementStartup.getBaseElement()) ? 
					baseElementStartup.getBaseElement().getConfiguration().getName() : 
					baseElementStartup.getParent().getBaseElement().getConfiguration().getName();
					
		if(!configNameOfBaseElement.equals(owningConfiguration.getName())) {
			retVal += ":" + configNameOfBaseElement;
		}
		return retVal;
	}

	@Override
	public Image getSecondColumnImage() {
		Image retVal = null;
		return retVal;
	}
	
	@Override
	public Image getThirdColumnImage() {
		Image retVal = null;
		if(!isEnabled()) {
			retVal = RcpUtils.getImage("icons/checkbox-disabled.png");
		}
		else if(getSimulationCodeForBaseElement().equals(getProductionCodeForBaseElement())) 
		{
			retVal = RcpUtils.getImage("icons/checkbox-equals.png");
		}
		else if( isSimulated()) {
			retVal = RcpUtils.getImage("icons/checkbox-set.png");
		}
		else {
			retVal = RcpUtils.getImage("icons/checkbox-unset.png");
		}
		return retVal;
	}
	
	@Override
	public String getSecondColumnText() {
		return null;
	}

	@Override
	public String getThirdColumnText() {
		return null;
	}

	@Override
	public Font getFont() {
		return null;
	}

	@Override
	public Color getForeground() {
		return null;
	}

	@Override
	public boolean hasChildren() {
		return getChildren().length > 0;
	}
	
	/**
     * @param The list of AssemblyRoles for a given LRUType
     */
    protected AssemblyRole[] computeRoleList( AssemblyType assType ) 
    {
    	try {
			AssemblyConversationUtils.getInstance().hydrateAssemblyType(assType);
		} catch (Exception e) {
			throw new RuntimeException("Could not hydrate assembly type", e);
		}
    	assType = ((ConfigHelperFactory)ConfigHelperFactory.getInstance()).getAssemblyType(assType);
        Set<AssemblyRole> roles = assType.getRoles();
        return roles.toArray( new AssemblyRole[0] );
    }
    
    private boolean isEnabled()
    {
    	return (component != null);
    }
    
	private Object[] getChildrenByType(BaseElementStartupType type) 
	{
		AssemblyType[] assemblyTypes = assemblyTypesMap.get(type);
		if(null == assemblyTypes)
		{
			try {
				assemblyTypes = ((ConfigHelperFactory)ConfigHelperFactory.getInstance()).findAssemblyTypesByBaseElementStartupType(type);
				assemblyTypesMap.put(type, assemblyTypes);
			} catch (Exception ex) {
				ex.printStackTrace();
				throw new RuntimeException("Could not find lrutypes for baseelement type", ex);
			}
		}
		
		Object[] retVal = childrenMap.get(type);
		if(retVal == null) 
		{
			List<Object> children = new ArrayList<Object>();

			// For each LRUType we find its possible roles.
			for( int i = 0; i < assemblyTypes.length; i++ ) {
				AssemblyType assType = assemblyTypes[i];
				AssemblyRole[] roles = computeRoleList( assType );
				for( int j = 0; j < roles.length; j++ ) {
					// For each LRUType/AssemblyRole coupling we create a new
					// LRUTypeRole instance, that is, a child of the input
					// BaseElementStartup
					AssemblyRole assemblyRole = roles[j];
					
					children.add( new LRUTypeRole( baseElementStartup, assemblyRole));
				}
			}

			retVal = children.toArray( new Object[0] );
			childrenMap.put(type, retVal);
		}
        
		return retVal;
	}
	
	public synchronized boolean isStarted() 
	{
		boolean retVal = isEnabled();
		return retVal;
	}
	
	private String getSimulationCodeForBaseElement()
	{
		String retVal = null;
		switch(baseElementStartup.getType())
		{
		case Antenna:
			retVal = "antennaSim";
			break;
		case Array:
			retVal = "arraySim";
			break;
		case AOSTiming:
			retVal = "AOSTimingSim";
			break;
		case CentralLO:
			retVal = "CentralLOSim";
			break;
		case FrontEnd:
			retVal = "FrontEndImpl";
			break;
		case Pad:
			retVal = "Pad";
			break;
		case HolographyTower:
			retVal = "Holography Tower";
			break;
		case PhotonicReference1:
		case PhotonicReference2:
		case PhotonicReference3:
		case PhotonicReference4:
		case PhotonicReference5:
		case PhotonicReference6:
			retVal = "PhotonicReference";
			break;
		case WeatherStationController:
			retVal = "WeatherStationController";
			break;
		}
		return retVal;
	}
	
	private String getProductionCodeForBaseElement() 
	{
		String retVal = null;
		switch(baseElementStartup.getType())
		{
		case Antenna:
			retVal = "antenna";
			break;
		case Array:
			retVal = "Array";
			break;
		case AOSTiming:
			retVal = "AOSTiming";
			break;
		case CentralLO:
			retVal = "CentralLO";
			break;
		case Pad:
			retVal = "Pad";
			break;
		case FrontEnd:
			retVal = "FrontEndImpl";
			break;
		case HolographyTower:
			retVal = "HolographyTower";
			break;
		case PhotonicReference1:
		case PhotonicReference2:
		case PhotonicReference3:
		case PhotonicReference4:
		case PhotonicReference5:
		case PhotonicReference6:
			retVal = "PhotonicReference";
			break;
		case WeatherStationController:
			retVal = "WeatherStationController";
			break;
		}
		return retVal;
	}
	
	public synchronized boolean isSimulated() 
	{
		boolean retVal = (this.baseElementStartup != null && this.baseElementStartup.getSimulated() != null) ? this.baseElementStartup.getSimulated() : false;
		return retVal;
	}

	/**
	 * Sets a baseelement startup to simulated (or non-simulated) per the value of the boolean 'sim'
	 * @param sim boolean indicating if the baseelementstartup (and all its children) should be set to simulated (true) or not (false)
	 * @throws Exception
	 */
	public synchronized void setSimulated(boolean sim) throws Exception
	{
		if(!isEnabled()) {
			return;
		}
		
		List<String> pathsOfErroneousComponents = new ArrayList<String>();
		
		Object[] children = getChildren();
		for(Object obj: children) 
		{
			if(obj instanceof LRUTypeRole) {
				LRUTypeRole lruTypeRole = (LRUTypeRole) obj;
				String errMsg = lruTypeRole.setSimulated(sim, false);
				if(errMsg != null) {
					pathsOfErroneousComponents.add(errMsg);
				}
			} else if(obj instanceof BaseElementStartup) {
				BaseElementStartup beStartup = (BaseElementStartup) obj;
				BaseElementStartupHelper helper = (BaseElementStartupHelper) StartupHelperFactory.getInstance().getHelper(beStartup);
				helper.setSimulated(sim);
			}
		}
		this.baseElementStartup.setSimulated(sim);
		
		if(pathsOfErroneousComponents.size() > 0) 
		{
			StringBuffer pathsBuffer = new StringBuffer();
			int count = 0;
			for(String str : pathsOfErroneousComponents ) {
				pathsBuffer.append(str);
				if(++count < pathsOfErroneousComponents.size()) {
					pathsBuffer.append("; ");
				} 
			}
			String errMsg = "These components have their 'code' field overridden (from the defaults defined in the Assembly Type libraries mappings): \n\n" 
				+ pathsBuffer.toString() + "\n\n Toggling simulated/not-simulated for these items will have no effect.";
			MessageDialog.openWarning(null, "Warning", errMsg);
		}
		StartupScenarioConversationUtils.getInstance().saveOrUpdateBaseElementStartup(this.baseElementStartup, ConversationToken.CONVERSATION_COMPLETED);
	}
	
	public synchronized void setStarted(boolean startedValue, boolean nested) throws Exception 
	{
		if(!isEnabled()) {
			return;
		}
		
		Object[] children = getChildren();
		int count = 0;
		int numLruChildren = countLruTypesInArray(children);
		for(Object obj: children) 
		{
			count++;
			if(obj instanceof LRUTypeRole) {
				LRUTypeRole lruTypeRole = (LRUTypeRole) obj;
				boolean commit = ((count == numLruChildren) && !nested) ? true : false;
				lruTypeRole.setStarted(startedValue, commit);
			}
			else if(obj instanceof BaseElementStartup) {
				BaseElementStartup beStartup = (BaseElementStartup) obj;
				BaseElementStartupHelper helper = (BaseElementStartupHelper) StartupHelperFactory.getInstance().getHelper(beStartup);
				helper.setStarted(startedValue, true);
				if(count >= children.length) {
					StartupScenarioConversationUtils.getInstance().saveOrUpdateBaseElementStartup(beStartup, ConversationToken.CONVERSATION_COMPLETED);
				}
			}
		}
	}


	private int countLruTypesInArray(Object[] children) {
		int retVal = 0;
		
		for(Object obj : children)
		{
			if(obj instanceof LRUTypeRole)
			{
				retVal++;
			}
		}
		
		return retVal;
	}

	public void clearBaseElementStartupHelperCaches() 
	{
		this.childrenMap.clear();
		this.assemblyTypesMap.clear();
		this.hydrated = false;
	}
}
