/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdbgui.views.providers.helpers.config;

import org.eclipse.swt.graphics.Image;

import alma.obops.tmcdb.alarms.ui.utils.RcpUtils;
import alma.obops.tmcdbgui.utils.conversation.BaseElementConversationUtils;
import alma.tmcdb.domain.AntennaToPad;
import alma.tmcdb.domain.HolographyTowerToPad;
import alma.tmcdb.domain.Pad;

/**
 * Helper class for the Pad domain object.
 * @author sharrington
 */
public class PadHelper extends BaseElementHelper 
{
	/**
	 * Constructor.
	 * @param pad the Pad domain object for which this 
	 * helper class provides info (e.g. images, text, children, etc.)
	 */
	public PadHelper(Pad pad) {
		super(pad);
	}

	@Override
	public Image getFirstColumnImage() {
		return RcpUtils.getImage("icons/pad.png");
	}
	
	@Override
	public String getFirstColumnText() {
		String retVal = super.getFirstColumnText();
		
		Pad pad = (Pad) baseElement;
		try {
			AntennaToPad[] a2ps = BaseElementConversationUtils.getInstance().findOpenAntennaToPadAssignmentsForPad(pad, pad.getConfiguration());
			if(null != a2ps && a2ps.length == 1)
			{
				if(!a2ps[0].getPad().getConfiguration().getId().equals(a2ps[0].getAntenna().getConfiguration().getId()))
				{
					return retVal;
				}
				retVal += " with ";
				retVal += a2ps[0].getAntenna().getName();
			}
		} catch (Exception e) {
			e.printStackTrace();
			throw new RuntimeException("Could not locate antenna to pad assignments for pad", e);
		}
		return retVal;
	}
	
	@Override
	public boolean hasChildren() 
	{
		boolean retVal = false;
		
		Pad pad = (Pad) this.baseElement;
		try {
			BaseElementConversationUtils.getInstance().hydratePad(pad);
		} catch (Exception e) {
			e.printStackTrace();
			throw new RuntimeException("Could not hydrate pad", e);
		}
		
		HolographyTowerToPad h2p = findCurrentHolographyTowerToPadForPad(pad);
		if(null != h2p) {
			retVal = true;
		} else {
			retVal = false;
		}
		
		return retVal;
	}
	
	@Override
	public Object[] getChildren() 
	{
		Object[] retVal = null;
		Pad pad = (Pad) this.baseElement;
		
		HolographyTowerToPad h2p = findCurrentHolographyTowerToPadForPad(pad);
		
		if(h2p != null) {
			retVal = new HolographyTowerToPad[1];
			retVal[0] = h2p;
		}
		
		return retVal;
	}
	
	public static HolographyTowerToPad findCurrentHolographyTowerToPadForPad(Pad pad) 
	{
		HolographyTowerToPad retVal = null;
		
		try {
			HolographyTowerToPad[] h2ps = BaseElementConversationUtils.getInstance().findHolographyTowerToPadAssignmentsForPad(pad);
			for(HolographyTowerToPad h2p : h2ps) {
				retVal = h2p;
				BaseElementConversationUtils.getInstance().hydrateHolographyTower(retVal.getHolographyTower());
				BaseElementConversationUtils.getInstance().hydratePad(retVal.getPad());
				break;
			}
		} catch (Exception e) {
			e.printStackTrace();
			throw new RuntimeException("Could not find holographytowertopad assignments for pad");
		}
		
		return retVal;
	}
}
