/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdbgui.views.providers.helpers.config;

import java.util.ArrayList;
import java.util.List;
import java.util.Set;

import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.Font;
import org.eclipse.swt.graphics.Image;

import alma.obops.tmcdb.alarms.ui.tree.helpers.ThreeColumnDomainObjectHelper;
import alma.obops.tmcdb.alarms.ui.utils.RcpUtils;
import alma.obops.tmcdbgui.utils.conversation.AssemblyConversationUtils;
import alma.obops.tmcdbgui.views.providers.typedlists.AntennaList;
import alma.obops.tmcdbgui.views.providers.typedlists.AssemblyList;
import alma.obops.tmcdbgui.views.providers.typedlists.BaseElementList;
import alma.obops.tmcdbgui.views.providers.typedlists.FrontEndList;
import alma.obops.tmcdbgui.views.providers.typedlists.PadList;
import alma.obops.tmcdbgui.views.providers.typedlists.TypeList;
import alma.tmcdb.domain.Antenna;
import alma.tmcdb.domain.Assembly;
import alma.tmcdb.domain.AssemblyType;
import alma.tmcdb.domain.BaseElement;
import alma.tmcdb.domain.BaseElementType;
import alma.tmcdb.domain.FrontEnd;
import alma.tmcdb.domain.HwConfiguration;
import alma.tmcdb.domain.Pad;

/**
 * Helper class for Configuration domain object.
 * @author sharrington
 *
 */
public class ConfigurationHelper implements ThreeColumnDomainObjectHelper
{
	protected HwConfiguration configuration;
	protected boolean hydrated;

	/**
	 * Constructor.
	 * @param configuration the configuration domain object for which this
	 * helper class provides info (e.g. images, text, children, etc.)
	 */
	public ConfigurationHelper(HwConfiguration configuration)
	{
		this.configuration = configuration;	
	}
	
	@Override
	public Object[] getChildren() 
	{
		Object[] retVal = new Object[0];
	
        List<Object> children = new ArrayList<Object>();

        AntennaList antennas = new AntennaList(configuration);
        PadList pads = new PadList(configuration);
        FrontEndList frontEnds = new FrontEndList(configuration);
        CentralRackList centralRacks = new CentralRackList(configuration);
        MasterClockList masterClocks = new MasterClockList(configuration);
        PhotonicReferenceList photonicRefs = new PhotonicReferenceList(configuration);
        HolographyTowerList holographyTowers = new HolographyTowerList(configuration);
        WeatherStationList weatherStations = new WeatherStationList(configuration);
        
        BaseElementList unknowns = new BaseElementList(configuration, null);
        
        for( BaseElement be : configuration.getBaseElements() ) {
            if( be instanceof Antenna )  {
                antennas.add( be );
            }
            else if( be instanceof Pad ) {
                pads.add( be );
            }
            else if( be instanceof FrontEnd ) {
                frontEnds.add( be );
            }
            else if(be.getType().equals(BaseElementType.CentralLO) ) {
                centralRacks.add( be );
            }
            else if(be.getType().equals(BaseElementType.AOSTiming) ) {
                masterClocks.add( be );
            }
            else if(be.getType().equals(BaseElementType.PhotonicReference) ) {
                photonicRefs.add( be );
            }
            else if(be.getType().equals(BaseElementType.HolographyTower) ) {
                holographyTowers.add( be );
            }
            else if(be.getType().equals(BaseElementType.WeatherStationController) ) {
                weatherStations.add( be );
            }
            else {
            	unknowns.add( be);
            }
        }
        children.add(new XpDelaysModel(configuration));
        children.add( antennas );
        children.add( pads );
        children.add( frontEnds );
        children.add( centralRacks );
        children.add( masterClocks );
        children.add( photonicRefs );
        children.add( holographyTowers );
        children.add( weatherStations );
        
        TypeList roles = computeTypesList( configuration.getAssemblies() );
        children.add( roles );

        if(unknowns.size() > 0) {
        	children.add( unknowns );
        }
        
        retVal = children.toArray( retVal );
        
        return retVal;
	}

	@Override
	public Image getFirstColumnImage() {
		return RcpUtils.getImage("icons/configuration.png");
	}

	@Override
	public String getFirstColumnText() {
		return configuration.getName();
	}

	@Override
	public Image getSecondColumnImage() {
		return null;
	}

	@Override
	public String getSecondColumnText() {
		return null;
	}

	@Override
	public Image getThirdColumnImage() {
		return null;
	}
	
	@Override
	public String getThirdColumnText() {
		return null;
	}
	
	@Override
	public boolean hasChildren() {
		return true;
	}

	@Override
	public Font getFont() {
		return null;
	}

	@Override
	public Color getForeground() {
		return null;
	}
	
	/**
     * Bin the input assemblies in lists, one list per LruType
     * @return The list of all bins
     */
    private TypeList computeTypesList( Set<Assembly> assemblies ) 
    {
    	try {
        	if(!hydrated) {
        		for(Assembly assembly : assemblies) 
        		{
        			AssemblyConversationUtils.getInstance().hydrateAssembly(assembly);
        			AssemblyConversationUtils.getInstance().hydrateAssemblyType(assembly.getAssemblyType());
        		}
        		hydrated = true;
        	}
		} catch (Exception ex) {
			hydrated = false;
			throw new RuntimeException("Unable to hydrate assemblies", ex);
		}
		
        // Compute all type names we have
        
		List<AssemblyType> types = 
			((ConfigHelperFactory)ConfigHelperFactory.getInstance()).getAssemblyTypes();

        // For each type, create a list of Assemblies of that type
        // and add that list to the list we return
        TypeList ret = new TypeList(configuration);
        for(AssemblyType assType : types)
        {
        	AssemblyList assemblyList = new AssemblyList( assType, this.configuration );
        	for( Assembly assembly : assemblies ) 
        	{
        		if( assembly.getAssemblyType().equals( assType ) ) 
        		{
        			assemblyList.add( assembly );
        		}
        	}
        	if(assemblyList.size() > 0)
        	{
        		ret.add( assemblyList );
        	}
        }

        return ret;
    }
    
//  /**
//  * @return The list of all roles for the input assemblies; roles are unique.
//  */
// private TypeList computeRolesList( Set<Assembly> assemblies ) {
//     
//     // Compute all role names we have
//     Set<AssemblyRole> roles = new HashSet<AssemblyRole>();
//     for( Assembly assembly : assemblies ) {
//         Set<AssemblyRole> aRoles = assembly.getAssemblyType().getRoles();
//         for( AssemblyRole role : aRoles ) {
//             roles.add( role );
//         }
//     }
//
//     // For each role name, create a list of Assemblies with that role
//     // and add that list to the list we return
//     TypeList ret = new TypeList();
//     for( AssemblyRole role : roles ) {
//         AssemblyList assemblyList = new AssemblyList( role );
//         ret.add( assemblyList );
//         for( Assembly assembly : assemblies ) {
//             for( AssemblyRole assemblyRole : assembly.getAssemblyType().getRoles() ) {
//                 if( assemblyRole.equals( role )) {
//                     assemblyList.add( assembly );
//                 }
//             }
//         }
//     }
//     
//     return ret;
// }
}
