/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdbgui.views.providers.helpers.config;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import alma.obops.tmcdb.alarms.ui.tree.helpers.ListHelper;
import alma.obops.tmcdb.alarms.ui.tree.helpers.ThreeColumnDomainObjectHelper;
import alma.obops.tmcdb.alarms.ui.tree.helpers.factory.ThreeColumnDomainObjectHelperFactory;
import alma.obops.tmcdbgui.utils.conversation.AssemblyTypeConversationUtils;
import alma.obops.tmcdbgui.utils.conversation.BaseElementConversationUtils;
import alma.obops.tmcdbgui.utils.conversation.HwConfigurationConversationUtils;
import alma.obops.tmcdbgui.utils.conversation.StartupScenarioConversationUtils;
import alma.obops.tmcdbgui.views.providers.typedlists.AntennaList;
import alma.obops.tmcdbgui.views.providers.typedlists.AssemblyList;
import alma.obops.tmcdbgui.views.providers.typedlists.FrontEndList;
import alma.obops.tmcdbgui.views.providers.typedlists.PadList;
import alma.obops.tmcdbgui.views.providers.typedlists.TypeList;
import alma.tmcdb.domain.AcaCorrDelays;
import alma.tmcdb.domain.Antenna;
import alma.tmcdb.domain.AntennaToPad;
import alma.tmcdb.domain.Assembly;
import alma.tmcdb.domain.AssemblyType;
import alma.tmcdb.domain.BaseElement;
import alma.tmcdb.domain.BaseElementStartupType;
import alma.tmcdb.domain.BaseElementType;
import alma.tmcdb.domain.FocusModel;
import alma.tmcdb.domain.FrontEnd;
import alma.tmcdb.domain.HolographyTowerToPad;
import alma.tmcdb.domain.HwConfiguration;
import alma.tmcdb.domain.Pad;
import alma.tmcdb.domain.PointingModel;

/**
 * Factory which creates helper objects for the configuration tree, as needed.
 * @author sharrington
 */
public class ConfigHelperFactory implements ThreeColumnDomainObjectHelperFactory
{
   private static ConfigHelperFactory singletonInstance = null;
   private static Map<HwConfiguration, Map<Object, ThreeColumnDomainObjectHelper>> configurationHelperInstances = new HashMap<HwConfiguration, Map<Object, ThreeColumnDomainObjectHelper>>();
   private List<AssemblyType> types;
   
   /**
    * Getter for the singleton instance, creating the instance if needed.
    * @return the singleton instance.
    */
   public synchronized static ThreeColumnDomainObjectHelperFactory getInstance()
   {
      if(null == singletonInstance)
      {
         singletonInstance = new ConfigHelperFactory();
         try {
        	 singletonInstance.types =  AssemblyTypeConversationUtils.getInstance().findAllAssemblyTypes();
         }
         catch(Exception ex) {
        	 ex.printStackTrace();
        	 throw new RuntimeException("Cold not hydrate assembly types", ex);
         }
      }
      return singletonInstance;
   }

   public List<AssemblyType> getAssemblyTypes()
   {
	   return this.types;
   }
   
   public AssemblyType getAssemblyType(AssemblyType assType) 
   {
	   AssemblyType retVal = null;
	   
	   for(AssemblyType type : types)
	   {
		   if(type.getName().equals(assType.getName()))
		   {
			   retVal = type;
			   break;
		   }
	   }
	   
	   return retVal;
   }
   
   public AssemblyType[] findAssemblyTypesByBaseElementStartupType(BaseElementStartupType type) 
   {
	   AssemblyType[] retVal = null;
	   List<AssemblyType> listOfTypes = new ArrayList<AssemblyType>();
	   for(AssemblyType assType : types)
	   {
		   if(assType.getBaseElementType().equals(StartupScenarioConversationUtils.getBaseElementTypeFromBaseElementStartupType(type)))
		   {
			   listOfTypes.add(assType);
		   }
	   }
	   retVal = new AssemblyType[listOfTypes.size()];
	   retVal = listOfTypes.toArray(retVal);
	   return retVal;
   }
   
   /**
	 * Clears the helper caches (which are used for performance optimization); 
	 * this method should usually be called after a reload of configurations, 
	 * so that we don't get hibernate duplicate object id exceptions, and so forth.
	 */
   public synchronized static void clearCaches() {
		configurationHelperInstances.clear();
   }
   
   @Override
   public ThreeColumnDomainObjectHelper getHelper(Object object)
   {
      ThreeColumnDomainObjectHelper retVal = null;
      
      if(object instanceof HwConfiguration) 
      {
         retVal = getHelper((HwConfiguration)object);   
      }
      else if(object instanceof Antenna)
      {
         retVal = getHelper((Antenna)object);
      }
      else if(object instanceof AcaCorrDelays)
      {
         retVal = getHelper((AcaCorrDelays)object);
      }
      else if(object instanceof Pad) 
      {
         retVal = getHelper((Pad)object);
      }
      else if(object instanceof FrontEnd)
      {
         retVal = getHelper((FrontEnd)object);
      }
      else if(object instanceof AntennaList)
      {
         retVal = getHelper((AntennaList)object);
      }
      else if(object instanceof PadList)
      {
         retVal = getHelper((PadList)object);
      }
      else if(object instanceof FrontEndList)
      {
         retVal = getHelper((FrontEndList)object);
      }
      else if(object instanceof CentralRackList)
      {
         retVal = getHelper((CentralRackList)object);
      }
      else if(object instanceof MasterClockList)
      {
         retVal = getHelper((MasterClockList)object);
      }
      else if(object instanceof PhotonicReferenceList)
      {
         retVal = getHelper((PhotonicReferenceList)object);
      }
      else if(object instanceof WeatherStationList)
      {
         retVal = getHelper((WeatherStationList)object);
      }
      else if(object instanceof HolographyTowerList)
      {
         retVal = getHelper((HolographyTowerList)object);
      }
      else if(object instanceof TypeList)
      {
         retVal = getHelper((TypeList)object);
      }
      else if(object instanceof AssemblyList)
      {
         retVal = getHelper((AssemblyList)object);
      }
      else if(object instanceof List<?>)
      {
         retVal = getHelper((List<?>)object);
      }
      else if(object instanceof Assembly)
      {
    	  retVal = getHelper((Assembly)object);
      }
      else if(object instanceof HolographyTowerToPad)
      {
         retVal = getHelper((HolographyTowerToPad)object);
      }
      else if(object instanceof PointingModel)
      {
    	  retVal = getHelper((PointingModel)object);
      }
      else if(object instanceof FocusModel)
      {
    	  retVal = getHelper((FocusModel)object);
      }
      else if(object instanceof DelayModel)
      {
    	  retVal = getHelper((DelayModel)object);
      }
      else if(object instanceof XpDelaysModel)
      {
    	  retVal = getHelper((XpDelaysModel)object);
      }
      else if(object instanceof AntennaToPad)
      {
    	  retVal = getHelper((AntennaToPad)object);
      }
      else if(object instanceof BaseElement)
      {
         retVal = getHelper((BaseElement)object);
      }
      else if(object == null) {
    	  retVal = null;
      }
      else 
      {
          failUnsupported(object);
      }
   
      return retVal;
   }

   /**
    * Private constructor to enforce singleton pattern.
    */
   private ConfigHelperFactory()
   {   
   }
   
   private static void failUnsupported(Object element) {
      // Should never happen
      String msg = "Unsupported class: " + element.getClass().getName();
      IllegalArgumentException e = new IllegalArgumentException( msg );
      e.printStackTrace();
      throw e;
   }
   
   public synchronized static ThreeColumnDomainObjectHelper getHelper(HwConfiguration config)
   {
	   Map<Object, ThreeColumnDomainObjectHelper> helperMapForConfig = configurationHelperInstances.get(config);
	   if(null == helperMapForConfig)
	   {
		   helperMapForConfig = new HashMap<Object, ThreeColumnDomainObjectHelper>();
		   configurationHelperInstances.put(config, helperMapForConfig);
	   }
	  
	  ThreeColumnDomainObjectHelper retVal = helperMapForConfig.get(config);
      if(retVal == null) {
         retVal = new ConfigurationHelper(config);
         helperMapForConfig.put(config, retVal);         
      }
      return retVal;
   }
   
   private synchronized static ThreeColumnDomainObjectHelper getHelper(Antenna antenna)
   {
	   Map<Object, ThreeColumnDomainObjectHelper> helperMapForConfig = configurationHelperInstances.get(antenna.getConfiguration());
	   if(null == helperMapForConfig)
	   {
		   helperMapForConfig = new HashMap<Object, ThreeColumnDomainObjectHelper>();
		   configurationHelperInstances.put(antenna.getConfiguration(), helperMapForConfig);
	   } 
      ThreeColumnDomainObjectHelper retVal = helperMapForConfig.get(antenna);
      if(retVal == null) {
         retVal = new AntennaHelper(antenna);
         helperMapForConfig.put(antenna, retVal);
      }
      return retVal;
   }
   
   public synchronized static void removeAntennaHelper(Antenna antenna)
   {
	   Map<Object, ThreeColumnDomainObjectHelper> helperMapForConfig = configurationHelperInstances.get(antenna.getConfiguration());
	   if(null != helperMapForConfig)
	   {
		   Object helper = helperMapForConfig.get(antenna);
		   helperMapForConfig.remove(helper);
	   }
   }
   
   private synchronized static ThreeColumnDomainObjectHelper getHelper(AntennaToPad a2p)
   {
	   if(!a2p.getPad().getConfiguration().getId().equals(a2p.getAntenna().getConfiguration().getId())) {
		   try {
			   HwConfigurationConversationUtils.getInstance().hydrateConfigurationHashCode(a2p.getPad().getConfiguration());
			   BaseElementConversationUtils.getInstance().hydrateAntenna(a2p.getAntenna());
		   } catch (Exception e) {
			   e.printStackTrace();
			   throw new RuntimeException("Could not hydrate antenna to pad mapping", e);
		   }
	   }
	   Map<Object, ThreeColumnDomainObjectHelper> helperMapForConfig = configurationHelperInstances.get(a2p.getPad().getConfiguration());
	   if(null == helperMapForConfig)
	   {
		   helperMapForConfig = new HashMap<Object, ThreeColumnDomainObjectHelper>();
		   configurationHelperInstances.put(a2p.getPad().getConfiguration(), helperMapForConfig);
	   } 
	   ThreeColumnDomainObjectHelper retVal = helperMapForConfig.get(a2p);
	   if(retVal == null) {
		   retVal = new AntennaToPadHelper(a2p);
		   helperMapForConfig.put(a2p, retVal);
	   }
	   return retVal;
   }
   
   
   private synchronized static ThreeColumnDomainObjectHelper getHelper(HolographyTowerToPad h2p)
   {
	   if(!h2p.getPad().getConfiguration().getId().equals(h2p.getHolographyTower().getConfiguration().getId())) {
		   try {
			   HwConfigurationConversationUtils.getInstance().hydrateConfigurationHashCode(h2p.getHolographyTower().getConfiguration());
			   BaseElementConversationUtils.getInstance().hydratePad(h2p.getPad());
			   HwConfigurationConversationUtils.getInstance().hydrateConfigurationHashCode(h2p.getPad().getConfiguration());
			   
		   } catch (Exception e) {
			   e.printStackTrace();
			   throw new RuntimeException("Could not hydrate holography tower to pad mapping", e);
		   }
	   }
	   Map<Object, ThreeColumnDomainObjectHelper> helperMapForConfig = configurationHelperInstances.get(h2p.getHolographyTower().getConfiguration());
	   if(null == helperMapForConfig)
	   {
		   helperMapForConfig = new HashMap<Object, ThreeColumnDomainObjectHelper>();
		   configurationHelperInstances.put(h2p.getHolographyTower().getConfiguration(), helperMapForConfig);
	   } 
	   ThreeColumnDomainObjectHelper retVal = helperMapForConfig.get(h2p);
	   if(retVal == null) {
		   retVal = new HolographyTowerToPadHelper(h2p);
		   helperMapForConfig.put(h2p, retVal);
	   }
	   return retVal;
   }

   private synchronized static ThreeColumnDomainObjectHelper getHelper(Pad pad)
   {
	   Map<Object, ThreeColumnDomainObjectHelper> helperMapForConfig = configurationHelperInstances.get(pad.getConfiguration());
	   if(null == helperMapForConfig)
	   {
		   helperMapForConfig = new HashMap<Object, ThreeColumnDomainObjectHelper>();
		   configurationHelperInstances.put(pad.getConfiguration(), helperMapForConfig);
	   } 
      ThreeColumnDomainObjectHelper retVal = helperMapForConfig.get(pad);
      if(null == retVal) {
         retVal =  new PadHelper(pad);
         helperMapForConfig.put(pad, retVal);
      }
      return retVal;   
   }
   
   private synchronized static ThreeColumnDomainObjectHelper getHelper(FrontEnd frontEnd)
   { 
	   Map<Object, ThreeColumnDomainObjectHelper> helperMapForConfig = configurationHelperInstances.get(frontEnd.getConfiguration());
	   if(null == helperMapForConfig)
	   {
		   helperMapForConfig = new HashMap<Object, ThreeColumnDomainObjectHelper>();
		   configurationHelperInstances.put(frontEnd.getConfiguration(), helperMapForConfig);
	   } 
      ThreeColumnDomainObjectHelper retVal = helperMapForConfig.get(frontEnd);
      if(null == retVal) {
         retVal =  new FrontEndHelper(frontEnd);
         helperMapForConfig.put(frontEnd, retVal);
      }
      return retVal;
   }
   
   private synchronized static ThreeColumnDomainObjectHelper getHelper(AcaCorrDelays acaDelays)
   { 
	   Map<Object, ThreeColumnDomainObjectHelper> helperMapForConfig = configurationHelperInstances.get(acaDelays.getAntenna().getConfiguration());
	   if(null == helperMapForConfig)
	   {
		   helperMapForConfig = new HashMap<Object, ThreeColumnDomainObjectHelper>();
		   configurationHelperInstances.put(acaDelays.getAntenna().getConfiguration(), helperMapForConfig);
	   } 
      ThreeColumnDomainObjectHelper retVal = helperMapForConfig.get(acaDelays);
      if(null == retVal) {
         retVal =  new AcaCorrDelaysHelper(acaDelays);
         helperMapForConfig.put(acaDelays, retVal);
      }
      return retVal;
   }
   
   private synchronized static ThreeColumnDomainObjectHelper getHelper(BaseElement baseElement)
   {
	  Map<Object, ThreeColumnDomainObjectHelper> helperMapForConfig = configurationHelperInstances.get(baseElement.getConfiguration());
      ThreeColumnDomainObjectHelper retVal = helperMapForConfig.get(baseElement);
      
      if(null == retVal) {
    	  if(baseElement.getType().equals(BaseElementType.CentralLO)) {
    		  retVal = new CentralRackHelper(baseElement);
    	  }
    	  else if(baseElement.getType().equals(BaseElementType.AOSTiming)) {
    		  retVal = new MasterClockHelper(baseElement);
    	  } 
    	  else if(baseElement.getType().equals(BaseElementType.PhotonicReference)) {
    		  retVal = new PhotonicReferenceHelper(baseElement);
    	  } 
    	  else if(baseElement.getType().equals(BaseElementType.WeatherStationController)) {
    		  retVal = new WeatherStationHelper(baseElement);
    	  } 
    	  else if(baseElement.getType().equals(BaseElementType.HolographyTower)) {
    		  retVal = new HolographyTowerHelper(baseElement);
    	  } 
    	  else {
            retVal = new BaseElementHelper(baseElement);
         }
         helperMapForConfig.put(baseElement, retVal);
      }
      return retVal;      
   }
   
   private static ThreeColumnDomainObjectHelper getHelper(AntennaList list)
   {
	   ThreeColumnDomainObjectHelper retVal = new AntennaListHelper(list);
      return retVal;
   }
   
   private static ThreeColumnDomainObjectHelper getHelper(PadList list)
   {
      ThreeColumnDomainObjectHelper retVal = new PadListHelper(list);
      return retVal;
   }
   
   private static ThreeColumnDomainObjectHelper getHelper(FrontEndList list)
   {
      ThreeColumnDomainObjectHelper retVal = new FrontEndListHelper(list);
      return retVal;
   }
   
   private static ThreeColumnDomainObjectHelper getHelper(CentralRackList list)
   {
      ThreeColumnDomainObjectHelper retVal = new CentralRackListHelper(list);
      return retVal;
   }
   
   private static ThreeColumnDomainObjectHelper getHelper(MasterClockList list)
   {
      ThreeColumnDomainObjectHelper retVal = new MasterClockListHelper(list);
      return retVal;
   }
   
   private static ThreeColumnDomainObjectHelper getHelper(PhotonicReferenceList list)
   {
      ThreeColumnDomainObjectHelper retVal = new PhotonicReferenceListHelper(list);
      return retVal;
   }
   
   private static ThreeColumnDomainObjectHelper getHelper(WeatherStationList list)
   {
      ThreeColumnDomainObjectHelper retVal = new WeatherStationListHelper(list);
      return retVal;
   }
   
   private static ThreeColumnDomainObjectHelper getHelper(HolographyTowerList list)
   {
      ThreeColumnDomainObjectHelper retVal = new HolographyTowerListHelper(list);
      return retVal;
   }
   
   private static ThreeColumnDomainObjectHelper getHelper(TypeList list)
   {
      ThreeColumnDomainObjectHelper retVal = new TypeListHelper(list);
      return retVal;
   }   
   
   private static ThreeColumnDomainObjectHelper getHelper(AssemblyList list)
   {
      ThreeColumnDomainObjectHelper retVal = new AssemblyListHelper(list);
      return retVal;
   }
   
   private static ThreeColumnDomainObjectHelper getHelper(List<?> list)
   {
      ThreeColumnDomainObjectHelper retVal = new ListHelper(list);
      return retVal;
   }
   
   private static ThreeColumnDomainObjectHelper getHelper(Assembly assembly)
   {
      ThreeColumnDomainObjectHelper retVal = new AssemblyHelper(assembly);
      return retVal;
   }
   
   private static ThreeColumnDomainObjectHelper getHelper(FocusModel focusModel)
   {
      ThreeColumnDomainObjectHelper retVal = new FocusModelHelper(focusModel);
      return retVal;
   }
   
   private static ThreeColumnDomainObjectHelper getHelper(DelayModel delayModel)
   {
      ThreeColumnDomainObjectHelper retVal = new DelayModelHelper(delayModel);
      return retVal;
   }
   
   private static ThreeColumnDomainObjectHelper getHelper(XpDelaysModel delayModel)
   {
      ThreeColumnDomainObjectHelper retVal = new XpDelaysModelHelper(delayModel);
      return retVal;
   }
   
   private static ThreeColumnDomainObjectHelper getHelper(PointingModel pointingModel)
   {
      ThreeColumnDomainObjectHelper retVal = new PointingModelHelper(pointingModel);
      return retVal;
   }

}
