/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdbgui.views.providers.helpers.config;

import org.eclipse.swt.graphics.Image;

import alma.obops.tmcdb.alarms.ui.utils.RcpUtils;
import alma.obops.tmcdbgui.utils.conversation.BaseElementConversationUtils;
import alma.tmcdb.domain.AcaCorrDelays;
import alma.tmcdb.domain.Antenna;
import alma.tmcdb.domain.AntennaToPad;
import alma.tmcdb.domain.FocusModel;
import alma.tmcdb.domain.Pad;
import alma.tmcdb.domain.PointingModel;

/**
 * Helper class for Antenna domain object.
 * @author sharrington
 */
public class AntennaHelper extends BaseElementHelper 
{
	/**
	 * Constructor.
	 * @param antenna the antenna for which this helper class will provide info (images, text, children, etc).
	 */
	public AntennaHelper(Antenna antenna) {
		super(antenna);
	}

	@Override
	public Image getFirstColumnImage() {
		return RcpUtils.getImage("icons/antenna.png");
	}
	
	@Override
	public Object[] getChildren() 
	{
		Antenna antenna = (Antenna) this.baseElement;
		try {
			BaseElementConversationUtils.getInstance().hydrateAntenna(antenna);
		} catch (Exception e) {
			e.printStackTrace();
			throw new RuntimeException("Could not hydrate antenna", e);
		}
		
		Object[] retVal = null;

		AntennaToPad a2p = findCurrentAntennaToPadForAntenna(antenna);
		if(a2p == null && antenna.getConfiguration().getGlobalConfiguration() != null) {
			try {
				a2p = BaseElementConversationUtils.getInstance().findOpenAntennaToPadAssignmentForAntennaInGlobal(antenna);
			} catch (Exception e) {
				e.printStackTrace();
				throw new RuntimeException("Could not find antenna to pad assignments for antenna", e);
			}
		}
		
		int numChildren = 4; // there are always at least 4 children for the antenna
		if(a2p != null) {
			// if there is an a2p assignment, then there are 5 children
			numChildren++;	
		}
		retVal = new Object[numChildren]; 

		if(antenna.getAcaCorrDelays() == null) {
			retVal[0] = new AcaCorrDelays(antenna, 0.0d, 0.0d, 0.0d, 0.0d);
		} else {
			retVal[0] = antenna.getAcaCorrDelays();
		}
		Pad pad = (a2p != null) ? a2p.getPad() : null;
		retVal[1] = new DelayModel(antenna, pad);
		if(antenna.getFocusModels().size() > 0) {
			retVal[2] = antenna.getFocusModels().iterator().next();
		} else {
			retVal[2] = new FocusModel();
			((FocusModel)retVal[2]).setAntenna(antenna);
		}
		if(antenna.getPointingModels().size() > 0) {
			retVal[3] = antenna.getPointingModels().iterator().next();
		} else {
			retVal[3] = new PointingModel();
			((PointingModel)retVal[3]).setAntenna(antenna);
		}
		if(a2p != null) {
			retVal[4] = a2p;
		}
		
		return retVal;
	}
	
	public static AntennaToPad findCurrentAntennaToPadForAntenna(Antenna antenna) 
	{
		try {
			AntennaToPad[] a2ps = BaseElementConversationUtils.getInstance().findOpenAntennaToPadAssignmentsForAntenna(antenna);
			for(AntennaToPad a2p : a2ps) {
				if(a2p.getEndTime() == null || a2p.getEndTime().equals(Long.valueOf((0l)))) {
					BaseElementConversationUtils.getInstance().hydratePad(a2p.getPad());
					BaseElementConversationUtils.getInstance().hydrateAntenna(a2p.getAntenna());
					return a2p;
				}
			}
		} catch (Exception e) {
			e.printStackTrace();
			throw new RuntimeException("Could not find antennatopad assignments for antenna");
		}
		
		return null;
	}
	
	@Override
	public boolean hasChildren() {
		return true; // there are always 3 (or 4) children for the antenna
	}
}
