/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdbgui.views.providers;

import org.eclipse.jface.viewers.CellEditor;
import org.eclipse.jface.viewers.EditingSupport;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.TextCellEditor;

import alma.obops.tmcdbgui.widgets.support.DirtyListener;

/**
 * Editing support for the table used to edit xp (cross polarization) delays.
 * @author sharring
 */
public class XpDelaysEditingSupport extends EditingSupport 
{
	private CellEditor _editor;
	private DirtyListener _listener;
	private int _column;
	
	public XpDelaysEditingSupport(final TableViewer viewer, int column, DirtyListener listener) 
	{
		super(viewer);
		this._listener = listener;
		_column = column;
		_editor = new TextCellEditor(viewer.getTable());
		_editor.getControl().addTraverseListener(new TabTraverseListener());
		_editor.getControl().addKeyListener(new TabKeyListener(_editor, viewer, _column));
		
		switch(column) {
		case 0:
			_editor.setValidator(null);
			break;
		case 1:
		case 2:
		case 3:
		case 4:
		case 5:
		case 6:
		case 7:
		case 8:
			_editor.setValidator(new ScientificNotationCellEditorValidator());
			break;
		}
	}

	@Override
	protected boolean canEdit(Object element) {
		return _column != 0;
	}

	@Override
	protected CellEditor getCellEditor(Object element) {
		return _editor;
	}

	@Override
	protected Object getValue(Object element) 
	{
		XpDelayModelRow row = (XpDelayModelRow)element;
		Object retVal = null;
		
		switch(_column) {
		case 0:
			retVal = "Band " + row.getBand();
			break;
		case 1:
			retVal = row.getUsbBasebandZeroDelay().getDelay().toString();
			break;
		case 2:
			retVal = row.getLsbBasebandZeroDelay().getDelay().toString();
			break;
		case 3:
			retVal = row.getUsbBasebandOneDelay().getDelay().toString();
			break;
		case 4:
			retVal = row.getLsbBasebandOneDelay().getDelay().toString();
			break;
		case 5:
			retVal = row.getUsbBasebandTwoDelay().getDelay().toString();
			break;
		case 6:
			retVal = row.getLsbBasebandTwoDelay().getDelay().toString();
			break;
		case 7:
			retVal = row.getUsbBasebandThreeDelay().getDelay().toString();
			break;
		case 8:
			retVal = row.getLsbBasebandThreeDelay().getDelay().toString();
			break;
		default:
			retVal = null;
			break;
		}
		return retVal;
	}

	@Override
	protected void setValue(Object element, Object value) 
	{
		XpDelayModelRow row = (XpDelayModelRow)element;
		if(value == null) {
			return;
		}
		String newValue = (String)value;
		switch(_column) 
		{
		case 0:
			break;
		case 1:
			if( !(row.getUsbBasebandZeroDelay().getDelay().equals(Double.valueOf(newValue))))
			{
				row.getUsbBasebandZeroDelay().setDelay(Double.valueOf(newValue));
				_listener.setDirty(true);
				getViewer().refresh(element, true);
			}
			break;
		case 2:
			if( !(row.getLsbBasebandZeroDelay().getDelay().equals(Double.valueOf(newValue))))
			{
				row.getLsbBasebandZeroDelay().setDelay(Double.valueOf(newValue));
				_listener.setDirty(true);
				getViewer().refresh(element, true);
			}
			break;
		case 3:
			if( !(row.getUsbBasebandOneDelay().getDelay().equals(Double.valueOf(newValue))))
			{
				row.getUsbBasebandOneDelay().setDelay(Double.valueOf(newValue));
				_listener.setDirty(true);
				getViewer().refresh(element, true);
			}
			break;
		case 4:
			if( !(row.getLsbBasebandOneDelay().getDelay().equals(Double.valueOf(newValue))))
			{
				row.getLsbBasebandOneDelay().setDelay(Double.valueOf(newValue));
				_listener.setDirty(true);
				getViewer().refresh(element, true);
			}
			break;
		case 5:
			if( !(row.getUsbBasebandTwoDelay().getDelay().equals(Double.valueOf(newValue))))
			{
				row.getUsbBasebandTwoDelay().setDelay(Double.valueOf(newValue));
				_listener.setDirty(true);
				getViewer().refresh(element, true);
			}
			break;
		case 6:
			if( !(row.getLsbBasebandTwoDelay().getDelay().equals(Double.valueOf(newValue))))
			{
				row.getLsbBasebandTwoDelay().setDelay(Double.valueOf(newValue));
				_listener.setDirty(true);
				getViewer().refresh(element, true);
			}
			break;
		case 7:
			if( !(row.getUsbBasebandThreeDelay().getDelay().equals(Double.valueOf(newValue))))
			{
				row.getUsbBasebandThreeDelay().setDelay(Double.valueOf(newValue));
				_listener.setDirty(true);
				getViewer().refresh(element, true);
			}
			break;
		case 8:
			if( !(row.getLsbBasebandThreeDelay().getDelay().equals(Double.valueOf(newValue))))
			{
				row.getLsbBasebandThreeDelay().setDelay(Double.valueOf(newValue));
				_listener.setDirty(true);
				getViewer().refresh(element, true);
			}
			break;
		default:
			break;
		}
	}
}
