/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdbgui.views.providers;

import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.ViewerSorter;

import alma.acs.tmcdb.AcsService;
import alma.acs.tmcdb.ChannelMapping;
import alma.acs.tmcdb.Component;
import alma.acs.tmcdb.Computer;
import alma.acs.tmcdb.Container;
import alma.acs.tmcdb.ContainerStartupOption;
import alma.acs.tmcdb.DomainsMapping;
import alma.acs.tmcdb.Manager;
import alma.acs.tmcdb.NotificationServiceMapping;
import alma.obops.tmcdbgui.views.providers.helpers.software.AcsServiceComparator;
import alma.obops.tmcdbgui.views.providers.helpers.software.ChannelMappingComparator;
import alma.obops.tmcdbgui.views.providers.helpers.software.ComponentComparator;
import alma.obops.tmcdbgui.views.providers.helpers.software.ComputerComparator;
import alma.obops.tmcdbgui.views.providers.helpers.software.ContainerComparator;
import alma.obops.tmcdbgui.views.providers.helpers.software.DomainsMappingComparator;
import alma.obops.tmcdbgui.views.providers.helpers.software.ManagerComparator;
import alma.obops.tmcdbgui.views.support.AcsServiceWrapper;

/**
 * Sorter for the SW Deployment View. It is used to sort Component, Container and Computer objects
 * within the view. It makes use of the \{@link ComponentComparator}, {@link ContainerComparator}
 * and {@link ComputerComparator} classes respectively to sort the objects. It also takes
 * care that the three main groups of the view (Computers, Undeployed Containers and Undeployed
 * Components) appear in that very same order in the view
 *  
 * @author rtobar
 */
public class SwDeploymentTreeSorter extends ViewerSorter {

	AcsServiceComparator acsServiceComparator = new AcsServiceComparator();
	ChannelMappingComparator channelMappingComparator = new ChannelMappingComparator();
	DomainsMappingComparator domainsMappingComparator = new DomainsMappingComparator();
	ComputerComparator  computerComparator  = new ComputerComparator();
	ContainerComparator containerComparator = new ContainerComparator();
	ComponentComparator componentComparator = new ComponentComparator();
	ManagerComparator managerComparator = new ManagerComparator();

	@Override
	public int compare(Viewer viewer, Object e1, Object e2) {
		if(e1 instanceof Manager && e2 instanceof Manager)  
			return managerComparator.compare((Manager)e1, (Manager)e2);
		else if(e1 instanceof Manager) 
			return -1;
		else if(e2 instanceof Manager) 
			return 1;
		else if( e1 instanceof Computer && e2 instanceof Computer )
			return computerComparator.compare((Computer)e1, (Computer)e2);
		else if( e1 instanceof Container && e2 instanceof Container )
			return containerComparator.compare((Container)e1, (Container)e2);
		else if( e1 instanceof Component && e2 instanceof Component )
			return componentComparator.compare((Component)e1, (Component)e2);
		else if( (e1 instanceof AcsService && e2 instanceof AcsService)) {
			return acsServiceComparator.compare((AcsService) e1, (AcsService)e2);
		} 
		else if(e1 instanceof AcsServiceWrapper && e2 instanceof AcsServiceWrapper) {
			return acsServiceComparator.compare( ((AcsServiceWrapper)e1).getAcsService(), ((AcsServiceWrapper)e2).getAcsService());
		}
		else if( e1 instanceof ChannelMapping && e2 instanceof ChannelMapping ) 
			return channelMappingComparator.compare((ChannelMapping) e1, (ChannelMapping)e2);
		else if( e1 instanceof DomainsMapping && e2 instanceof DomainsMapping ) 
			return domainsMappingComparator.compare((DomainsMapping) e1, (DomainsMapping)e2);
		else if( e1 instanceof ContainerStartupOption && e2 instanceof Component ) {
			return -1;
		}
		else if( e1 instanceof ContainerStartupOption[] && e2 instanceof Component ) {
			return -1;
		}
		else if( e1 instanceof AcsService[] && e2 instanceof AcsService ) {
			return -1;
		}

		// Top level elements are arrenged in a specific way
		// We do the check both ways just in case
		if( e1 instanceof Computer[] && (e2 instanceof Container[] || e2 instanceof AcsService[] || e2 instanceof Component[]) )
			return -1;
		else if( e1 instanceof Container[] && (e2 instanceof Component[] || e2 instanceof AcsService[]))
			return -1;
		else if( e1 instanceof Container[] && e2 instanceof Computer[] )
			return 1;
		else if( e1 instanceof Component[] && (e2 instanceof Computer[] || e2 instanceof Container[]))
			return 1;
		else if( e1 instanceof Component[] && (e2 instanceof AcsService[] || e2 instanceof NotificationServiceMapping[]))
			return -1;
		else if( e1 instanceof Manager[] && (e2 instanceof AcsService[] || e2 instanceof NotificationServiceMapping[])) {
			return -1;
		} 
		else if( e1 instanceof Manager[] && !(e2 instanceof AcsService[] || e2 instanceof NotificationServiceMapping[])) {
			return 1;
		}
		else if( e1 instanceof NotificationServiceMapping[] && e2 instanceof AcsService[]) {
			return -1;
		}
		else if( e1 instanceof NotificationServiceMapping[] && !(e2 instanceof AcsService[])) {
			return 1;
		}
		else if( (e1 instanceof AcsService[] && ((AcsService[])e1).length == 0) && ( e2 instanceof Component[])
				                            || ( e2 instanceof Container[]) 
				                            || ( e2 instanceof Computer[])
				                            || ( e2 instanceof Manager[]) 
				                            || ( e2 instanceof NotificationServiceMapping[])) {
			return 1;
		}
		else if(e2 instanceof AcsService[] && ((AcsService[])e2).length == 0) {
			return -1;
		}

		return super.compare(viewer, e1, e2);
	}

}
