/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
/**
 * ConfigurationTreeLabelProvider.java
 *
 * Copyright European Southern Observatory 2008
 */

package alma.obops.tmcdbgui.views.providers;

import org.eclipse.jface.viewers.ColumnLabelProvider;
import org.eclipse.jface.viewers.StyledString;
import org.eclipse.jface.viewers.DelegatingStyledCellLabelProvider.IStyledLabelProvider;
import org.eclipse.swt.graphics.Image;

import alma.acs.tmcdb.AcsService;
import alma.acs.tmcdb.BACIProperty;
import alma.acs.tmcdb.ChannelMapping;
import alma.acs.tmcdb.Component;
import alma.acs.tmcdb.Computer;
import alma.acs.tmcdb.Container;
import alma.acs.tmcdb.ContainerStartupOption;
import alma.acs.tmcdb.DomainsMapping;
import alma.acs.tmcdb.Manager;
import alma.acs.tmcdb.NotificationServiceMapping;
import alma.obops.tmcdb.alarms.ui.utils.RcpUtils;
import alma.obops.tmcdbgui.utils.ImageHelper;
import alma.obops.tmcdbgui.utils.LabelHelper;
import alma.obops.tmcdbgui.views.support.AcsServiceWrapper;

/**
 * Provide object name and label for a SW deployment tree
 * 
 * @author rtobar, Feb 19, 2010
 * 
 */


public class SwDeploymentTreeLabelProvider extends ColumnLabelProvider implements IStyledLabelProvider
{
	
	public SwDeploymentTreeLabelProvider() {
	}
	
    @Override
    public Image getImage( Object element ) 
    {
    	// Simple elements
    	if( element instanceof Computer )
    		return ImageHelper.getImage((Computer)element);
    	else if ( element instanceof Container )
    		return ImageHelper.getImage((Container)element);
    	else if ( element instanceof ContainerStartupOption )
    		return ImageHelper.getImage((ContainerStartupOption)element);
    	else if ( element instanceof BACIProperty )
    		return ImageHelper.getImage((BACIProperty)element);
    	else if ( element instanceof Component )
    		return ImageHelper.getImage((Component)element);
    	else if( element instanceof AcsService ) 
    		return ImageHelper.getImage(((AcsService)element));
    	else if( element instanceof AcsServiceWrapper ) 
    		return ImageHelper.getImage(((AcsServiceWrapper)element).getAcsService());
    	else if ( element instanceof Manager )
    		return ImageHelper.getImage((Manager)element);
    	else if( element instanceof NotificationServiceMapping ) 
    		return ImageHelper.getImage(((NotificationServiceMapping)element));
    	else if( element instanceof DomainsMapping ) 
    		return ImageHelper.getImage(((DomainsMapping)element));
    	else if( element instanceof ChannelMapping ) 
    		return ImageHelper.getImage(((ChannelMapping)element));

    	// The collections of computers, undeployed containers, 
    	// undeployed components, and undeployed ACS services, plus Container Startup Options
    	else if ( element instanceof Computer[] )
    		return RcpUtils.getImage("icons/computers.png");
    	else if ( element instanceof Container[] )
    		return RcpUtils.getImage("icons/undeployed_containers.gif");
    	else if ( element instanceof ContainerStartupOption[] )
    		return RcpUtils.getImage("icons/containerstartupoption.png");
    	else if ( element instanceof Component[] )
    		return RcpUtils.getImage("icons/undeployed_components.gif");
    	else if ( element instanceof AcsService[] && ((AcsService[]) element).length == 0 ) 
    		return RcpUtils.getImage("icons/undeployed_services.png");
    	else if ( element instanceof AcsService[] && ((AcsService[]) element).length != 0 && ((AcsService[]) element)[0].getComputer() != null ) 
    		return RcpUtils.getImage("icons/deployed_services.png");
    	else if ( element instanceof AcsServiceWrapper[]) 
    		return RcpUtils.getImage("icons/deployed_services.png");
    	else if ( element instanceof NotificationServiceMapping[]) 
    		return ImageHelper.getImage(new NotificationServiceMapping());
    	else if( element instanceof DomainsMapping[] ) 
    		return ImageHelper.getImage(new DomainsMapping());
    	else if( element instanceof ChannelMapping[] ) 
    		return ImageHelper.getImage(new ChannelMapping());
    	else if( element instanceof Manager[] ) 
    		return ImageHelper.getImage(new Manager());

    	else
    		failUnsupported(element);

    	return null;
    }

    @Override
    public String getText( Object element ) {

    	// Simple objects
    	if( element instanceof Computer )
    		return LabelHelper.getComputerLabel((Computer)element);
    	else if ( element instanceof BACIProperty )
    		return LabelHelper.getFullPath((BACIProperty)element, false);
    	else if ( element instanceof Container )
    		return LabelHelper.getFullPath((Container)element, false);
    	else if ( element instanceof ContainerStartupOption )
    		return ((ContainerStartupOption)element).getOptionName();
    	else if ( element instanceof Component )
    		return LabelHelper.getFullPath((Component)element, false);
    	else if( element instanceof AcsService ) {
    		AcsService service = (AcsService) element;
    		String instanceName =  service.getServiceInstanceName() == null ? "" : "." + service.getServiceInstanceName();
    		return ((AcsService) element).getServiceType() + instanceName;
    	}
    	else if( element instanceof AcsServiceWrapper ) {
    		AcsServiceWrapper serviceHolder = (AcsServiceWrapper) element;
    		StringBuffer instanceName = new StringBuffer(serviceHolder.getAcsService().getServiceInstanceName() == null ? 
    					"" : "." + serviceHolder.getAcsService().getServiceInstanceName());
    		instanceName.append(" on ").append(serviceHolder.getAcsService().getComputer().getNetworkName());
    		return ((AcsServiceWrapper) element).getAcsService().getServiceType() + instanceName.toString();
    	}
    	else if( element instanceof NotificationServiceMapping ) {
    		return ((NotificationServiceMapping) element).getDefaultNotificationService();
    	}
    	else if( element instanceof ChannelMapping ) {
    		return ((ChannelMapping) element).getName();
    	}
    	else if( element instanceof DomainsMapping ) {
    		return ((DomainsMapping) element).getName();
    	}
    	
    	else if( element instanceof Manager ) {
    		return ("Manager");
    	}

    	// Arrays of objects
    	else if( element instanceof Computer[] )
    		return "Computers";
		else if( element instanceof Container[] )
			return "Undeployed Containers";
		else if( element instanceof ContainerStartupOption[] )
			return "Startup options";
		else if( element instanceof Component[] )
			return "Undeployed Components";
		else if( element instanceof AcsService[] && ((AcsService[]) element).length != 0 && ((AcsService[]) element)[0].getComputer() != null)
			return "Services";
		else if( element instanceof NotificationServiceMapping[])
			return "Notification Service Mappings";
		else if( element instanceof ChannelMapping[])
			return "Channel Mappings";
		else if( element instanceof DomainsMapping[])
			return "Domain Mappings";
		else if( element instanceof Manager[])
			return "Managers";
		else if( element instanceof AcsServiceWrapper[] )
    		return "Services";

    	// Should never happen
    	else
    		failUnsupported(element);

    	// This also should never happen
    	return "";
    }

    private void failUnsupported(Object element) {
		// Should never happen
        String msg = "Unsupported class: " + element.getClass().getName();
        IllegalArgumentException e = new IllegalArgumentException( msg );
        e.printStackTrace();
        throw e;
	}

	@Override
	public boolean isLabelProperty(Object element, String property) {
		if( element instanceof Container ) {
			if( property.equals("path") || property.equals("containerName") )
				return true;
		}
		else if( element instanceof ContainerStartupOption ) {
			if( property.equals("optionName") )
				return true;
		}
		else if( element instanceof AcsService ) {
			if( property.equals("serviceInstanceName") || property.equals("serviceType") )
				return true;
		}
		else if( element instanceof AcsServiceWrapper ) {
			if( property.equals("serviceInstanceName") || property.equals("serviceType") )
				return true;
		}
		else if( element instanceof Component ) {
			if( property.equals("path") || property.equals("componentName") || property.equals("implLang") )
				return true;
		}
		else if( element instanceof Computer ) {
			if( property.equals("name") || property.equals("networkName") )
				return true;
		}
		else if( element instanceof BACIProperty ) {
			if( property.equals("propertyName") )
				return true;
		}
		return false;
	}

	@Override
	public StyledString getStyledText(Object element) {
		return new StyledString(getText(element));
	}

}
