/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
/**
 * SwDeploymentTreeContentsProvider.java
 *
 * Copyright European Southern Observatory 2008
 */

package alma.obops.tmcdbgui.views.providers;

import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;
import java.util.List;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.dialogs.ProgressMonitorDialog;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.eclipse.jface.viewers.ITreeContentProvider;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.viewers.Viewer;

import alma.acs.tmcdb.AcsService;
import alma.acs.tmcdb.BACIProperty;
import alma.acs.tmcdb.ChannelMapping;
import alma.acs.tmcdb.Component;
import alma.acs.tmcdb.Computer;
import alma.acs.tmcdb.Container;
import alma.acs.tmcdb.ContainerStartupOption;
import alma.acs.tmcdb.DomainsMapping;
import alma.acs.tmcdb.Manager;
import alma.acs.tmcdb.NetworkDevice;
import alma.acs.tmcdb.NotificationServiceMapping;
import alma.obops.tmcdbgui.utils.conversation.AcsServiceConversationUtils;
import alma.obops.tmcdbgui.utils.conversation.BaciConversationUtils;
import alma.obops.tmcdbgui.utils.conversation.ChannelMappingConversationUtils;
import alma.obops.tmcdbgui.utils.conversation.ComponentConversationUtils;
import alma.obops.tmcdbgui.utils.conversation.ComputerConversationUtils;
import alma.obops.tmcdbgui.utils.conversation.ContainerConversationUtils;
import alma.obops.tmcdbgui.utils.conversation.DomainsMappingConversationUtils;
import alma.obops.tmcdbgui.utils.conversation.HwConfigurationConversationUtils;
import alma.obops.tmcdbgui.utils.conversation.SwConfigurationConversationUtils;
import alma.obops.tmcdbgui.views.support.AcsServiceWrapper;
import alma.tmcdb.domain.HwConfiguration;

/**
 * Contents provide for a tree of software deployment
 * 
 * @author rtobar, Feb 19, 2010
 * 
 */



public class SwDeploymentTreeContentsProvider implements ITreeContentProvider, PropertyChangeListener 
{
	private static final String SERVICE_INSTANCE_NAME = "serviceInstanceName";
	private static final String SERVICE_TYPE = "serviceType";
	private HwConfiguration _currentConfig;
	private TreeViewer viewer;
	private Object[] topLevelObjects = new Object[] { new Manager[0], new Computer[0], new Container[0], new Component[0], new AcsServiceWrapper[0], new NotificationServiceMapping[0]};

    /**
     * @see org.eclipse.jface.viewers.IContentProvider#dispose()
     */
    public void dispose() {
        // no-op
    }

    /**
     * @see org.eclipse.jface.viewers.ITreeContentProvider#getChildren(java.lang.Object)
     */
    public Object[] getChildren( Object parent ) 
    {
    	try {
    		if( parent instanceof HwConfiguration )
    			return getElements(parent);
    		
    		else if( parent instanceof Component ) {
    			BaciConversationUtils.getInstance().hydrateBACIProperties((Component)parent);
    			List<BACIProperty> baciProps = new ArrayList<BACIProperty>(((Component)parent).getBACIProperties());
    			for(BACIProperty baciProp: baciProps) {
    				baciProp.addPropertyChangeListener("propertyName", this);
    			}
    			return baciProps.toArray();
    		}
    		else if( parent instanceof Computer ) {
    			ComputerConversationUtils.getInstance().hydrateContainers((Computer)parent);
    			ComputerConversationUtils.getInstance().hydrateAcsServices((Computer)parent);
    			List<Object> children = new ArrayList<Object>();
    			List<Container> conts = new ArrayList<Container>(((Computer)parent).getContainers());
    			for(Container cont: conts) {
    				cont.addPropertyChangeListener("containerName", this);
    				cont.addPropertyChangeListener("path", this);
    			}
    			if(conts.size() > 0) {
    				children.addAll(conts);
    			}
    			if (((Computer)parent).getAcsServices().size() > 0 ) {
    				children.add(((Computer) parent).getAcsServices().toArray(new AcsService[0]));
    			}
    			return children.toArray();
    		}
    		else if( parent instanceof Container ) {
    			ComponentConversationUtils.getInstance().hydrateComponents((Container)parent);
    			ContainerConversationUtils.getInstance().hydrateContainerStartupOptions((Container)parent);
    			List<Object> children = new ArrayList<Object>();
    			if(((Container)parent).getContainerStartupOptions().size() > 0) {
    				children.add(((Container) parent).getContainerStartupOptions().toArray(new ContainerStartupOption[0]));
    			}
    			children.addAll(((Container)parent).getComponents());
    			
    			for(Object obj: children) {
    				if(obj instanceof Component)
    				{
    					Component comp = (Component) obj;
    					comp.addPropertyChangeListener("componentName", this);
    					comp.addPropertyChangeListener("path", this);
    					comp.addPropertyChangeListener("implLang", this);
    					comp.addPropertyChangeListener("code", this);
    				}
    			}
    			return children.toArray();
    		}
    		else if( parent instanceof NotificationServiceMapping ) {
    			DomainsMappingConversationUtils.getInstance().hydrateDomainsMappings((NotificationServiceMapping)parent);
    			ChannelMappingConversationUtils.getInstance().hydrateChannelMappings((NotificationServiceMapping)parent);
    			List<Object> children = new ArrayList<Object>();
    			if(((NotificationServiceMapping)parent).getChannelMappings().size() > 0) {
    				children.add(((NotificationServiceMapping) parent).getChannelMappings().toArray(new ChannelMapping[0]));
    			}
    			if(((NotificationServiceMapping)parent).getDomainsMappings().size() > 0) {
    				children.add(((NotificationServiceMapping) parent).getDomainsMappings().toArray(new DomainsMapping[0]));
    			}
    			
    			return children.toArray();
    		}

    		else if( parent instanceof AcsService[]) {
    			AcsService[] retVal = (AcsService[]) parent;
    			for(AcsService service: retVal) {
    				service.addPropertyChangeListener(SERVICE_TYPE, this);
    				service.addPropertyChangeListener(SERVICE_INSTANCE_NAME, this);
    			}
    			return retVal;
    		}
    		
    		else if( parent instanceof AcsServiceWrapper[]) {
    			AcsServiceConversationUtils.getInstance().hydrateAcsServices(_currentConfig.getSwConfiguration());
    			List<AcsServiceWrapper> services = new ArrayList<AcsServiceWrapper>();
    			for(AcsService service : _currentConfig.getSwConfiguration().getAcsServices()) {
    				ComputerConversationUtils.getInstance().hydrateComputer(service.getComputer());
    				AcsServiceWrapper wrapper = new AcsServiceWrapper(service);
    				services.add(wrapper);
    				wrapper.addPropertyChangeListener(SERVICE_TYPE, this);
    				wrapper.addPropertyChangeListener(SERVICE_INSTANCE_NAME, this);
    			}
    			return services.toArray();
    		}

    		else if( parent instanceof Computer[] ) {
    			ComputerConversationUtils.getInstance().hydrateComputers(_currentConfig.getSwConfiguration());
    			List<Computer> computers = new ArrayList<Computer>();
    			for(NetworkDevice nd: _currentConfig.getSwConfiguration().getNetworkDevices()) {
    				nd.addPropertyChangeListener("networkName", this);
    				nd.addPropertyChangeListener("name", this);
    				if( nd instanceof Computer )
    					computers.add((Computer)nd);
    			}
    			return computers.toArray();
    		}
    		
    		else if( parent instanceof Container[] ) {
    			ContainerConversationUtils.getInstance().hydrateContainers(_currentConfig.getSwConfiguration());
    			List<Container> conts = new ArrayList<Container>();
    			for(Container cont: _currentConfig.getSwConfiguration().getContainers()) {
    				cont.addPropertyChangeListener("containerName", this);
    				cont.addPropertyChangeListener("path", this);
    				if( cont.getComputer() == null ) {
        				conts.add(cont);
    				}
    			}
    			return conts.toArray();
    		}
    		else if( parent instanceof ContainerStartupOption[] ) {
    			ContainerStartupOption[] retVal = (ContainerStartupOption[]) parent;
    			for(ContainerStartupOption cont: retVal) {
    				cont.addPropertyChangeListener("optionName", this);
    			}
    			return retVal;
    		}
    		else if( parent instanceof Component[] ) {

    			// This is a long time-consuming operation sometimes, let's run it as a Job
    			final List<Component> comps = new ArrayList<Component>();
    			ProgressMonitorDialog pd = new ProgressMonitorDialog(viewer.getControl().getShell());
    			pd.run(true, false, new IRunnableWithProgress() {
    				public void run(IProgressMonitor monitor) throws InvocationTargetException, InterruptedException {

    					monitor.beginTask("Getting undeployed components", IProgressMonitor.UNKNOWN);
    					try {
    						ComponentConversationUtils.getInstance().hydrateComponents(_currentConfig.getSwConfiguration());
    					} catch (Exception e) {
    						e.printStackTrace();
    						MessageDialog.openError(viewer.getControl().getShell(), "Error while loading data",
    		    		    "There was an unexpected error while loading the list of components from the TMCDB.");
    						return;
    					}
    					
    					for(Component comp: _currentConfig.getSwConfiguration().getComponents()) {
    						comp.addPropertyChangeListener("componentName", SwDeploymentTreeContentsProvider.this);
    						comp.addPropertyChangeListener("path", SwDeploymentTreeContentsProvider.this);
    						comp.addPropertyChangeListener("implLang", SwDeploymentTreeContentsProvider.this);
    						comp.addPropertyChangeListener("code", SwDeploymentTreeContentsProvider.this);
    						if(comp.getContainer() == null )
    							comps.add(comp);
    					}
    					monitor.done();
    			}
				});
    			return comps.toArray();
    		}
    		else if( parent instanceof NotificationServiceMapping[] ) {
    			SwConfigurationConversationUtils.getInstance().hydrateNotificationServiceMappings(_currentConfig.getSwConfiguration());
    			List<NotificationServiceMapping> mappings = new ArrayList<NotificationServiceMapping>();
    			for(NotificationServiceMapping mapping: _currentConfig.getSwConfiguration().getNotificationServiceMappings()) {
    				mapping.addPropertyChangeListener("defaultNotificationService", this);
       				mappings.add(mapping);
    			}
    			return mappings.toArray();
    		}
    		
    		else if( parent instanceof DomainsMapping[] ) {
    			for(DomainsMapping mapping: ((DomainsMapping[]) parent)) {
    				mapping.addPropertyChangeListener("name", this);
    			}
    			return ((DomainsMapping[]) parent);
    		}
    		
    		else if( parent instanceof ChannelMapping[] ) {
    			for(ChannelMapping mapping: ((ChannelMapping[]) parent)) {
    				mapping.addPropertyChangeListener("name", this);
    			}
    			return ((ChannelMapping[]) parent);
    		}
    		else if( parent instanceof Manager[] ) {
    			SwConfigurationConversationUtils.getInstance().hydrateManagers(_currentConfig.getSwConfiguration());
    			return (_currentConfig.getSwConfiguration().getManagers().toArray());
    		}
    		
    	} catch(Exception e) {
    		e.printStackTrace();
    		MessageDialog.openError(viewer.getControl().getShell(), "Error while loading data",
    		    "There was an unexpected error while loading the Software information from the TMCDB.");
    	}

    	// Shouldn't happen
    	return new Object[0];
    }

    /**
     * 
     * 
     * @see org.eclipse.jface.viewers.IStructuredContentProvider#getElements(java.lang.Object)
     */
    public Object[] getElements( Object element )
    {
        if( ! (element instanceof HwConfiguration) ) {
        	failUnsupported(element);
        	return null;
        }

        _currentConfig = (HwConfiguration)element;
        try {
    		_currentConfig = HwConfigurationConversationUtils.getInstance().findConfigurationById(_currentConfig.getId().longValue());
    	} catch(Exception e) {
    		e.printStackTrace();
    		MessageDialog.openError(viewer.getControl().getShell(), "Error while loading data",
    		"There was an unexpected error while loading the Configuration information from the TMCDB.");
    	}

        return topLevelObjects;
    }

    /**
     * @see org.eclipse.jface.viewers.ITreeContentProvider#getParent(java.lang.Object)
     */
    public Object getParent( Object element ) {

    	/* If a Container or Component doesn't have a parent
    	 * then their parent is the Configuration that contains them */
    	
    	if( element instanceof Component ) {
    		Component comp = (Component)element;
    		if( comp.getContainer() != null )
    			return comp.getContainer();
			return comp.getConfiguration();
    	}
    	
    	else if( element instanceof Container ) {
    		Container cont = (Container)element;
    		if( cont.getComputer() != null )
    			return cont.getComputer();
			return cont.getConfiguration();
    	}
    	
    	else if( element instanceof ContainerStartupOption ) {
    		ContainerStartupOption contStartupOpt = (ContainerStartupOption)element;
    		return contStartupOpt.getContainer();
    	}
    	
    	else if( element instanceof Computer )
    		return ((Computer)element).getConfiguration();
    	
    	else if( element instanceof AcsService ) {
    		AcsService service = (AcsService)element;
    		return service.getComputer();
    	}
    	
    	else if( element instanceof AcsServiceWrapper ) {
    		return new AcsServiceWrapper[0];
    	}
    	
    	else if( element instanceof NotificationServiceMapping ) {
    		NotificationServiceMapping mapping = (NotificationServiceMapping)element;
    		return mapping.getConfiguration();
    	}
    	
    	else if( element instanceof DomainsMapping ) {
    		DomainsMapping mapping = (DomainsMapping)element;
    		return mapping.getNotificationServiceMapping();
    	}
    	
    	else if( element instanceof ChannelMapping ) {
    		ChannelMapping mapping = (ChannelMapping)element;
    		return mapping.getNotificationServiceMapping();
    	}
    	
    	else if( element instanceof BACIProperty) {
    		return ((BACIProperty)element).getComponent();
    	}
    	
    	else if( element instanceof Manager) {
    		return ((Manager)element).getConfiguration();
    	}

        return null;
    }

    /**
     * @see org.eclipse.jface.viewers.ITreeContentProvider#hasChildren(java.lang.Object)
     */
    public boolean hasChildren( Object element ) 
    {
    	// Before asking the real children, we assume that everyone except BACProperty
    	// has elements (Containers have Components, Computers have Containers, etc...)
    	// After expanding the node, the actual hydration is done

    	boolean retVal = true;
    	
    	if(element instanceof BACIProperty || element instanceof ContainerStartupOption 
    			|| element instanceof AcsService
    			|| element instanceof AcsServiceWrapper
    			|| element instanceof ChannelMapping 
    			|| element instanceof DomainsMapping || element instanceof Manager)
    	{
    		retVal = false;
    	}
    	
    	return retVal;
    }

    /**
     * @see org.eclipse.jface.viewers.IContentProvider#inputChanged(org.eclipse.jface.viewers.Viewer,
     *      java.lang.Object, java.lang.Object)
     */
    public void inputChanged( Viewer theViewer, Object oldIn, Object newIn ) {
    	this.viewer = (TreeViewer)theViewer;
    }

	private void failUnsupported(Object element) {
		// Should never happen
        String msg = "Unsupported class: " + element.getClass().getName();
        IllegalArgumentException e = new IllegalArgumentException( msg );
        e.printStackTrace();
        throw e;
	}

	@Override
	public void propertyChange(PropertyChangeEvent evt) {
		Object o = evt.getSource();
		viewer.update(o, new String[]{evt.getPropertyName()});
	}

}