/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdbgui.views.providers;

import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.ViewerSorter;

import alma.obops.tmcdbgui.views.providers.typedlists.BaseElementStartupList;
import alma.obops.tmcdbgui.views.providers.typedlists.LRUTypeRole;
import alma.tmcdb.domain.AssemblyStartup;
import alma.tmcdb.domain.BaseElementStartup;
import alma.tmcdb.domain.HwConfiguration;
import alma.tmcdb.domain.StartupScenario;

public class StartupScenariosTreeSorter extends ViewerSorter 
{
	private static final String COLDCART_PREFIX = "ColdCart";
	private static final String POWERDIST_PREFIX = "PowerDist";
	private static final String WCA_PREFIX = "WCA";
	private static final String TEN = "10";

	@Override
	public int compare(Viewer viewer, Object obj1, Object obj2)
	{
		int retVal = 0;
		
		if(obj1 instanceof HwConfiguration || obj1 instanceof HwConfiguration[] || obj1 instanceof StartupScenario[])
		{
			retVal = -1;
		}
		else if(obj1 instanceof StartupScenario)
		{
			retVal = compareStartupScenarioTo((StartupScenario)obj1, obj2);
		}
		else if(obj1 instanceof BaseElementStartup)
		{
			retVal = compareBaseElementStartupTo((BaseElementStartup)obj1, obj2);
		}
		else if(obj1 instanceof AssemblyStartup)
		{
			retVal = compareAssemblyStartupTo((AssemblyStartup)obj1, obj2);
		}
		else if(obj1 instanceof BaseElementStartupList)
		{
			retVal = compareBaseElementStartupListTo((BaseElementStartupList)obj1, obj2);
		}
		else if(obj1 instanceof LRUTypeRole)
		{
			retVal = compareLRUTypeRoleTo((LRUTypeRole)obj1, obj2);
		}

		return retVal;
	}

	private int compareLRUTypeRoleTo(LRUTypeRole obj1, Object obj2) 
	{
		int retVal = 0;
		if(obj2 instanceof LRUTypeRole)
		{
			LRUTypeRole role1 = obj1;
			LRUTypeRole role2 = (LRUTypeRole)obj2;
			
			if(role1.getAssemblyRole().getName().startsWith(WCA_PREFIX) && role2.getAssemblyRole().getName().startsWith(WCA_PREFIX)) {
				retVal = hackedCompareLRUTypeRole(role1, role2);
			}
			else if(role1.getAssemblyRole().getName().startsWith(POWERDIST_PREFIX) && role2.getAssemblyRole().getName().startsWith(POWERDIST_PREFIX)) {
				retVal = hackedCompareLRUTypeRole(role1, role2);
			}
			else if(role1.getAssemblyRole().getName().startsWith(COLDCART_PREFIX) && role2.getAssemblyRole().getName().startsWith(COLDCART_PREFIX)) {
				retVal = hackedCompareLRUTypeRole(role1, role2);
			}
			else {
				retVal = role1.getAssemblyRole().getName().compareTo(role2.getAssemblyRole().getName());
			}
		} else if(obj2 instanceof BaseElementStartup) {
			retVal = 1;
		}
		return retVal;
	}

	private int hackedCompareLRUTypeRole(LRUTypeRole role1, LRUTypeRole role2) {
		int retVal;
		// special hack to sort WCA10, ColdCart10, and/or PowerDist10 at the end of the grouping
		if(role1.getAssemblyRole().getName().endsWith(TEN)) {
			retVal = 1;
		} else if(role2.getAssemblyRole().getName().endsWith(TEN)) {
			retVal = -1;
		} else {
			retVal = role1.getAssemblyRole().getName().compareTo(role2.getAssemblyRole().getName());
		}
		return retVal;
	}

	private int compareBaseElementStartupListTo(BaseElementStartupList obj1, Object obj2) 
	{
		int retVal = 0;
		if(obj2 instanceof BaseElementStartupList) {
			BaseElementStartupList list2 = (BaseElementStartupList)obj2;
			retVal = obj1.compareTo(list2);
		} 
		return retVal;
	}

	private int compareAssemblyStartupTo(AssemblyStartup obj1, Object obj2) 
	{
		int retVal = 0;
		if(obj2 instanceof AssemblyStartup)
		{
			AssemblyStartup item1 = obj1;
			AssemblyStartup item2 = (AssemblyStartup)obj2;
			
			if(item1.getAssemblyRole().getName().startsWith(WCA_PREFIX) && item2.getAssemblyRole().getName().startsWith(WCA_PREFIX)) {
				retVal = hackedCompareAssemblyStartup(item1, item2);
			}
			else if(item1.getAssemblyRole().getName().startsWith(POWERDIST_PREFIX) && item2.getAssemblyRole().getName().startsWith(POWERDIST_PREFIX)) {
				retVal = hackedCompareAssemblyStartup(item1, item2);
			}
			else if(item1.getAssemblyRole().getName().startsWith(COLDCART_PREFIX) && item2.getAssemblyRole().getName().startsWith(COLDCART_PREFIX)) {
				retVal = hackedCompareAssemblyStartup(item1, item2);
			}
			else {
				retVal = item1.getAssemblyRole().getName().compareTo(item2.getAssemblyRole().getName());
			}
		} else if(obj2 instanceof BaseElementStartup) {
			retVal = 1;
		}
		return retVal;
	}

	private int hackedCompareAssemblyStartup(AssemblyStartup item1, AssemblyStartup item2) {
		int retVal;
		// special hack to sort WCA10, ColdCart10, and/or PowerDist10 at the end of the grouping
		if(item1.getAssemblyRole().getName().endsWith(TEN)) {
			retVal = 1;
		} else if(item2.getAssemblyRole().getName().endsWith(TEN)) {
			retVal = -1;
		} else {
			retVal = item1.getAssemblyRole().getName().compareTo(item2.getAssemblyRole().getName());
		}
		return retVal;
	}

	private int compareBaseElementStartupTo(BaseElementStartup obj1, Object obj2) 
	{
		int retVal = 0;
		if(obj2 instanceof BaseElementStartup)
		{
			BaseElementStartup item1 = obj1;
			BaseElementStartup item2 = (BaseElementStartup)obj2;
			
			if(obj2 instanceof AssemblyStartup || obj2 instanceof LRUTypeRole) {
				retVal = -1;
			}
			else if(item1.getBaseElement() == null && item2.getBaseElement() == null) {
				retVal = item1.getType().name().compareTo(item2.getType().name());
			} 
			else {
				retVal = item1.getBaseElement().getName().compareTo(item2.getBaseElement().getName());
			}
		}
		
		return retVal;
	}

	private int compareStartupScenarioTo(StartupScenario obj1, Object obj2) 
	{
		int retVal = 0;
		StartupScenario startup1 = obj1;
		
		if(obj2 instanceof StartupScenario) {
			StartupScenario startup2 = (StartupScenario)obj2;
			if(startup1.getId().equals(startup2.getId())) {
				retVal = 0;
			}
			else {
				retVal = startup1.getName().compareTo(startup2.getName());
			}
		}
		return retVal;
	}
}
