/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdbgui.views.providers;

import org.eclipse.jface.viewers.CellEditor;
import org.eclipse.jface.viewers.EditingSupport;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.TextCellEditor;

import alma.obops.tmcdbgui.editors.IPointingModelTermUpdateable;
import alma.obops.tmcdbgui.widgets.support.DirtyListener;

/**
 * Editing support for the table which is used to edit pointing model terms/offsets
 * @author sharring
 */
public class PointingModelEditingSupport extends EditingSupport 
{
	private static final String ZERO = "0.0";
	private CellEditor _editor;
	private int _column;
	private DirtyListener _listener;
	private IPointingModelTermUpdateable pmUpdateable;
	
	public PointingModelEditingSupport(final TableViewer viewer, int column, DirtyListener listener, IPointingModelTermUpdateable pmUpdateable) 
	{
		super(viewer);
		this.pmUpdateable = pmUpdateable;
		this._listener = listener;
		_column = column;
		_editor = new TextCellEditor(viewer.getTable());
		_editor.getControl().addTraverseListener(new TabTraverseListener());
		_editor.getControl().addKeyListener(new TabKeyListener(_editor, viewer, _column));
		
		switch(column) {
		case 0:
			_editor.setValidator(null);
			break;
		case 1:
		case 2:
		case 3:
		case 4:
		case 5:
		case 6:
		case 7:
		case 8:
		case 9:
		case 10:
		case 11:
			_editor.setValidator(new ScientificNotationCellEditorValidator());
			break;
		}
	}

	@Override
	protected boolean canEdit(Object element) 
	{
		boolean retVal = true;
		
		if(_column == 0) {
			// disallow editing of coeff name; to perform an edit
			// the user may simply delete and re-add with a new name.
			retVal = false;
		}
		
		return retVal;
	}

	@Override
	protected CellEditor getCellEditor(Object element) {
		return _editor;
	}

	@Override
	protected Object getValue(Object element) 
	{
		PointingModelRow row = (PointingModelRow)element;
		Object retVal = null;
		
		switch(_column) {
		case 0:
			retVal = row.getCoeffName();
			break;
		case 1:
			retVal = row.getCoeffValue().toString();
			break;
		case 2:
			retVal = row.getOffset1() == null ? ZERO : row.getOffset1().toString();
			break;
		case 3:
			retVal = row.getOffset2() == null ? ZERO : row.getOffset2().toString();
			break;
		case 4:
			retVal = row.getOffset3() == null ? ZERO : row.getOffset3().toString();
			break;
		case 5:
			retVal = row.getOffset4() == null ? ZERO : row.getOffset4().toString();
			break;
		case 6:
			retVal = row.getOffset5() == null ? ZERO : row.getOffset5().toString();
			break;
		case 7:
			retVal = row.getOffset6() == null ? ZERO : row.getOffset6().toString();
			break;
		case 8:
			retVal = row.getOffset7() == null ? ZERO : row.getOffset7().toString();
			break;
		case 9:
			retVal = row.getOffset8() == null ? ZERO : row.getOffset8().toString();
			break;
		case 10:
			retVal = row.getOffset9() == null ? ZERO : row.getOffset9().toString();
			break;
		case 11:
			retVal = row.getOffset10() == null ? ZERO : row.getOffset10().toString();
			break;
		default:
			retVal = null;
			break;
		}
		return retVal;
	}

	@Override
	protected void setValue(Object element, Object value) 
	{
		PointingModelRow row = (PointingModelRow)element;
		if(value == null) {
			return;
		}
		String newValue = (String)value;
		switch(_column) {
		case 0:
			if( !(row.getCoeffName().equals(newValue)) )
			{
				pmUpdateable.updatePointingModelCoeffName(row.getCoeffName(), newValue);
				row.setCoeffName(newValue);
				_listener.setDirty(true);
				getViewer().refresh(element, true);
			}
			break;
		case 1:
			if(!(row.getCoeffValue().equals(Float.valueOf(newValue))) )
			{
				row.setCoeffValue(Float.valueOf(newValue));
				_listener.setDirty(true);
				getViewer().refresh(element, true);
			}
			break;
		case 2:
			if( null == row.getOffset1() || !(row.getOffset1().equals(Double.valueOf(newValue))))
			{
				row.setOffset1(Double.valueOf(newValue));
				_listener.setDirty(true);
				getViewer().refresh(element, true);
			}
			break;
		case 3:
			if( null == row.getOffset2() || !(row.getOffset2().equals(Double.valueOf(newValue))))
			{
				row.setOffset2(Double.valueOf(newValue));
				_listener.setDirty(true);
				getViewer().refresh(element, true);
			}
			break;
		case 4:
			if( null == row.getOffset3() || !(row.getOffset3().equals(Double.valueOf(newValue))))
			{
				row.setOffset3(Double.valueOf(newValue));
				_listener.setDirty(true);
				getViewer().refresh(element, true);
			}
			break;
		case 5:
			if( null == row.getOffset4() || !(row.getOffset4().equals(Double.valueOf(newValue))))
			{
				row.setOffset4(Double.valueOf(newValue));
				_listener.setDirty(true);
				getViewer().refresh(element, true);
			}
			break;
		case 6:
			if( null == row.getOffset5() || !(row.getOffset5().equals(Double.valueOf(newValue))))
			{
				row.setOffset5(Double.valueOf(newValue));
				_listener.setDirty(true);
				getViewer().refresh(element, true);
			}
			break;
		case 7:
			if( null == row.getOffset6() || !(row.getOffset6().equals(Double.valueOf(newValue))))
			{
				row.setOffset6(Double.valueOf(newValue));
				_listener.setDirty(true);
				getViewer().refresh(element, true);
			}
			break;
		case 8:
			if( null == row.getOffset7() || !(row.getOffset7().equals(Double.valueOf(newValue))))
			{
				row.setOffset7(Double.valueOf(newValue));
				_listener.setDirty(true);
				getViewer().refresh(element, true);
			}
			break;
		case 9:
			if( null == row.getOffset8() || !(row.getOffset8().equals(Double.valueOf(newValue))))
			{
				row.setOffset8(Double.valueOf(newValue));
				_listener.setDirty(true);
				getViewer().refresh(element, true);
			}
			break;
		case 10:
			if( null == row.getOffset9() || !(row.getOffset9().equals(Double.valueOf(newValue))))
			{
				row.setOffset9(Double.valueOf(newValue));
				_listener.setDirty(true);
				getViewer().refresh(element, true);
			}
			break;
		case 11:
			if( null == row.getOffset10() || !(row.getOffset10().equals(Double.valueOf(newValue))))
			{
				row.setOffset10(Double.valueOf(newValue));
				_listener.setDirty(true);
				getViewer().refresh(element, true);
			}
			break;
		}
	}
}
