/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
/**
 * 
 */
package alma.obops.tmcdbgui.views.providers;

import org.eclipse.jface.viewers.CellEditor;
import org.eclipse.jface.viewers.ColumnViewer;
import org.eclipse.jface.viewers.EditingSupport;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.TextCellEditor;

import alma.obops.tmcdb.alarms.ui.utils.RcpUtils;
import alma.obops.tmcdbgui.utils.conversation.AssemblyTypeConversationUtils;
import alma.tmcdb.domain.AssemblyType;

/**
 * Edition support for the DeviceLibraries table viewer. It allows to edit
 * the in-memory assembly types, modifying the production and simulation code.
 * No change is flushed to the database; instead, a separate action is available
 * in the view to do so.
 * 
 * @author rtobar, Sep 23rd, 2010
 *
 */
public class DeviceLibrariesEditingSupport extends EditingSupport {

	private CellEditor _editor;
	private int _column;

	public DeviceLibrariesEditingSupport(ColumnViewer viewer, int column) {
		super(viewer);
		_column = column;
		switch(column) {
			case 1:
			case 2:
				_editor = new TextCellEditor(((TableViewer)viewer).getTable());
		}
	}

	@Override
	protected boolean canEdit(Object element) {
		return (_column != 0);
	}

	@Override
	protected CellEditor getCellEditor(Object element) {
		return _editor;
	}

	@Override
	protected Object getValue(Object element) {
		AssemblyType at = (AssemblyType)element;
		switch(_column) {
			case 1:
				return at.getProductionCode();
			case 2:
				return at.getSimulatedCode();
		}
		return null;
	}

	@Override
	protected void setValue(Object element, Object value) {
		AssemblyType at = (AssemblyType)element;
		String newValue =(String)value;
		switch(_column) {
			case 1:
				if( at.getProductionCode().equals(newValue) )
					return;
				at.setProductionCode(newValue);
				updateAssemblyType(at);
				getViewer().refresh(element, true);
				break;
			case 2:
				if( at.getSimulatedCode().equals(newValue) )
					return;
				at.setSimulatedCode(newValue);
				updateAssemblyType(at);
				getViewer().refresh(element, true);
				break;
		}
	}

	private void updateAssemblyType(AssemblyType at) {
		try {
			AssemblyTypeConversationUtils.getInstance().updateAssemblyType(at);
		} catch(Exception e) {
			RcpUtils.errorMessage(e, getViewer().getControl().getShell(), "Cannot update Assembly Type",
					"There was an unexpected error while trying to update Assembly Type '" + at.getName() + "'");
		}
	}
}
