/*******************************************************************************
  * ALMA - Atacama Large Millimeter Array
  * Copyright (c) NRAO - National Radio Astronomy Observatory, 2012
  * (in the framework of the ALMA collaboration).
  * All rights reserved.
  *
  * This library is free software; you can redistribute it and/or
  * modify it under the terms of the GNU Lesser General Public
  * License as published by the Free Software Foundation; either
  * version 2.1 of the License, or (at your option) any later version.
  *
  * This library is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  * Lesser General Public License for more details.
  *
  * You should have received a copy of the GNU Lesser General Public
  * License along with this library; if not, write to the Free Software
  * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 
*******************************************************************************/
package alma.obops.tmcdbgui.views.providers;

import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.TimeZone;

import org.eclipse.jface.viewers.ITableLabelProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.swt.graphics.Image;

import alma.acs.util.UTCUtility;
import alma.obops.tmcdb.alarms.ui.utils.RcpUtils;
import alma.tmcdb.domain.AntennaToPad;
import alma.tmcdb.domain.HwConfiguration;

/**
 * @author sharring
 *
 */
public class AntennaPadAssignmentHistoryTableLabelProvider extends LabelProvider implements ITableLabelProvider  
{
	private HwConfiguration localConfig;
	
	public AntennaPadAssignmentHistoryTableLabelProvider(HwConfiguration config)
	{
		this.localConfig = config;
	}
	
	@Override
	public Image getColumnImage(Object element, int columnIndex) 
	{
		Image retVal = null;
		if(columnIndex == 0) {
			retVal = RcpUtils.getImage("icons/pad.png");
		}
		
		return retVal;
	}

	@Override
	public String getColumnText(Object element, int columnIndex) 
	{
		if( !(element instanceof AntennaToPad) )
			return null;

		AntennaToPad record = (AntennaToPad)element;
		switch(columnIndex) 
		{
			case 0:
				if(record.getPad().getConfiguration().getId().equals(localConfig.getId())) {
					return record.getPad().getName();	
				}
				return (record.getPad().getConfiguration().getName() + ":" + record.getPad().getName());
			case 1:
				// format the date into something meaningful to the user
				DateFormat dfm = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
				dfm.setTimeZone(TimeZone.getTimeZone("UTC"));

				String dateString = null;
				if(null != record.getStartTime()) {
					long millisecondsSinceEpoch = record.getStartTime();
					// convert to 100ns chunks (see comments in UTCUtility...)
					millisecondsSinceEpoch = UTCUtility.utcOmgToJava(millisecondsSinceEpoch);
					dateString = dfm.format(millisecondsSinceEpoch);
				}
				return dateString;
			case 2:
				// format the date into something meaningful to the user
				DateFormat dfm2 = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
				dfm2.setTimeZone(TimeZone.getTimeZone("UTC"));

				String dateString2 = null;
				if(null != record.getEndTime()) {
					long millisecondsSinceEpoch = record.getEndTime();
					// convert to 100ns chunks (see comments in UTCUtility...)
					millisecondsSinceEpoch = UTCUtility.utcOmgToJava(millisecondsSinceEpoch);
					dateString2 = dfm2.format(millisecondsSinceEpoch);
				}
				return dateString2;
			default:
				return null;
		}
	}

	public String getText(Object element) {
		return getColumnText(element, 0);
	}
}
