/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
/**
 * TmcdbObjectTransferHelper.java
 */
package alma.obops.tmcdbgui.views.dnd;

import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;

import alma.acs.tmcdb.AcsService;
import alma.acs.tmcdb.Component;
import alma.acs.tmcdb.Computer;
import alma.acs.tmcdb.Container;
import alma.tmcdb.domain.BaseElement;
import alma.tmcdb.domain.HwConfiguration;

/**
 * Helper class for all DnD operations done through the {@link TmcdbObjectTransfer} transfer type.
 * The idea behind this helper class is to aid the drop adapters to get the already present
 * instances of the TMCDB objects by looking into the selection, instead of having to
 * hydrate the objects once again in the view where they are dropped.
 * 
 * @author rtobar, Mar 24, 2010
 */
public class TmcdbObjectTransferHelper {

	/**
	 * This method helps the drop adapter to obtain an already hydrated object,
	 * if any is available across the application current selection. For example,
	 * 
	 * @param c The HwConfiguration to look for
	 * @return An already existing Computer object if exiting across the application selection,
	 *         the computer passed as argument otherwise
	 */
	public static HwConfiguration getHwConfigurationFromSelection(ISelection s, HwConfiguration c) {

		if( s instanceof IStructuredSelection ) {
			IStructuredSelection ss = (IStructuredSelection)s;
			for(Object o: ss.toArray()) {
				if( o instanceof HwConfiguration ) {
					HwConfiguration conf = (HwConfiguration)o;
					if( conf.getId() != null && conf.getId().equals(c.getId()) )
						return conf;
				}
			}
		}
		return c;
	}

	/**
	 * This method helps the drop adapter to obtain an already hydrated object,
	 * if any is available across the application current selection. For example,
	 * 
	 * @param c The Computer to look for
	 * @return An already existing Computer object if exiting across the application selection,
	 *         the computer passed as argument otherwise
	 */
	public static Container getContainerFromSelection(ISelection s, Container c) {

		if( s instanceof IStructuredSelection ) {
			IStructuredSelection ss = (IStructuredSelection)s;
			for(Object o: ss.toArray()) {
				if( o instanceof Container ) {
					Container cont = (Container)o;
					if( cont.getContainerId() != null &&
							cont.getContainerId().equals(c.getContainerId()) )
						return cont;
				}
			}
		}
		return c;
	}

	/**
	 * This method helps the drop adapter to obtain an already hydrated object,
	 * if any is available across the application current selection. For example,
	 * 
	 * @param service The AcsService to look for
	 * @return An already existing AcsService object if exiting across the application selection,
	 *         the AcsService passed as argument otherwise
	 */
	public static AcsService getAcsServiceFromSelection(ISelection selection,
			AcsService s) 
	{
		if( selection instanceof IStructuredSelection ) {
			IStructuredSelection ss = (IStructuredSelection)selection;
			for(Object o: ss.toArray()) {
				if( o instanceof AcsService ) {
					AcsService service = (AcsService)o;
					if( service.getAcsServiceId() != null &&
							service.getAcsServiceId().equals(s.getAcsServiceId()) )
						return service;
				}
			}
		}
		return s;
	}
	
    /**
     * This method helps the drop adapter to obtain an already hydrated object,
     * if any is available across the application current selection. For example,
     * 
     * 
     * @param c The Computer to look for
     * @return An already existing Computer object if exiting across the application selection,
     *         the computer passed as argument otherwise
     */
    public static Computer getComputerFromSelection(ISelection s, Computer c) {

    	if( s instanceof IStructuredSelection ) {
    		IStructuredSelection ss = (IStructuredSelection)s;
    		for(Object o: ss.toArray()) {
    			if( o instanceof Computer ) {
    				Computer comp = (Computer)o;
    				if( comp.getNetworkDeviceId() != null &&
    				    comp.getNetworkDeviceId().equals(c.getNetworkDeviceId()) )
    					return comp;
    			}
    		}
    	}
    	return c;
    }

    /**
     * This method helps the drop adapter to obtain an already hydrated object,
     * if any is available across the application current selection. For example,
     * 
     * 
     * @param c The Computer to look for
     * @return An already existing Computer object if exiting across the application selection,
     *         the computer passed as argument otherwise
     */
    public static Component getComponentFromSelection(ISelection s, Component c) {

    	if( s instanceof IStructuredSelection ) {
    		IStructuredSelection ss = (IStructuredSelection)s;
    		for(Object o: ss.toArray()) {
    			if( o instanceof Component ) {
    				Component comp = (Component)o;
    				if( comp.getComponentId() != null &&
    				    comp.getComponentId().equals(c.getComponentId()) )
    					return comp;
    			}
    		}
    	}
    	return c;
    }


    /**
     * This method helps the drop adapter to obtain an already hydrated object,
     * if any is available across the application current selection. For example,
     * 
     * 
     * @param be The BaseElement to look for
     * @return An already existing BaseElement object if exiting across the application selection,
     *         the BaseElement passed as argument otherwise
     */
    public static BaseElement getBaseElementFromSelection(ISelection s, BaseElement be) {

    	if( s instanceof IStructuredSelection ) {
    		IStructuredSelection ss = (IStructuredSelection)s;
    		for(Object o: ss.toArray()) {
    			if( o instanceof BaseElement ) {
    				BaseElement baseElement = (BaseElement)o;
    				if( baseElement.getId() != null &&
    				    baseElement.getId().equals(be.getId()) )
    					return baseElement;
    			}
    		}
    	}
    	return be;
    }
}
