/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdbgui.views.dnd;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;

import org.eclipse.swt.dnd.ByteArrayTransfer;
import org.eclipse.swt.dnd.TransferData;

import alma.acs.tmcdb.AcsService;
import alma.acs.tmcdb.AcsServiceServiceType;
import alma.acs.tmcdb.Component;
import alma.acs.tmcdb.ComponentImplLang;
import alma.acs.tmcdb.Computer;
import alma.acs.tmcdb.Container;
import alma.acs.tmcdb.ContainerImplLang;
import alma.obops.tmcdbgui.utils.conversation.HwConfigurationConversationUtils;
import alma.tmcdb.domain.BaseElement;
import alma.tmcdb.domain.BaseElementType;
import alma.tmcdb.domain.HwConfiguration;

/**
 * Transfer class used for serialization of {@link HwConfiguration}, {@link Component},
 * {@link Container} or other domain objects used across the TMCDB when performing DnD
 * or cut/copy/paste operations on the TMCDB Explorer.
 * 
 * The objects are not serialized completely, but instead only the important fields (id, name, path, etc...).
 * On the drop side this information should be used for getting an actual hydrated object,
 * whether it be using a service, or looking into the current contents of each view.
 * 
 * Currently supported objects are:
 * <ul>
 * 	<li><code>Component[]</code></li>
 *  <li><code>Container[]</code></li>
 *  <li><code>Computer[]</code></li>
 *  <li><code>HwConfiguration</code></li>
 *  <li><code>BaseElement</code></li>
 * </ul>
 * 
 * @author rtobar, Feb 23, 2010
 */
public class TmcdbObjectTransfer extends ByteArrayTransfer {

	private static final String ACSSERVICES = "ACSSERVICES";
	private static final String BASE_ELEMENT = "BASEELEMENT";
	private static final String COMPONENTS = "COMPONENTS";
	private static final String CONTAINERS = "CONTAINERS";
	private static final String COMPUTERS = "COMPUTERS";
	private static final String CONFIGURATION = "CONFIGURATION";

	private static TmcdbObjectTransfer _instance;

	private static final String TYPE_NAME = "tmcdb-objects-format";
	private static final int TYPEID = registerType(TYPE_NAME);

	private TmcdbObjectTransfer(){ }

	@Override
	public int[] getTypeIds() {
		return new int[] { TYPEID };
	}

	@Override
	public String[] getTypeNames() {
		return new String[] { TYPE_NAME };
	}

	public static TmcdbObjectTransfer getInstance() {
		if( _instance == null )
			_instance = new TmcdbObjectTransfer();
		return _instance;
	}

	public void javaToNative(Object o, TransferData transferData) {

		if( o == null || (
			!(o instanceof AcsService[]) &&
			!(o instanceof Component[]) &&
			!(o instanceof Container[]) &&
			!(o instanceof Computer[]) &&
			!(o instanceof HwConfiguration)) &&
			!(o instanceof BaseElement))
			return;
		
		if( isSupportedType(transferData) ) {
			if( o instanceof Component[] )
				super.javaToNative(toByteArray((Component[])o), transferData);
			else if( o instanceof Container[] )
				super.javaToNative(toByteArray((Container[])o), transferData);
			else if( o instanceof AcsService[] )
				super.javaToNative(toByteArray((AcsService[])o), transferData);
			else if( o instanceof Computer[] )
				super.javaToNative(toByteArray((Computer[])o), transferData);
			else if( o instanceof HwConfiguration )
				super.javaToNative(toByteArray((HwConfiguration)o), transferData);
			else if( o instanceof BaseElement )
				super.javaToNative(toByteArray((BaseElement)o), transferData);
		}
	}

	public Object nativeToJava(TransferData transferData) {
		 byte[] bytes = (byte[])super.nativeToJava(transferData);
		 return fromByteArray(bytes);
	 }

	 private Object fromByteArray(byte[] bytes) {

		 DataInputStream in = new DataInputStream(new ByteArrayInputStream(bytes));
		 try {
			 String type = in.readUTF();
			 if( type.equals(CONTAINERS) )
				 return containersFromByteArray(in);
			 else if( type.equals(ACSSERVICES) )
				 return acsServicesFromByteArray(in);
			 else if( type.equals(COMPONENTS) )
				 return componentsFromByteArray(in);
			 else if( type.equals(COMPUTERS) )
				 return computersFromByteArray(in);
			 else if( type.equals(CONFIGURATION) )
				 return configurationFromByteArray(in);
			 else if( type.equals(BASE_ELEMENT) )
				 return baseElementFromByteArray(in);
		 } catch (IOException e) {
			 return null;
		 }

		 return null;
	 }

	 /*********************** ACSSERVICES RELATED METHODS ********************************/
	 private Object acsServicesFromByteArray(DataInputStream in) {
		 try {
			 int n = in.readInt();
			 AcsService[] services = new AcsService[n];
			 for (int i = 0; i < n; i++) {
				 AcsService serv = readAcsService(in);
				 if (serv == null) {
					 return null;
				 }
				 services[i] = serv;
			 }
			 return services;
		 } catch (IOException e) {
			 e.printStackTrace();
			 return null;
		 }
	}

	private AcsService readAcsService(DataInputStream dataIn) throws IOException {
		 AcsService serv = new AcsService();
		 serv.setAcsServiceId( dataIn.readInt() );
		 if( dataIn.readBoolean() ) {
			 serv.setServiceInstanceName( dataIn.readUTF() );	
		 }
		 serv.setServiceType( AcsServiceServiceType.valueOfForEnum(dataIn.readUTF()) );
		 if( dataIn.readBoolean() ) {
			 Computer comp = new Computer();
			 comp.setNetworkDeviceId( dataIn.readInt() );
			 serv.setComputer(comp);
		 }
		 return serv;
	}
	
	 /**
	  * Writes the given acs service to the stream.
	  */
	 private void writeAcsService(AcsService service, DataOutputStream dataOut) throws IOException {
		 dataOut.writeInt( service.getAcsServiceId() );
		 dataOut.writeBoolean(service.getServiceInstanceName() != null);
		 if(service.getServiceInstanceName() != null) {
			dataOut.writeUTF( service.getServiceInstanceName());
		 }
		 dataOut.writeUTF( service.getServiceType().toString() );
		 dataOut.writeBoolean( service.getComputer() != null );
		 if( service.getComputer() != null )
			 dataOut.writeInt( service.getComputer().getNetworkDeviceId() );
	 }
	 
	private Object toByteArray(AcsService[] services) 
	 {
		 ByteArrayOutputStream byteOut = new ByteArrayOutputStream();
		 DataOutputStream out = new DataOutputStream(byteOut);

		 byte[] bytes = null;

		 try {
			 out.writeUTF( ACSSERVICES );
			 out.writeInt(services.length);
			 for (int i = 0; i < services.length; i++) {
				 writeAcsService(services[i], out);
			 }
			 out.close();
			 bytes = byteOut.toByteArray();
		 } catch (IOException e) {
			 e.printStackTrace();
			 bytes= null;
		 } catch(Throwable th) {
			 th.printStackTrace();
			 bytes = null;
		 }
		 return bytes;
	 }

	/*********************** COMPONENTS RELATED METHODS ********************************/
	 private Component[] componentsFromByteArray(DataInputStream in) {

		 try {
			 int n = in.readInt();
			 Component[] components = new Component[n];
			 for (int i = 0; i < n; i++) {
				 Component comp = readComponent(in);
				 if (comp == null) {
					 return null;
				 }
				 components[i] = comp;
			 }
			 return components;
		 } catch (IOException e) {
			 return null;
		 }
	 }

	 /**
	  * Reads and returns a single component from the given stream.
	  */
	 private Component readComponent(DataInputStream dataIn) throws IOException {
		 Component comp = new Component();
		 comp.setComponentId( dataIn.readInt() );
		 comp.setComponentName( dataIn.readUTF() );
		 comp.setPath( dataIn.readUTF() );
		 comp.setImplLang( ComponentImplLang.valueOfForEnum(dataIn.readUTF()) );
		 if( dataIn.readBoolean() ) {
			 Container cont = new Container();
			 cont.setContainerId( dataIn.readInt() );
			 comp.setContainer(cont);
		 }
		 return comp;
	 }
	 
	 private byte[] toByteArray(Component[] components) {

		 ByteArrayOutputStream byteOut = new ByteArrayOutputStream();
		 DataOutputStream out = new DataOutputStream(byteOut);

		 byte[] bytes = null;

		 try {
			 out.writeUTF( COMPONENTS );
			 out.writeInt(components.length);
			 for (int i = 0; i < components.length; i++) {
				 writeComponent(components[i], out);
			 }
			 out.close();
			 bytes = byteOut.toByteArray();
		 } catch (IOException e) {
			 bytes= null;
		 }
		 return bytes;
	 }

	 /**
	  * Writes the given component to the stream.
	  */
	 private void writeComponent(Component comp, DataOutputStream dataOut) throws IOException {
		 dataOut.writeInt( comp.getComponentId() );
		 dataOut.writeUTF( comp.getComponentName() );
		 dataOut.writeUTF( comp.getPath() );
		 dataOut.writeUTF( comp.getImplLang().toString() );
		 dataOut.writeBoolean( comp.getContainer() != null );
		 if( comp.getContainer() != null )
			 dataOut.writeInt( comp.getContainer().getContainerId() );
	 }




	 /************************ CONTAINER RELATED METHODS ********************************/
	 private Container[] containersFromByteArray(DataInputStream in) {

		 try {
			 int n = in.readInt();
			 Container[] containers = new Container[n];
			 for (int i = 0; i < n; i++) {
				 Container cont = readContainer(in);
				 if (cont == null) {
					 return null;
				 }
				 containers[i] = cont;
			 }
			 return containers;
		 } catch (IOException e) {
			 return null;
		 }
	 }

	 /**
	  * Reads and returns a single container from the given stream.
	  */
	 private Container readContainer(DataInputStream dataIn) throws IOException {
		 Container container = new Container();
		 container.setContainerId( dataIn.readInt() );
		 container.setContainerName( dataIn.readUTF() );
		 container.setImplLang( ContainerImplLang.valueOfForEnum(dataIn.readUTF()) );
		 container.setPath( dataIn.readUTF() );
		 if( dataIn.readBoolean() ) {
			 Computer comp = new Computer();
			 comp.setNetworkDeviceId( dataIn.readInt() );
			 container.setComputer(comp);
		 }
		 
		 return container;
	 }

	 private byte[] toByteArray(Container[] containers) {

		 ByteArrayOutputStream byteOut = new ByteArrayOutputStream();
		 DataOutputStream out = new DataOutputStream(byteOut);

		 byte[] bytes = null;

		 try {
			 out.writeUTF( CONTAINERS );
			 out.writeInt(containers.length);
			 for (int i = 0; i < containers.length; i++) {
				 writeContainer(containers[i], out);
			 }
			 out.close();
			 bytes = byteOut.toByteArray();
		 } catch (IOException e) {
			 bytes= null;
		 }
		 return bytes;
	 }

	 /**
	  * Writes the given container to the stream.
	  */
	 private void writeContainer(Container container, DataOutputStream dataOut) throws IOException {

		 dataOut.writeInt( container.getContainerId() != null ? container.getContainerId() : -1 );
		 dataOut.writeUTF( container.getContainerName() );
		 dataOut.writeUTF( container.getImplLang().toString() );
		 dataOut.writeUTF( container.getPath() );
		 dataOut.writeBoolean( container.getComputer() != null );
		 if( container.getComputer() != null )
			 dataOut.writeInt( container.getComputer().getNetworkDeviceId() );

	 }




	 /************************ COMPUTER RELATED METHODS ********************************/
	 private Computer[] computersFromByteArray(DataInputStream in) {

		 try {
			 int n = in.readInt();
			 Computer[] computers = new Computer[n];
			 for (int i = 0; i < n; i++) {
				 Computer comp = readComputer(in);
				 if (comp == null) {
					 return null;
				 }
				 computers[i] = comp;
			 }
			 return computers;
		 } catch (IOException e) {
			 return null;
		 }
	 }

	 /**
	  * Reads and returns a single container from the given stream.
	  */
	 private Computer readComputer(DataInputStream dataIn) throws IOException {
		 Computer computer = new Computer();
		 computer.setNetworkDeviceId( dataIn.readInt() );
		 computer.setName( dataIn.readUTF() );
		 computer.setNetworkName( dataIn.readUTF() );
		 
		 return computer;
	 }

	 private byte[] toByteArray(Computer[] computers) {

		 ByteArrayOutputStream byteOut = new ByteArrayOutputStream();
		 DataOutputStream out = new DataOutputStream(byteOut);

		 byte[] bytes = null;

		 try {
			 out.writeUTF( COMPUTERS );
			 out.writeInt(computers.length);
			 for (int i = 0; i < computers.length; i++) {
				 writeComputer(computers[i], out);
			 }
			 out.close();
			 bytes = byteOut.toByteArray();
		 } catch (IOException e) {
			 bytes= null;
		 }
		 return bytes;
	 }

	 /**
	  * Writes the given container to the stream.
	  */
	 private void writeComputer(Computer computer, DataOutputStream dataOut) throws IOException {
		 dataOut.writeInt( computer.getNetworkDeviceId() != null ? computer.getNetworkDeviceId() : -1 );
		 dataOut.writeUTF( computer.getName() );
		 dataOut.writeUTF( computer.getNetworkName() );
	 }





	 /*************************** HW CONFIGURATION RELATED METHODS **************************/
	 private Object configurationFromByteArray(DataInputStream in) {

		 try {
			 Long id = in.readLong();
//			 String name = in.readUTF();
//			 Integer swConfId = in.readInt();
//			 Configuration swconf = new Configuration();
//			 swconf.setConfigurationId( swConfId );
//			 swconf.setConfigurationName( name );
//			 HwConfiguration conf = new HwConfiguration(swconf);
			 HwConfiguration conf;
			try {
				conf = HwConfigurationConversationUtils.getInstance().findConfigurationById(id);
			} catch (Exception e) {
				conf = null;
			}
			 return conf;
		 } catch (IOException e) {
		 }
		 return null;
	 }

	 private byte[] toByteArray(HwConfiguration conf) {
		 ByteArrayOutputStream byteOut = new ByteArrayOutputStream();
		 DataOutputStream out = new DataOutputStream(byteOut);

		 byte[] bytes = null;

		 try {
			 out.writeUTF( CONFIGURATION );
			 out.writeLong( conf.getId() );
			 out.writeUTF( conf.getName() );
			 out.writeInt(conf.getSwConfiguration().getConfigurationId());
			 out.close();
			 bytes = byteOut.toByteArray();
		 } catch (IOException e) {
			 bytes= null;
		 }
		 return bytes;
	 }


	 
	 /*************************** BASE ELEMENT RELATED METHODS **************************/
	 private Object baseElementFromByteArray(DataInputStream in) {

		 try {
			 Long id = in.readLong();
			 BaseElementType type = BaseElementType.valueOf( in.readUTF() );

			 BaseElement baseElement = new BaseElement();
			 baseElement.setId( id );
			 baseElement.setType( type );
			 return baseElement;
		 } catch (IOException e) {
		 }
		 return null;
	 }

	 private byte[] toByteArray(BaseElement be) {
		 ByteArrayOutputStream byteOut = new ByteArrayOutputStream();
		 DataOutputStream out = new DataOutputStream(byteOut);

		 byte[] bytes = null;

		 try {
			 out.writeUTF( BASE_ELEMENT );
			 out.writeLong( be.getId() );
			 out.writeUTF( be.getType().toString() );
			 out.close();
			 bytes = byteOut.toByteArray();
		 } catch (IOException e) {
			 bytes= null;
		 }
		 return bytes;
	 }
}
