/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdbgui.views.dnd;

import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.TreePath;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.ViewerDropAdapter;
import org.eclipse.swt.dnd.TransferData;
import org.eclipse.ui.IViewPart;

import alma.acs.tmcdb.AcsService;
import alma.acs.tmcdb.Component;
import alma.acs.tmcdb.Computer;
import alma.acs.tmcdb.Container;
import alma.obops.tmcdb.alarms.ui.utils.RcpUtils;
import alma.obops.tmcdbgui.utils.LabelHelper;
import alma.obops.tmcdbgui.utils.conversation.AcsServiceConversationUtils;
import alma.obops.tmcdbgui.utils.conversation.ComponentConversationUtils;
import alma.obops.tmcdbgui.utils.conversation.ContainerConversationUtils;
import alma.obops.tmcdbgui.views.SoftwareDeploymentView;
import alma.tmcdb.domain.HwConfiguration;

public class SwDeploymentTreeDropAdapter extends ViewerDropAdapter {

	public SwDeploymentTreeDropAdapter(Viewer viewer) {
		super(viewer);
	}

	@Override
	public boolean performDrop(Object data) {

		Object target = getCurrentTarget();

		if( data == null )
			return false;

		IViewPart viewpart = RcpUtils.findView(SoftwareDeploymentView.ID);
		ISelection selection = viewpart.getSite().getSelectionProvider().getSelection();
		if( TmcdbObjectTransfer.getInstance().isSupportedType(getCurrentEvent().currentDataType) ) {
			// If moving Containers...
			if( data instanceof Container[] && target != null &&
			    (target instanceof Computer || target instanceof Container[]) ) {

				for(Container cont: (Container[])data) {

					Computer comp = null;
					if( target instanceof Computer )
						comp = (Computer)target;

					try {
						Container c = TmcdbObjectTransferHelper.getContainerFromSelection(selection, cont);
						c.setComputer(comp);
						ContainerConversationUtils.getInstance().saveOrUpdateContainer(c);
						refreshView(target);
					} catch(Exception e) {
						e.printStackTrace();
						MessageDialog.openError(getViewer().getControl().getShell(),
						    "Error while moving Container",
						    "There was an unexpected error while moving Container '" + LabelHelper.getFullPath(cont,false) + "'");
					}
				}

			}
			// else if moving ACS Services...
			if( data instanceof AcsService[] && target != null && target instanceof Computer ) 
			{
				Computer computer = (Computer)target;
				for(AcsService serv: (AcsService[])data) 
				{
					try {
						AcsService s = TmcdbObjectTransferHelper.getAcsServiceFromSelection(selection, serv);
						s.setComputer(computer);
						AcsServiceConversationUtils.getInstance().saveOrUpdateAcsService(s);
						refreshView(target);
					} catch(Exception e) {
						e.printStackTrace();
						MessageDialog.openError(getViewer().getControl().getShell(),
						    "Error while moving ACS Service",
						    "There was an unexpected error while moving ACS Service '" + LabelHelper.getAcsServiceLabel(serv) + "'");
					}
				}
			}
			// else, if moving Components...
			else if( data instanceof Component[] && target != null &&
					(target instanceof Container || target instanceof Component[]) ) {

				for(Component comp: (Component[])data) {

					Container cont = null;
					if( target instanceof Container )
						cont = (Container)target;

					if( cont != null && !cont.getImplLang().toString().equals( comp.getImplLang().toString() ) ) {
						MessageDialog.openWarning(getViewer().getControl().getShell(),
								"Not moving Component",
								"Component " + comp.getComponentName() + " is not going to be moved " +
								"because its implementation language (" + comp.getImplLang() +  ") " +
								"doesn't match the container's implementation language (" + cont.getImplLang() + ")" );
						continue;
					}
					try {
						Component c = TmcdbObjectTransferHelper.getComponentFromSelection(selection, comp);
						c.setContainer(cont);
						ComponentConversationUtils.getInstance().saveOrUpdateComponent(c);
						refreshView(target);
					} catch(Exception e) {
						e.printStackTrace();
						MessageDialog.openError(getViewer().getControl().getShell(),
						    "Error while moving Component",
						    "There was an unexpected error while moving Component '" + LabelHelper.getFullPath(comp,false) + "'");
					}
				}
			}
			// We are receiving a new HwConfiguration to show its SwDeployment
			else if( data instanceof HwConfiguration ) {
				SoftwareDeploymentView view = (SoftwareDeploymentView) RcpUtils.findView(SoftwareDeploymentView.ID);
				view.setInput(TmcdbObjectTransferHelper.getHwConfigurationFromSelection(selection, (HwConfiguration)data));
			}
			// Else, we shouldn't do anything
			else
				return false;
		}
		else
			return false;

		return true;
	}

	private void refreshView(Object target) {
		TreeViewer v = (TreeViewer)getViewer();
		TreePath[] paths = v.getExpandedTreePaths();
		v.refresh();
		v.setExpandedTreePaths(paths);
		v.expandToLevel(target, 1);
	}

	@Override
	public boolean validateDrop(Object target, int operation,
			TransferData transferType) {

		// Can't DnD over "Computers" or over a Component
		if( target instanceof Computer[] ||
			target instanceof Component )
			return false;

		return true;
	}
	
}
