/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
/**
 * StartupScenarioDropAdapter.java
 */
package alma.obops.tmcdbgui.views.dnd;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.viewers.ViewerDropAdapter;
import org.eclipse.swt.SWT;
import org.eclipse.swt.dnd.DND;
import org.eclipse.swt.dnd.DropTargetEvent;
import org.eclipse.swt.dnd.TransferData;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IPropertyListener;

import alma.obops.tmcdb.alarms.ui.utils.RcpUtils;
import alma.obops.tmcdbgui.rcp.TmcdbExplorer;
import alma.obops.tmcdbgui.utils.GuiUtils;
import alma.obops.tmcdbgui.utils.conversation.StartupScenarioConversationUtils;
import alma.obops.tmcdbgui.views.providers.typedlists.AntennaList;
import alma.tmcdb.domain.Antenna;
import alma.tmcdb.domain.BaseElement;
import alma.tmcdb.domain.BaseElementStartup;
import alma.tmcdb.domain.BaseElementStartupType;
import alma.tmcdb.domain.BaseElementType;
import alma.tmcdb.domain.FrontEnd;
import alma.tmcdb.domain.HwConfiguration;
import alma.tmcdb.domain.StartupScenario;
import alma.tmcdb.domain.WeatherStationController;

/**
 * This listener handles the drop part of drag and drop, dropping of a
 * BaseElement on top of a StartupScenario
 */
public class StartupScenarioDropAdapter extends ViewerDropAdapter 
{
	private static final String CANNOT_COMPLETE_DRAG_N_DROP = "Cannot complete drag-n-drop";
	private static final String CANNOT_DRAG_N_DROP_BETWEEN_CONFIGURATIONS = "Cannot drag-n-drop between configurations";
	private List<IPropertyListener> propertyListeners;

    public StartupScenarioDropAdapter( TreeViewer viewer ) {
    	super( viewer );
        this.propertyListeners = new ArrayList<IPropertyListener>();
    }

    /** Add a listener to our list of IPropertyListeners */
    public void addPropertyListener( IPropertyListener listener ) {
        this.propertyListeners.add( listener );
    }

    /**
     * Inform our listeners that the target startup scenario has changed
     */
    void firePropertyChange( Object source, int i ) {
        for( IPropertyListener listener : propertyListeners ) {
            listener.propertyChanged( source, i );
        }
    }
    /**
     * Standard "mouse over node" behavior: if it's a StartupScenario
     * tree node we're hovering over, expand it and allow dropping.
     * 
     * @see org.eclipse.swt.dnd.DropTargetAdapter#dragOver(org.eclipse.swt.dnd.DropTargetEvent)
     */
    public void dragOver( DropTargetEvent event ) 
    {
    	super.dragOver(event);
        if( event.item == null ) {
            return;
        }

        super.dragOver(event);
        Object target = getCurrentTarget();
        
        ISelection selection = TmcdbExplorer.getDefault().getWorkbench().getActiveWorkbenchWindow().getSelectionService().getSelection();
        BaseElement source = null;
        if(selection instanceof StructuredSelection)
        {
        	StructuredSelection structuredSel = (StructuredSelection) selection;
        	if(structuredSel.getFirstElement() instanceof BaseElement) 
        	{
        		source = (BaseElement) structuredSel.getFirstElement();
        	}
        }
        
        if(source != null)
        {
        	HwConfiguration sourceConfig = source.getConfiguration();
        	
        	if( target instanceof StartupScenario )  
        	{
        		StartupScenario destination = (StartupScenario)target;
        		HwConfiguration destinationConfig = destination.getConfiguration();
        		
        		if(destinationConfig.getId().equals(sourceConfig.getId()) && isSupportedSourceToDropOnStartup(source, (StartupScenario)target))
        		{
        			event.detail = DND.DROP_COPY;
        			event.feedback = DND.FEEDBACK_EXPAND | DND.FEEDBACK_SCROLL;

        			// Keep the following lines in case we want to do 
        			// something fancy when dropping
        			//            Display display = tree.getShell().getDisplay();                    
        			//            Point pt = display.map( null, tree, event.x, event.y );
        			//            Rectangle bounds = item.getBounds();
        			//            if( pt.y < bounds.y + bounds.height / 3 ) {
        			//                event.feedback |= DND.FEEDBACK_INSERT_BEFORE;
        			//            }
        			//            else if( pt.y > bounds.y + 2 * bounds.height / 3 ) {
        			//                event.feedback |= DND.FEEDBACK_INSERT_AFTER;
        			//            }
        			//            else {
        			//                event.feedback |= DND.FEEDBACK_SELECT;
        			//            }

        			event.feedback |= DND.FEEDBACK_SELECT;
        		}
        		else {
            		event.detail = DND.DROP_NONE;
            	}
        	}
        	else if(target instanceof BaseElementStartup &&
        			((BaseElementStartup)target).getType().equals(BaseElementStartupType.Antenna) &&
        			isSupportedSourceToDropOnAntenna(source)) 
        	{
        		BaseElementStartup destination = (BaseElementStartup)target;
        		HwConfiguration destinationConfig = destination.getBaseElement().getConfiguration();
        		
        		if(destinationConfig.getId().equals(sourceConfig.getId()))
        		{
        			event.detail = DND.DROP_COPY;
        			event.feedback = DND.FEEDBACK_EXPAND | DND.FEEDBACK_SCROLL;
        			event.feedback |= DND.FEEDBACK_SELECT;
        		}
        		else {
            		event.detail = DND.DROP_NONE;
            	}
        	}
        	else if(target instanceof BaseElementStartup &&
        			((BaseElementStartup)target).getType().equals(BaseElementStartupType.CentralLO) &&
        			isSupportedSourceToDropOnCentralRack(source)) 
        	{
        		BaseElementStartup destination = (BaseElementStartup)target;
        		HwConfiguration destinationConfig = destination.getBaseElement().getConfiguration();
        		
        		if(destinationConfig.getId().equals(sourceConfig.getId()))
        		{
        			event.detail = DND.DROP_COPY;
        			event.feedback = DND.FEEDBACK_EXPAND | DND.FEEDBACK_SCROLL;
        			event.feedback |= DND.FEEDBACK_SELECT;
        		}
        		else {
            		event.detail = DND.DROP_NONE;
            	}
        	}
        	else {
        		event.detail = DND.DROP_NONE;
        	}
        }
    }

	private boolean isSupportedSourceToDropOnAntenna(BaseElement source) 
	{
		boolean retVal = false;

		if(source instanceof FrontEnd) {
			retVal = true;
		} 

		return retVal;
	}
	
	private boolean isSupportedSourceToDropOnCentralRack(BaseElement source) 
	{
		boolean retVal = false;

		 if(source.getType().equals(BaseElementType.PhotonicReference)) {
			retVal = true;
		}

		return retVal;
	}

	private boolean isSupportedSourceToDropOnStartup(BaseElement source, StartupScenario startup) 
	{
		boolean retVal = false;
		
		if(source instanceof Antenna) {
			retVal = true;
			// make sure we don't already have the antenna in the startup
			for(BaseElementStartup bes: startup.getBaseElementStartups()) {
				if(bes.getBaseElement().getName().equals(source.getName()) ) {
					retVal = false;
				}
			}
		} 
		else if(source instanceof WeatherStationController)
		{
			retVal = true;
		}
		else if(source.getType().equals(BaseElementType.AOSTiming) || 
				source.getType().equals(BaseElementType.CentralLO))
		{
			retVal = true;
		}
		
		return retVal;
	}

	@Override
	public boolean performDrop(Object data) {

        Object target = getCurrentTarget();

        if( target == null || data == null ) {
            String message = "Something is null here!";
            RuntimeException e = new RuntimeException( message );
            e.printStackTrace();
            Shell outerShell = getViewer().getControl().getShell();
            RcpUtils.errorMessage( e, outerShell, "Internal error", message );
            return false;
        }

        // -----------------------------------------------------------------------------------
        // NOTE: the hierarchy of baseelements is hard-coded; e.g. the type(s) of base
        // element(s) which can reside in a given base element is hard-coded. There is no way
        // (at the present time) to determine these things programmatically. Say, for example,
        // it is possible for a front end to reside inside an antenna, but not vice versa; this
        // is business logic that is hard-coded. 
        // -----------------------------------------------------------------------------------
        // On the other hand, the type(s) of assembly(ies)
        // that can reside in a base element *can* be determined programmatically; e.g. the assembly
        // role has an assembly type associated with it, in which there is a field for baseelementtype
        // that denotes that this assembly type must reside in that base element type.
        // -----------------------------------------------------------------------------------

        ISelection selection = TmcdbExplorer.getDefault().getWorkbench().getActiveWorkbenchWindow().getSelectionService().getSelection();

        Shell shell = getViewer().getControl().getShell();
        try 
        {
        	if( TmcdbObjectTransfer.getInstance().isSupportedType(getCurrentEvent().currentDataType) ) {

        		shell.setCursor(shell.getDisplay().getSystemCursor(SWT.CURSOR_WAIT));
        		
        		BaseElement dragged = TmcdbObjectTransferHelper.getBaseElementFromSelection(selection, (BaseElement)data);
        		HwConfiguration sourceConfig = dragged.getConfiguration();
        		
        		if(target instanceof StartupScenario && dragged instanceof Antenna) 
        		{
        			dropAntennaOnStartup(sourceConfig, target, dragged);
        		}
        		else if(target instanceof BaseElementStartup && 
        				((BaseElementStartup)target).getType().equals(BaseElementStartupType.Antenna)
        				&& dragged instanceof FrontEnd) 
        		{
        			dropFrontEndOnAntenna(sourceConfig, target);
        		}
        		else if(target instanceof StartupScenario &&  
        				dragged.getType().equals(BaseElementType.CentralLO)) 
        		{
        			dropCentralLoOnStartup(sourceConfig, target, dragged);
        		}
        		else if(target instanceof StartupScenario && 
        				dragged.getType().equals(BaseElementType.AOSTiming)) 
        		{
        			dropAOSTimingOnStartup(sourceConfig, target, dragged);
        		}
        		else if(target instanceof BaseElementStartup && 
        				((BaseElementStartup)target).getType().equals(BaseElementStartupType.CentralLO) &&
        				dragged.getType().equals(BaseElementType.PhotonicReference))
        		{
        			dropPhotonicReferenceOnCentralLo(sourceConfig, target, dragged);
        		}
        		if(target instanceof StartupScenario && dragged instanceof WeatherStationController) 
        		{
        			dropWeatherStationOnStartup(sourceConfig, target, dragged);
        		}
        	}
        }
        catch( Exception e ) {
            throw new RuntimeException("Problem encountered adding element to startup scenario.", e);
        }
        finally {
        	shell.setCursor(null);
        }
		return false;
	}
	
	private void dropWeatherStationOnStartup(HwConfiguration sourceConfig, Object target, BaseElement dragged) throws Exception
	{
		StartupScenario scenario = (StartupScenario)target;
		HwConfiguration destinationConfig = scenario.getConfiguration();
		
		// Add the dragged BaseElement to the StartupScenario, save the
		// configuration back to the database, and redisplay
		int property = 0;
		BaseElementStartup added = null;
		
		if(sourceConfig.getId().equals(destinationConfig.getId())) 
		{
			added = StartupScenarioConversationUtils.getInstance().addBaseElementToStartupScenario(dragged, scenario);
			property = GuiUtils.DROP_WEATHER_STATION;
		}
		else
		{
			Shell shell = getViewer().getControl().getShell();
			MessageDialog.openWarning(shell, CANNOT_COMPLETE_DRAG_N_DROP, CANNOT_DRAG_N_DROP_BETWEEN_CONFIGURATIONS);
		}
		
		if( added != null ) {
			// inform all of our listeners
			firePropertyChange( added, property );  
		}
	}
	
	private void dropPhotonicReferenceOnCentralLo(HwConfiguration sourceConfig, Object target, BaseElement dragged) throws Exception
	{
		BaseElementStartup centralRackStartup = (BaseElementStartup) target;
		HwConfiguration destinationConfig = centralRackStartup.getBaseElement().getConfiguration();
		
		// Add the dragged BaseElement to the StartupScenario, save the
		// configuration back to the database, and redisplay
		int property = 0;
		BaseElementStartup added = null;
		
		if(sourceConfig.getId().equals(destinationConfig.getId())) 
		{
			added = StartupScenarioConversationUtils.getInstance().addPhotonicReferenceToCentralRackStartup(dragged, centralRackStartup);
			property = GuiUtils.DROP_PHOTONIC_REFERENCE;
		}
		else
		{
			Shell shell = getViewer().getControl().getShell();
			MessageDialog.openWarning(shell, CANNOT_COMPLETE_DRAG_N_DROP, CANNOT_DRAG_N_DROP_BETWEEN_CONFIGURATIONS);
		}
		
		if( added != null ) {
			// inform all of our listeners
			firePropertyChange( added, property );  
		}
	}
	
	
	private void dropAOSTimingOnStartup(HwConfiguration sourceConfig, Object target, BaseElement dragged) throws Exception
	{
		StartupScenario scenario = (StartupScenario)target;
		HwConfiguration destinationConfig = scenario.getConfiguration();
		
		// Add the dragged BaseElement to the StartupScenario, save the
		// configuration back to the database, and redisplay
		int property = 0;
		BaseElementStartup added = null;
		
		if(sourceConfig.getId().equals(destinationConfig.getId())) 
		{
			added = StartupScenarioConversationUtils.getInstance().addBaseElementToStartupScenario(dragged, scenario);
			property = GuiUtils.DROP_MASTER_CLOCK;
		}
		else
		{
			Shell shell = getViewer().getControl().getShell();
			MessageDialog.openWarning(shell, CANNOT_COMPLETE_DRAG_N_DROP, CANNOT_DRAG_N_DROP_BETWEEN_CONFIGURATIONS);
		}
		
		if( added != null ) {
			// inform all of our listeners
			firePropertyChange( added, property );  
		}
	}
	
	private void dropCentralLoOnStartup(HwConfiguration sourceConfig, Object target, BaseElement dragged) throws Exception
	{
		StartupScenario scenario = (StartupScenario)target;
		HwConfiguration destinationConfig = scenario.getConfiguration();
		
		// Add the dragged BaseElement to the StartupScenario, save the
		// configuration back to the database, and redisplay
		int property = 0;
		BaseElementStartup added = null;
		
		if(sourceConfig.getId().equals(destinationConfig.getId())) 
		{
			added = StartupScenarioConversationUtils.getInstance().addBaseElementToStartupScenario(dragged, scenario);
			property = GuiUtils.DROP_CENTRAL_RACK;
		}
		else
		{
			Shell shell = getViewer().getControl().getShell();
			MessageDialog.openWarning(shell, CANNOT_COMPLETE_DRAG_N_DROP, CANNOT_DRAG_N_DROP_BETWEEN_CONFIGURATIONS);
		}
		
		if( added != null ) {
			// inform all of our listeners
			firePropertyChange( added, property );  
		}
	}
	
	private void dropFrontEndOnAntenna(HwConfiguration sourceConfig, Object target) throws Exception
	{
		BaseElementStartup antennaStartup = (BaseElementStartup) target;
		HwConfiguration destinationConfig = antennaStartup.getBaseElement().getConfiguration();
		
		// Add the dragged BaseElement to the StartupScenario, save the
		// configuration back to the database, and redisplay
		int property = 0;
		BaseElementStartup added = null;
		
		if(sourceConfig.getId().equals(destinationConfig.getId())) 
		{
			added = StartupScenarioConversationUtils.getInstance().addFrontEndStartupToAntennaStartup(antennaStartup);
			property = GuiUtils.DROP_FRONT_END;
		}
		else
		{
			Shell shell = getViewer().getControl().getShell();
			MessageDialog.openWarning(shell, CANNOT_COMPLETE_DRAG_N_DROP, CANNOT_DRAG_N_DROP_BETWEEN_CONFIGURATIONS);
		}
		
		if( added != null ) {
			// inform all of our listeners
			firePropertyChange( added, property );  
		}
	}
	
	private void dropAntennaOnStartup(HwConfiguration sourceConfig, Object target, BaseElement dragged) throws Exception
	{ 
		StartupScenario scenario = (StartupScenario) target;
		HwConfiguration destinationConfig = scenario.getConfiguration();
		
		// Add the dragged BaseElement to the StartupScenario, save the
		// configuration back to the database, and redisplay
		int property = 0;
		BaseElementStartup added = null;
		
		if(sourceConfig.getId().equals(destinationConfig.getId())) 
		{
			added = StartupScenarioConversationUtils.getInstance().addBaseElementToStartupScenario( dragged, scenario );
			property = GuiUtils.DROP_ANTENNA;	
		}
		else
		{
			Shell shell = getViewer().getControl().getShell();
			MessageDialog.openWarning(shell, CANNOT_COMPLETE_DRAG_N_DROP, CANNOT_DRAG_N_DROP_BETWEEN_CONFIGURATIONS);
		}
		
		if( added != null ) {
			// inform all of our listeners
			firePropertyChange( added, property );  
		}
	}

	@Override
	public boolean validateDrop(Object target, int operation, TransferData transferType) 
	{
		boolean retVal = false;
		
		if(target instanceof BaseElementStartup || target instanceof AntennaList || target instanceof StartupScenario)
		{
			retVal = true;
		}
		
		return retVal;
	}

}