/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
/**
 * HwConfigurationBaseElementDropAdapter.java
 */
package alma.obops.tmcdbgui.views.dnd;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.ViewerDropAdapter;
import org.eclipse.swt.SWT;
import org.eclipse.swt.dnd.DND;
import org.eclipse.swt.dnd.DropTargetEvent;
import org.eclipse.swt.dnd.TransferData;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IPropertyListener;

import alma.obops.tmcdb.alarms.ui.utils.RcpUtils;
import alma.obops.tmcdbgui.handlers.AssignAntennaToPadAction;
import alma.obops.tmcdbgui.handlers.AssignPadToHolographyTowerAction;
import alma.obops.tmcdbgui.utils.GuiUtils;
import alma.obops.tmcdbgui.utils.conversation.BaseElementConversationUtils;
import alma.obops.tmcdbgui.views.ConfigurationsView;
import alma.tmcdb.domain.Antenna;
import alma.tmcdb.domain.BaseElement;
import alma.tmcdb.domain.BaseElementType;
import alma.tmcdb.domain.HolographyTower;
import alma.tmcdb.domain.HwConfiguration;
import alma.tmcdb.domain.Pad;


/**
 * Drop adapter for the {@link ConfigurationsView} view
 * 
 * @author amchavan, as separate class by rtobar, Mar 25, 2010
 */
public class HwConfigurationBaseElementDropAdapter extends ViewerDropAdapter 
{
	
	private List<IPropertyListener> propertyListeners;

    public HwConfigurationBaseElementDropAdapter( Viewer viewer ) {
    	super( viewer );
        this.propertyListeners = new ArrayList<IPropertyListener>();
    }

    /** Add a listener to our list of IPropertyListeners */
    public void addPropertyListener( IPropertyListener listener ) {
        this.propertyListeners.add( listener );
    }

    /**
     * Inform our listeners that the target startup scenario has changed
     */
    void firePropertyChange( Object source, int i ) {
        for( IPropertyListener listener : propertyListeners ) {
            listener.propertyChanged( source, i );
        }
    }

    @Override
    public void dragOver( DropTargetEvent event ) 
    {
    	super.dragOver(event);
    	if( getCurrentTarget() == null ) {
    		return;
    	}
    	
    	ISelection selection = getViewer().getSelection();
    	Object objSelected = ((IStructuredSelection) selection).getFirstElement();
    	
    	if(objSelected instanceof BaseElement) 
    	{
    		BaseElement beSelected = (BaseElement)objSelected;
        	
        	if(beSelected.getType().equals(BaseElementType.Antenna)) 
        	{
        		Object target = getCurrentTarget();
            	
            	if( target instanceof HwConfiguration ) {
            		event.detail = DND.DROP_COPY;
            		event.feedback = DND.FEEDBACK_EXPAND | DND.FEEDBACK_SCROLL | DND.FEEDBACK_SELECT;
            	}
            	else if( target instanceof BaseElement && ((BaseElement)target).getType().equals(BaseElementType.Pad) ) 
            	{
            		event.detail = DND.DROP_COPY;
            		event.feedback = DND.FEEDBACK_EXPAND | DND.FEEDBACK_SCROLL | DND.FEEDBACK_SELECT;
            	}
        	}
        	else if(beSelected.getType().equals(BaseElementType.Pad))
        	{
        		Object target = getCurrentTarget();
            	
            	if( target instanceof HolographyTower ) 
            	{
            		event.detail = DND.DROP_COPY;
            		event.feedback = DND.FEEDBACK_EXPAND | DND.FEEDBACK_SCROLL | DND.FEEDBACK_SELECT;
            	}
        	}
    	}
    }

	@Override
	public boolean performDrop(Object data) {

      if( !GuiUtils.isGodUser() && 
          !(data instanceof BaseElement && ((BaseElement)data).getType().equals(BaseElementType.Pad)) ) 
      {
         return false;
      }

		Object target = getCurrentTarget();

      if( target == null || data == null ) 
      {
            String message = "Drop target is null!";
            RuntimeException e = new RuntimeException( message );
            e.printStackTrace();
            Shell outerShell = getViewer().getControl().getShell();
            RcpUtils.errorMessage( e, outerShell, "Internal error", message );
            return false;
      }

		ISelection selection = getViewer().getSelection();
		if( TmcdbObjectTransfer.getInstance().isSupportedType(getCurrentEvent().currentDataType) ) 
		{
			// If dropping BaseElements
			BaseElement dragged = TmcdbObjectTransferHelper.getBaseElementFromSelection(selection, (BaseElement)data);
			if( data instanceof BaseElement && 
				((BaseElement)data).getType() == BaseElementType.Antenna &&
				target instanceof HwConfiguration)  
			{
				Shell shell = getViewer().getControl().getShell();
		        dropBaseElementOnConfiguration(target, dragged, shell);
			}
			else if( data instanceof BaseElement && 
				 ((BaseElement)data).getType().equals(BaseElementType.Antenna) &&
				 target instanceof Pad )
			{
				Shell shell = getViewer().getControl().getShell();
		        dropAntennaOnPad((Antenna)dragged, (Pad)target, shell);
			}
			else if( data instanceof BaseElement && 
					 ((BaseElement)data).getType().equals(BaseElementType.Pad) &&
					 target instanceof HolographyTower )
				{
					Shell shell = getViewer().getControl().getShell();
			        dropPadOnHolographyTower((Pad)dragged, (HolographyTower)target, shell);
				}
		}
        
		return false;
	}

	private void dropPadOnHolographyTower(Pad pad, HolographyTower tower, Shell shell) 
	{
		try 
		{
			AssignPadToHolographyTowerAction.addPadToHolographyTower(pad, tower);
		}
		catch( Exception e ) {
			throw new RuntimeException("Problem assigning Pad to HolographyTower.", e);
		}
		finally {
			shell.setCursor(null);
		}
	}

	private void dropAntennaOnPad(Antenna antenna, Pad pad, Shell shell) 
	{
		try 
		{
			// set a wait cursor
			shell.setCursor(shell.getDisplay().getSystemCursor(SWT.CURSOR_WAIT));
			
			AssignAntennaToPadAction.addAntennaToPad(antenna, pad);
		}
		catch( Exception e ) {
			throw new RuntimeException("Problem assigning Antenna to Pad.", e);
		}
		finally {
			shell.setCursor(null);
		}
	}

	private void dropBaseElementOnConfiguration(Object target,
			BaseElement dragged, Shell shell) 
	{
		try 
		{
			shell.setCursor(shell.getDisplay().getSystemCursor(SWT.CURSOR_WAIT));
			int property = 0;
			BaseElement added = null;

			HwConfiguration targetConfig = (HwConfiguration)target;
			added = BaseElementConversationUtils.getInstance().copyBaseElement(dragged, dragged.getName(), targetConfig);
			property = GuiUtils.DROP_ANTENNA;
			firePropertyChange( added, property );
		}
		catch( Exception e ) {
			throw new RuntimeException("Problem adding base element to configuration.", e);
		}
		finally {
			shell.setCursor(null);
		}
	}

	@Override
	public boolean validateDrop(Object target, int operation,
			TransferData transferType) 
	{
		return true;
	}
}
