/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
/**
 * EditorAreaDropAdapter.java
 */
package alma.obops.tmcdbgui.views.dnd;

import org.eclipse.jface.viewers.ISelection;
import org.eclipse.swt.dnd.DND;
import org.eclipse.swt.dnd.DropTargetAdapter;
import org.eclipse.swt.dnd.DropTargetEvent;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PartInitException;

import alma.acs.tmcdb.Component;
import alma.acs.tmcdb.Computer;
import alma.acs.tmcdb.Container;
import alma.obops.tmcdbgui.editors.ComponentEditor;
import alma.obops.tmcdbgui.editors.ComputerEditor;
import alma.obops.tmcdbgui.editors.ContainerEditor;
import alma.obops.tmcdbgui.editors.inputs.ComponentEditorInput;
import alma.obops.tmcdbgui.editors.inputs.ComputerEditorInput;
import alma.obops.tmcdbgui.editors.inputs.ContainerEditorInput;

/**
 * Drop adapter for the editor area. Its mission is to support all the object types
 * that have an editor associated to it. Currently, this list includes Computers,
 * Container and Components. The datatype supported is an array of only these objects,
 * as supported by the transfer type.
 * 
 * @author rtobar, Feb 26, 2010
 * @see TmcdbObjectTransfer
 */
public class EditorAreaDropAdapter extends DropTargetAdapter {

    private IWorkbenchWindow window;

    /**
     * Constructs a new EditorAreaDropAdapter.
     * @param window the workbench window
     */
    public EditorAreaDropAdapter(IWorkbenchWindow window) {
        this.window = window;
    }

    public void dragEnter(DropTargetEvent event) {
    	event.detail = DND.DROP_COPY;
    }

    public void dragOperationChanged(DropTargetEvent event) {
    	event.detail = DND.DROP_COPY;
    }

    public void drop(final DropTargetEvent event) {

        Display d = window.getShell().getDisplay();
        final IWorkbenchPage page = window.getActivePage();
        final ISelection selection = window.getSelectionService().getSelection();

        if (page != null)
        	d.asyncExec(new Runnable() {
        		public void run() {
        			if (TmcdbObjectTransfer.getInstance().isSupportedType(event.currentDataType)) {
        				if( event.data instanceof Container[] ) {
        					Container[] containers = (Container[]) event.data;
        					for (int i = 0; i < containers.length; i++) {
        						Container c = TmcdbObjectTransferHelper.getContainerFromSelection(selection, containers[i]);
        						ContainerEditorInput input = new ContainerEditorInput(c);
        						try {
        							page.openEditor(input, ContainerEditor.ID);
        						} catch (PartInitException e) {
        							e.printStackTrace();
        						}
        					}
        				}
        				else if( event.data instanceof Component[] ) {
        					Component[] components = (Component[]) event.data;
        					for (int i = 0; i < components.length; i++) {
        						Component c = TmcdbObjectTransferHelper.getComponentFromSelection(selection, components[i]);
        						ComponentEditorInput input = new ComponentEditorInput(c);
        						try {
        							page.openEditor(input, ComponentEditor.ID);
        						} catch (PartInitException e) {
        							e.printStackTrace();
        						}
        					}
        				}
        				else if( event.data instanceof Computer[] ) {
        					Computer[] computers = (Computer[]) event.data;
        					for (int i = 0; i < computers.length; i++) {
        						Computer c = TmcdbObjectTransferHelper.getComputerFromSelection(selection, computers[i]);
        						ComputerEditorInput input = new ComputerEditorInput(c);
        						try {
        							page.openEditor(input, ComputerEditor.ID);
        						} catch (PartInitException e) {
        							e.printStackTrace();
        						}
        					}
        				}
        			}
        		}
        	});
    }


}