/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
/**
 * RawDataView.java
 *
 * Copyright European Southern Observatory 2008
 */

package alma.obops.tmcdbgui.views;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Set;

import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.IMenuListener;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.viewers.DoubleClickEvent;
import org.eclipse.jface.viewers.IDoubleClickListener;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.TreePath;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.viewers.TreeViewerColumn;
import org.eclipse.swt.SWT;
import org.eclipse.swt.dnd.DND;
import org.eclipse.swt.dnd.Transfer;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.ui.IPropertyListener;
import org.eclipse.ui.ISelectionListener;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.IWorkbenchPartSite;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.part.ViewPart;

import alma.obops.tmcdb.alarms.ui.utils.RcpUtils;
import alma.obops.tmcdbgui.domain.IModelChangeListener;
import alma.obops.tmcdbgui.handlers.AddBaseElementsStartupAction;
import alma.obops.tmcdbgui.handlers.CloneStartupScenarioAction;
import alma.obops.tmcdbgui.handlers.DeleteStartupScenarioAction;
import alma.obops.tmcdbgui.handlers.EditStartupScenarioAction;
import alma.obops.tmcdbgui.handlers.NewStartupScenarioAction;
import alma.obops.tmcdbgui.handlers.RemoveBaseElementStartupAction;
import alma.obops.tmcdbgui.utils.GuiUtils;
import alma.obops.tmcdbgui.views.dnd.StartupScenarioDropAdapter;
import alma.obops.tmcdbgui.views.dnd.TmcdbObjectTransfer;
import alma.obops.tmcdbgui.views.providers.StartupScenarioTreeContentsProvider;
import alma.obops.tmcdbgui.views.providers.StartupScenarioTreeLabelProvider;
import alma.obops.tmcdbgui.views.providers.StartupScenariosTreeSorter;
import alma.obops.tmcdbgui.views.providers.helpers.startup.BaseElementStartupHelper;
import alma.obops.tmcdbgui.views.providers.helpers.startup.CentralRackStartupList;
import alma.obops.tmcdbgui.views.providers.helpers.startup.MasterClockStartupList;
import alma.obops.tmcdbgui.views.providers.helpers.startup.StartupHelperFactory;
import alma.obops.tmcdbgui.views.providers.helpers.startup.WeatherStationStartupList;
import alma.obops.tmcdbgui.views.providers.typedlists.AntennaStartupList;
import alma.obops.tmcdbgui.views.providers.typedlists.LRUTypeRole;
import alma.obops.tmcdbgui.views.support.StartupScenarioEditingSupport;
import alma.tmcdb.domain.Assembly;
import alma.tmcdb.domain.AssemblyStartup;
import alma.tmcdb.domain.BaseElement;
import alma.tmcdb.domain.BaseElementStartup;
import alma.tmcdb.domain.HwConfiguration;
import alma.tmcdb.domain.StartupScenario;

/**
 * Shows a single Configuration and all its StartupScenarios
 * 
 * @author amchavan, Sep 3, 2008
 * 
 */



public class StartupScenariosView extends ViewPart 
   implements ISelectionListener, IPropertyListener, IModelChangeListener
{
    public static final int DEFAULT_EXPAND_LEVEL = 2;
    public static final String ID = "startup-scenarios.view";
	private static final String SIMULATED = "Simulated";
	private static final String NOT_SIMULATED = "Not Simulated";
	private static final String STARTED = "Started";
	private static final String START_ALL = "Startup all devices";
	private static final String STOP_ALL = "Don't startup any devices";
    
    private TreeViewer startupsTree;
    private HwConfiguration configuration;
    
    // our actions
    private NewStartupScenarioAction addStartupScenarioAction;
    private CloneStartupScenarioAction cloneStartupScenarioAction;
    private EditStartupScenarioAction editStartupScenarioAction;
    private RemoveBaseElementStartupAction removeBaseElementStartupAction;
    private DeleteStartupScenarioAction deleteStartupScenarioAction;
    private AddBaseElementsStartupAction addBaseElementStartupAction;
    
    /**
     * @see org.eclipse.ui.part.WorkbenchPart#createPartControl(org.eclipse.swt.widgets.Composite)
     */
    @Override
    public void createPartControl( Composite parent ) {

        int style = SWT.BORDER | SWT.FULL_SELECTION | SWT.MULTI ;

        startupsTree = new TreeViewer( parent, style );
        startupsTree.setContentProvider( new StartupScenarioTreeContentsProvider() );
        startupsTree.getTree().setLinesVisible( true );
        startupsTree.getTree().setHeaderVisible( true );
        startupsTree.setSorter(new StartupScenariosTreeSorter());
        this.startupsTree.getTree().setEnabled(GuiUtils.isGodUser());
        
        // First column -- name and icon for all tree nodes
        TreeViewerColumn col0 = new TreeViewerColumn( startupsTree, SWT.NONE );
        col0.getColumn().setWidth( 250 );
        col0.getColumn().setMoveable( false );
        col0.getColumn().setText( "Name" );
        col0.setLabelProvider( 
            new StartupScenarioTreeLabelProvider( 0 ));

        // Second column -- a checkbox
        TreeViewerColumn col2 = new TreeViewerColumn( startupsTree, SWT.NONE );
        col2.getColumn().setWidth( 55 );
        col2.getColumn().setMoveable( false );
        col2.getColumn().setText( STARTED );
        col2.setLabelProvider( new StartupScenarioTreeLabelProvider( 1 ));
        col2.setEditingSupport( new StartupScenarioEditingSupport( startupsTree, StartupScenarioEditingSupport.FieldToEdit.IS_STARTED ));
        
        // Third column -- a checkbox
        TreeViewerColumn col3 = new TreeViewerColumn( startupsTree, SWT.NONE );
        col3.getColumn().setWidth( 55 );
        col3.getColumn().setMoveable( false );
        col3.getColumn().setText( SIMULATED );
        col3.setLabelProvider( new StartupScenarioTreeLabelProvider( 2 ));
        col3.setEditingSupport( new StartupScenarioEditingSupport( startupsTree, StartupScenarioEditingSupport.FieldToEdit.IS_SIMULATED));
               
        // Creation actions and context menu
        makeActions();
        makeContextMenu();
        
        // Allow other views to subscribe to the selection changed events 
        // originating on our viewer
        IWorkbenchPartSite site = getSite();
        site.setSelectionProvider( startupsTree );
        
        // subscribe to Configuration selection in the Configurations view
        site.getWorkbenchWindow().getSelectionService()
            .addSelectionListener( ConfigurationsView.ID, this );

        // Hook-up to the drag and drop infrastructure
        Transfer []types = { TmcdbObjectTransfer.getInstance() };
        StartupScenarioDropAdapter dropAdapter = new StartupScenarioDropAdapter(startupsTree);
        dropAdapter.addPropertyListener( this );

        startupsTree.addDropSupport(DND.DROP_COPY | DND.DROP_MOVE, types, dropAdapter);
    }

    /**
     * @return The Tree which represents out startup scenario
     */
    public TreeViewer getStartupScenariosTree() {
        return startupsTree;
    }
    
    /**
     * This method gets invoked when some editor signals they are done with
     * editing -- we refresh our tree.
     * 
     * @see org.eclipse.ui.IPropertyListener#propertyChanged(java.lang.Object, int)
     */
    @Override
    public void propertyChanged( Object source, int propId ) {
        startupsTree.refresh();
        
        if( propId == GuiUtils.DROP_ANTENNA && source instanceof BaseElementStartup) {
            // Need to expose the added antenna
            BaseElementStartup added = (BaseElementStartup) source;
            expandToAntennas( added.getStartup() );
        } else if( propId == GuiUtils.DROP_FRONT_END  && source instanceof BaseElementStartup) {
        	// Need to expose the added frontend
            BaseElementStartup added = (BaseElementStartup) source;
            // TODO: expandToFrontEnds (?)
            StartupScenario startup = added.getStartup() != null ? added.getStartup() : added.getParent().getStartup();
            expandToAntennas( startup );
        } else if( propId == GuiUtils.DROP_CENTRAL_RACK && source instanceof BaseElementStartup) {
        	// Need to expose the added centralrack
            BaseElementStartup added = (BaseElementStartup) source;
            // TODO: expandToCentralRacks (?)
            expandToCentralRacks( added.getStartup() );
        } else if( propId == GuiUtils.DROP_MASTER_CLOCK && source instanceof BaseElementStartup) {
        	// Need to expose the added masterclock
            BaseElementStartup added = (BaseElementStartup) source;
            // TODO: expandToMasterClocks (?)
            expandToMasterClocks( added.getStartup() );
        }
        else if( propId == GuiUtils.DROP_WEATHER_STATION && source instanceof BaseElementStartup) {
        	// Need to expose the added weather station
            BaseElementStartup added = (BaseElementStartup) source;
            // TODO: expandToWeatherStations (?)
            expandToWeatherStations( added.getStartup() );
        }
        
    }

    /**
     * Here we react to selection changes in the Configurations view. 
     * 
     * If the user selected a Configuration, we set that as the
     * root of this tree. If they selected a child node, we retrieve the root
     * Configuration and set that to be the root.
     * 
     * @see org.eclipse.ui.ISelectionListener#selectionChanged(org.eclipse.ui.IWorkbenchPart,org.eclipse.jface.viewers.ISelection)
     */
    @Override
    public void selectionChanged( IWorkbenchPart part, ISelection selection ) {

        if( selection instanceof IStructuredSelection ) {
            IStructuredSelection sselection = (IStructuredSelection) selection;
            Object selected = sselection.getFirstElement();

            // why can it be null here?
            if( selected == null ) {
                return;
            }
            
            HwConfiguration root = null;
            if( selected instanceof HwConfiguration ) {
                root = (HwConfiguration) selected;
            }
            else if( selected instanceof BaseElement ) {
                root = ((BaseElement) selected).getConfiguration();
            }
            else if( selected instanceof Assembly ) {
            	root = ((Assembly) selected).getConfiguration();
            }
            // Add more child types here, if needed:
            // else if( selected instanceof <some-class>) {
            //     root[0] = ...
            // }
            else {
                // simply ignore all other selection types
                return;
            }
            
            if( root != this.configuration ) {
            	setInput( root );
            }
        }
    }
    
    /**
     * @see org.eclipse.ui.part.WorkbenchPart#setFocus()
     */
    @Override
    public void setFocus() {
        // no-op
    }

    /**
     * Display the input configuration in the tree.
     */
    public void setInput( HwConfiguration conf ) 
    {
    	try 
    	{
    		startupsTree.getTree().setRedraw(false);

    		// store information which can be used to restore 
    		// the current selection, if any
    		Long id = computeActiveStartupScenarioId();
    		ISelection selection = startupsTree.getSelection();
    		Object[] expandedElements = startupsTree.getExpandedElements();

    		this.configuration = conf;
    		addStartupScenarioAction.setConfiguration(conf);
    		StartupHelperFactory.getInstance().clearCaches();
    		StartupHelperFactory.getInstance().setConfiguration(conf);
    		if(conf == null) {
    			startupsTree.setInput(null);
    		} else {
    			startupsTree.setInput( new HwConfiguration[] { conf });
    		}

    		// restore the current selection, if any
    		restorePreviousSelectionInStartupScenario(id, selection, expandedElements);
    	}
    	finally {
    		startupsTree.getTree().setRedraw(true);
    	}
    }

	private void restorePreviousSelectionInStartupScenario(Long id, ISelection sel, Object[] expandedElements) 
	{
		if(null != id) 
    	{
    		StartupScenario selectedScenario = findStartupScenarioInTreeViewerById(id);
    		
    		if(selectedScenario != null && sel instanceof IStructuredSelection) 
    		{
    			IStructuredSelection selection = (IStructuredSelection)sel;
    			
    			if( selection.size() == 1 &&
    					(selection.getFirstElement() instanceof StartupScenario) )
    			{
    				resetSelectionToStartupScenario(selectedScenario);
    			} 
    			
    			else if(selection.size() == 1 &&
    					(selection.getFirstElement() instanceof BaseElementStartup)) 
    			{
					resetSelectionToBaseElementStartup(id, selection);
    			}
    			
    			else if(selection.size() == 1 &&
    					(selection.getFirstElement() instanceof AssemblyStartup))
    			{
    				resetSelectionToAssemblyStartup(id, selection);
    			}
    			
    			else if(selection.size() == 1 &&
    					(selection.getFirstElement() instanceof AntennaStartupList))
    			{
    				resetSelectionToAntennaStartupList(id);
    			}
    			
    			else if(selection.size() == 1 &&
    					(selection.getFirstElement() instanceof LRUTypeRole))
    			{
    				resetSelectionToLRUTypeRole(id);
    			}
    		}
    	} 	
		for (Object element : expandedElements) {
			startupsTree.setExpandedState(element, true);
		}
	}

	private void resetSelectionToLRUTypeRole(Long id) 
	{
		Set<StartupScenario> startups = ((HwConfiguration[])startupsTree.getInput())[0].getStartupScenarios();
		
		// NOTE: this is a linear search within the treeviewer's input; 
		// if performance is an issue we can optimize this later
		for(StartupScenario startupScenario : startups)
		{
			if(startupScenario.getId().equals(id))
			{
				StartupScenarioTreeContentsProvider provider = (StartupScenarioTreeContentsProvider)(startupsTree.getContentProvider());
				Object[] children = provider.getChildren(startupScenario);
				for(Object obj : children)
				{
					if(obj instanceof LRUTypeRole) 
					{
						StructuredSelection newSelection = new StructuredSelection(obj);
						startupsTree.setSelection(newSelection, true);
						startupsTree.reveal(newSelection);
						return;
					}
				}
			}
		}
	}

	private void resetSelectionToAntennaStartupList(Long id) 
	{
		Set<StartupScenario> startups = ((HwConfiguration[])startupsTree.getInput())[0].getStartupScenarios();
		
		// NOTE: this is a linear search within the treeviewer's input; 
		// if performance is an issue we can optimize this later
		for(StartupScenario startupScenario : startups)
		{
			if(startupScenario.getId().equals(id))
			{
				StartupScenarioTreeContentsProvider provider = (StartupScenarioTreeContentsProvider)(startupsTree.getContentProvider());
				Object[] children = provider.getChildren(startupScenario);
				for(Object obj : children)
				{
					if(obj instanceof AntennaStartupList) 
					{
						AntennaStartupList list = (AntennaStartupList) obj;
						StructuredSelection newSelection = new StructuredSelection((Object)list);
						startupsTree.setSelection(newSelection, true);
						startupsTree.reveal(newSelection);
						return;
					}
				}
			}
		}
	}

	private void resetSelectionToAssemblyStartup(Long id, IStructuredSelection selection) 
	{
		AssemblyStartup previousAssemblyStartupSelected = (AssemblyStartup)(selection.getFirstElement());
		
		Set<StartupScenario> startups = ((HwConfiguration[])startupsTree.getInput())[0].getStartupScenarios();
		
		// NOTE: this is a linear search within the treeviewer's input; 
		// if performance is an issue we can optimize this later
		for(StartupScenario startupScenario : startups)
		{
			if(startupScenario.getId().equals(id))
			{
				for(AssemblyStartup ass: startupScenario.getAssemblyStartups())
				{
					if(ass.getId().equals(previousAssemblyStartupSelected.getId())) 
					{
						StructuredSelection newSelection = new StructuredSelection(ass);
						startupsTree.setSelection(newSelection, true);
						startupsTree.reveal(newSelection);
						return;
					}
				}
			}
		}
	}

	private void resetSelectionToBaseElementStartup(Long id, IStructuredSelection selection) 
	{
		BaseElementStartup previousBaseElementStartupSelected = (BaseElementStartup)(selection.getFirstElement());
		
		Set<StartupScenario> startups = ((HwConfiguration[])startupsTree.getInput())[0].getStartupScenarios();
		
		// NOTE: this is a linear search within the treeviewer's input; 
		// if performance is an issue we can optimize this later
		for(StartupScenario startupScenario : startups)
		{
			if(startupScenario.getId().equals(id))
			{
				for(BaseElementStartup bes: startupScenario.getBaseElementStartups())
				{
					if(bes.getId().equals(previousBaseElementStartupSelected.getId())) 
					{
						StructuredSelection newSelection = new StructuredSelection(bes);
						startupsTree.setSelection(newSelection, true);
						startupsTree.reveal(newSelection);
						return;
					}
				}
			}
		}
	}

	private void resetSelectionToStartupScenario(StartupScenario selectedScenario) 
	{
		StructuredSelection newSelection = new StructuredSelection(selectedScenario);    		
		startupsTree.setSelection(newSelection, true);
		startupsTree.reveal(newSelection);		
	}

	private StartupScenario findStartupScenarioInTreeViewerById(Long id) 
	{
		StartupScenario selectedStartupScenario = null;
		Set<StartupScenario> startups = ((HwConfiguration[])startupsTree.getInput())[0].getStartupScenarios();
		
		for(StartupScenario startup : startups) 
		{
			if(startup.getId().equals(id)) 
			{
				selectedStartupScenario = startup;
				break;
			}
		}
		return selectedStartupScenario;
	}

	private Long computeActiveStartupScenarioId() 
	{
		Long retVal = null;
		
		if( this.startupsTree.getSelection() instanceof IStructuredSelection )
		{
			IStructuredSelection selection = (IStructuredSelection) this.startupsTree.getSelection();
			if( selection.size() == 1 &&
					   (selection.getFirstElement() instanceof StartupScenario) )
			{
				retVal = ((StartupScenario)selection.getFirstElement()).getId();
			} 
			else if(selection.size() == 1 &&
					   (selection.getFirstElement() instanceof BaseElementStartup)) 
			{
				BaseElementStartup bes = ((BaseElementStartup) selection.getFirstElement());
				StartupScenario startupScene = bes.getStartup() == null ? bes.getParent().getStartup() : bes.getStartup(); 
				retVal = startupScene.getId();
			}
			else if(selection.size() == 1 &&
					   (selection.getFirstElement() instanceof AssemblyStartup))
			{
				retVal = (((AssemblyStartup)selection.getFirstElement()).getBaseElementStartup().getStartup()).getId();
			}
			else if(selection.size() == 1 &&
					   (selection.getFirstElement() instanceof AntennaStartupList))
			{
				retVal = ((AntennaStartupList)selection.getFirstElement()).getStartup().getId();
			}
			else if(selection.size() == 1 &&
					   (selection.getFirstElement() instanceof LRUTypeRole))
			{
				retVal = ((LRUTypeRole)selection.getFirstElement()).getStartup().getId();
			}
		}
		
		return retVal;
	}

	@Override
	public void internalModelChange() {
		TreePath[] paths = startupsTree.getExpandedTreePaths();
		startupsTree.refresh();
		startupsTree.setExpandedTreePaths(paths);
	}
	
	/**
	 * Removes an item from the tree, refreshes, clears caches, etc.
	 * @param item the item to be removed (e.g. a baseelementstartup object that is being 
	 * removed from the startupscenario).
	 */
	public void removeObjectFromTree(Object item) 
	{
		this.startupsTree.remove(item);
		StartupHelperFactory.getInstance().clearCaches();
		this.startupsTree.refresh();
	}

	@Override
	public void externalModelChange() 
	{
		try {
			this.getSite().getShell().setCursor(this.getSite().getShell().getDisplay().getSystemCursor(SWT.CURSOR_WAIT));	
			StartupHelperFactory.getInstance().clearCaches();
			internalModelChange();
		} catch (Exception e) {
			throw new RuntimeException("Problem refreshing from external model change", e);
		} finally {
			this.getSite().getShell().setCursor(null);	
		}
	}
	
	private void makeActions() 
	{
		IWorkbenchWindow win = getSite().getWorkbenchWindow();
		
		// actions
		addStartupScenarioAction = new NewStartupScenarioAction( win );
		addStartupScenarioAction.setConfiguration(configuration);
        addStartupScenarioAction.addModelChangeListener(this);
    	
    	cloneStartupScenarioAction = new CloneStartupScenarioAction( win );
    	cloneStartupScenarioAction.addModelChangeListener(this);

    	editStartupScenarioAction = new EditStartupScenarioAction( win, this );
    	
    	addBaseElementStartupAction = new AddBaseElementsStartupAction( win, this );
    	
    	removeBaseElementStartupAction = new RemoveBaseElementStartupAction( win, this );
    	
    	deleteStartupScenarioAction = new DeleteStartupScenarioAction( win );

    	startupsTree.addDoubleClickListener(new IDoubleClickListener() {
			public void doubleClick(DoubleClickEvent event) {
				if( event.getSelection() instanceof IStructuredSelection ) {
					IStructuredSelection selection = (IStructuredSelection)event.getSelection();
					if( selection.getFirstElement() instanceof StartupScenario ) {
						editStartupScenarioAction.selectionChanged(StartupScenariosView.this, selection);
						editStartupScenarioAction.run();
					}
				}
			}
		});
	}
	
	private void makeContextMenu() {
		final MenuManager mgr = new MenuManager();
        mgr.setRemoveAllWhenShown(true);
        mgr.addMenuListener( new IMenuListener() {
			public void menuAboutToShow(IMenuManager manager) {
				fillContextMenu(manager);
			}
		});
        Control ctrl = startupsTree.getControl();
        ctrl.setMenu( mgr.createContextMenu( ctrl ));
	}

	private void fillContextMenu(IMenuManager manager) 
	{	
		ISelection selection = startupsTree.getSelection();
        if( selection.isEmpty() ) {
            return;
        }

        if( selection instanceof IStructuredSelection ) 
        {
            IStructuredSelection sselection = (IStructuredSelection) selection;
            Object selNode = sselection.getFirstElement();
            if( ! (selNode instanceof HwConfiguration) && 
            	! (selNode instanceof StartupScenario) && 
            	!(selNode instanceof BaseElementStartup) &&
            	!(selNode instanceof LRUTypeRole))
            {
                return;
            }
            
            if( selNode instanceof HwConfiguration && sselection.size() == 1 ) 
            {
                manager.add( addStartupScenarioAction );
                addStartupScenarioAction.setConfiguration((HwConfiguration)selNode);
                manager.add( new ClearCacheAction());
            } 
            else if( selNode instanceof StartupScenario && sselection.size() == 1) 
            {
            	manager.add( addBaseElementStartupAction );
            	addBaseElementStartupAction.setDestinationStartupScenario((StartupScenario) selNode);
                manager.add( cloneStartupScenarioAction );
                manager.add( deleteStartupScenarioAction );
                cloneStartupScenarioAction.setScenario((StartupScenario)selNode);
                manager.add( new ClearCacheAction());
            }
            else if( selNode instanceof BaseElementStartup && GuiUtils.onlyItemsOfClassSelected(sselection, BaseElementStartup.class)) 
            {
            	manager.add( removeBaseElementStartupAction );
            	Object[] selectedObjects = sselection.toArray();
            	BaseElementStartup[] baseElementStartupsSelected = new BaseElementStartup[selectedObjects.length];
            	int i = 0;
            	for(Object obj: selectedObjects) {
            		baseElementStartupsSelected[i++] = (BaseElementStartup)obj;
            	}
            	removeBaseElementStartupAction.setBaseElementStartups(baseElementStartupsSelected);
            	
            	manager.add(new SetBaseElementStartedAction(baseElementStartupsSelected, true));
            	manager.add(new SetBaseElementStartedAction(baseElementStartupsSelected, false));
            	
            	manager.add(new SetBaseElementSimulatedAction(baseElementStartupsSelected, true));
            	manager.add(new SetBaseElementSimulatedAction(baseElementStartupsSelected, false));
            }
            else if( selNode instanceof LRUTypeRole ) 
            {
            	List<LRUTypeRole> lruTypeRoles = new ArrayList<LRUTypeRole>();
            	for( Object obj: sselection.toList() ) {
            		if(obj instanceof LRUTypeRole) {
            			lruTypeRoles.add((LRUTypeRole)obj);
            		}
            	}
            	manager.add(new SetLruTypeRoleStartedAction(lruTypeRoles, true));
            	manager.add(new SetLruTypeRoleStartedAction(lruTypeRoles, false));
            	manager.add(new SetLruTypeRoleSimulatedAction(lruTypeRoles, true));
            	manager.add(new SetLruTypeRoleSimulatedAction(lruTypeRoles, false));
            }
        }
	}

	private class ClearCacheAction extends Action
	{
		public ClearCacheAction() 
		{
			this.setText("Refresh");
			this.setImageDescriptor(RcpUtils.getImageDescriptor( "icons/arrow_refresh.png" ));
		}
		
		@Override
		public final void run() 
		{
			try {
				getSite().getShell().setCursor(getSite().getShell().getDisplay().getSystemCursor(SWT.CURSOR_WAIT));	
				StartupHelperFactory.getInstance().clearCaches();
				refresh();
			}
			finally {
				getSite().getShell().setCursor(null);	
			}
		}
	}
	
	private class SetLruTypeRoleStartedAction extends Action
	{
		@SuppressWarnings("hiding")
		private static final String ID = "start_assemblies.action";
		private List<LRUTypeRole> lruTypeRoles;
		private boolean value;
		
		public SetLruTypeRoleStartedAction(List<LRUTypeRole> lruTypeRoles, boolean val)
		{
			this.lruTypeRoles = lruTypeRoles;
			setId(ID);
			if(val) {
				setText( START_ALL );	
			} 
			else
			{
				setText( STOP_ALL );	
			}
			
			value = val;
		}
		
		@Override
		public void run() 
		{
			if(lruTypeRoles.size() > 0)
			{
				try 
				{
					getSite().getShell().setCursor(getSite().getShell().getDisplay().getSystemCursor(SWT.CURSOR_WAIT));
					int count = 0;
					for(LRUTypeRole lruTypeRole: lruTypeRoles)
					{
						count++;
						boolean commit = false;
						if(count == lruTypeRoles.size()) {
							commit = true;
						}
						lruTypeRole.setStarted(value, commit);
					}
					startupsTree.refresh();
				} catch (Exception e) {
					RcpUtils.errorMessage(e, getSite().getShell(), "Error trying to modify startup", e.getMessage());
				} finally {
					getSite().getShell().setCursor(null);
				}
			}
		}
	}
	
	private class SetLruTypeRoleSimulatedAction extends Action
	{
		@SuppressWarnings("hiding")
		private static final String ID = "start_assemblies.action";
		private List<LRUTypeRole> lruTypeRoles;
		private boolean value;
		
		public SetLruTypeRoleSimulatedAction(List<LRUTypeRole> lruTypeRoles, boolean val)
		{
			this.lruTypeRoles = lruTypeRoles;
			setId(ID);
			if(val) {
				setText( SIMULATED );	
			} 
			else
			{
				setText( NOT_SIMULATED );	
			}
			
			value = val;
		}
		
		@Override
		public void run() 
		{
			if(lruTypeRoles.size() > 0)
			{
				try 
				{
					getSite().getShell().setCursor(getSite().getShell().getDisplay().getSystemCursor(SWT.CURSOR_WAIT));
					for(LRUTypeRole lruTypeRole: lruTypeRoles)
					{
						lruTypeRole.setSimulated(value, true);
					}
					startupsTree.refresh();
				} catch (Exception e) {
					RcpUtils.errorMessage(e, getSite().getShell(), "Error trying to modify simulated value", e.getMessage());
				} finally {
					getSite().getShell().setCursor(null);
				}
			}
		}
	}
	
	private class SetBaseElementStartedAction extends Action
	{
		@SuppressWarnings("hiding")
		private static final String ID = "start_baseelement.action";
		private BaseElementStartup[] beStartups;
		private boolean value;
		
		public SetBaseElementStartedAction(BaseElementStartup[] bes, boolean val)
		{
			this.beStartups = bes;
			setId(ID);
			if(val) {
				setText( START_ALL );	
			} 
			else
			{
				setText(STOP_ALL );	
			}
			
			value = val;
		}
		
		@Override
		public void run() 
		{
			try
			{
				getSite().getShell().setCursor(getSite().getShell().getDisplay().getSystemCursor(SWT.CURSOR_WAIT));
				for(BaseElementStartup bes: beStartups) {
					BaseElementStartupHelper helper = (BaseElementStartupHelper) StartupHelperFactory.getInstance().getHelper(bes);
					helper.setStarted(value, false);
				}
			}
			catch (Exception e) {
				RcpUtils.errorMessage(e, getSite().getShell(), "Error trying to modify startups", e.getMessage());
			} finally {
				startupsTree.refresh();
				getSite().getShell().setCursor(null);
			}
		}
	}
	
	private class SetBaseElementSimulatedAction extends Action
	{
		@SuppressWarnings("hiding")
		private static final String ID = "simulate_baseelement.action";
		private BaseElementStartup[] beStartups;
		private boolean value;
		
		public SetBaseElementSimulatedAction(BaseElementStartup[] bes, boolean val)
		{
			this.beStartups = bes;
			this.value = val;
			
			if(val) {
				setText( SIMULATED );	
			} 
			else
			{
				setText( NOT_SIMULATED );	
			}
			
			setId(ID);
		}
		
		@Override
		public void run() 
		{
			try {
				getSite().getShell().setCursor(getSite().getShell().getDisplay().getSystemCursor(SWT.CURSOR_WAIT));
				for(BaseElementStartup bes: beStartups)
				{
					BaseElementStartupHelper helper = (BaseElementStartupHelper) StartupHelperFactory.getInstance().getHelper(bes);
					helper.setSimulated(value);
				}
			} catch (Exception e) {
				RcpUtils.errorMessage(e, getSite().getShell(), "Error trying to modify simulated value", e.getMessage());
			} finally {
				startupsTree.refresh();
				getSite().getShell().setCursor(null);
			}
		}
	}
	
	
	/**
     * Expand the startup scenarios tree to display all antennas 
     * associated with the input startup scenario
     */
    private void expandToAntennas( StartupScenario scenario ) {
        
        // We need to recover the complete chain of objects we want to display,
        // and because of the way this tree is built, we need to invoke
        // the tree's contents provider
        StartupScenarioTreeContentsProvider provider = 
            (StartupScenarioTreeContentsProvider) 
            startupsTree.getContentProvider();
        
        // The chain of objects to be expanded needs to include the 
        // list of antennas
        Object[] baseElementGroups = provider.getChildren( scenario );
        Object antennaList = null;
        for( Object object : baseElementGroups ) {
            if( object instanceof AntennaStartupList ) {
                antennaList = object;
            }
        }
        Object[] toBeExpanded = {
                scenario.getConfiguration(), 
                scenario,
                antennaList
        };

        Object[] expanded = startupsTree.getExpandedElements();
        Object[] merged = Arrays.copyOf( expanded, expanded.length + 3 );
        for( int i = 0; i < 3; i++ ) {
            merged[expanded.length + i] = toBeExpanded[i];
        }
        startupsTree.setExpandedElements( merged );
    }
    
    /**
     * Expand the startup scenarios tree to display all weather stations 
     * associated with the input startup scenario
     */
    private void expandToWeatherStations( StartupScenario scenario ) {
        
        // We need to recover the complete chain of objects we want to display,
        // and because of the way this tree is built, we need to invoke
        // the tree's contents provider
        StartupScenarioTreeContentsProvider provider = 
            (StartupScenarioTreeContentsProvider) 
            startupsTree.getContentProvider();
        
        // The chain of objects to be expanded needs to include the 
        // list of antennas
        Object[] baseElementGroups = provider.getChildren( scenario );
        Object weatherStationList = null;
        for( Object object : baseElementGroups ) {
            if( object instanceof WeatherStationStartupList ) {
                weatherStationList = object;
            }
        }
        Object[] toBeExpanded = {
                scenario.getConfiguration(), 
                scenario,
                weatherStationList
        };

        Object[] expanded = startupsTree.getExpandedElements();
        Object[] merged = Arrays.copyOf( expanded, expanded.length + 3 );
        for( int i = 0; i < 3; i++ ) {
            merged[expanded.length + i] = toBeExpanded[i];
        }
        startupsTree.setExpandedElements( merged );
    }
    
    /**
     * Expand the startup scenarios tree to display all central racks
     * associated with the input startup scenario
     */
    private void expandToCentralRacks( StartupScenario scenario ) {
        
        // We need to recover the complete chain of objects we want to display,
        // and because of the way this tree is built, we need to invoke
        // the tree's contents provider
        StartupScenarioTreeContentsProvider provider = 
            (StartupScenarioTreeContentsProvider) 
            startupsTree.getContentProvider();
        
        // The chain of objects to be expanded needs to include the 
        // list of antennas
        Object[] baseElementGroups = provider.getChildren( scenario );
        Object centralRackList = null;
        for( Object object : baseElementGroups ) {
            if( object instanceof CentralRackStartupList ) {
                centralRackList = object;
            }
        }
        Object[] toBeExpanded = {
                scenario.getConfiguration(), 
                scenario,
                centralRackList
        };

        Object[] expanded = startupsTree.getExpandedElements();
        Object[] merged = Arrays.copyOf( expanded, expanded.length + 3 );
        for( int i = 0; i < 3; i++ ) {
            merged[expanded.length + i] = toBeExpanded[i];
        }
        startupsTree.setExpandedElements( merged );
    }
    
    /**
     * Expand the startup scenarios tree to display all master clocks
     * associated with the input startup scenario
     */
    private void expandToMasterClocks( StartupScenario scenario ) {
        
        // We need to recover the complete chain of objects we want to display,
        // and because of the way this tree is built, we need to invoke
        // the tree's contents provider
        StartupScenarioTreeContentsProvider provider = 
            (StartupScenarioTreeContentsProvider) 
            startupsTree.getContentProvider();
        
        // The chain of objects to be expanded needs to include the 
        // list of antennas
        Object[] baseElementGroups = provider.getChildren( scenario );
        Object masterClockList = null;
        for( Object object : baseElementGroups ) {
            if( object instanceof MasterClockStartupList ) {
                masterClockList = object;
            }
        }
        Object[] toBeExpanded = {
                scenario.getConfiguration(), 
                scenario,
                masterClockList
        };

        Object[] expanded = startupsTree.getExpandedElements();
        Object[] merged = Arrays.copyOf( expanded, expanded.length + 3 );
        for( int i = 0; i < 3; i++ ) {
            merged[expanded.length + i] = toBeExpanded[i];
        }
        startupsTree.setExpandedElements( merged );
    }

	public void refresh() {
		this.startupsTree.refresh();
	}
}
