/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
/**
 * SoftwareDeploymentView.java
 *
 * Copyright European Southern Observatory 2010
 */

package alma.obops.tmcdbgui.views;

import org.eclipse.jface.action.IAction;
import org.eclipse.jface.action.IMenuListener;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.action.Separator;
import org.eclipse.jface.viewers.DecoratingStyledCellLabelProvider;
import org.eclipse.jface.viewers.DoubleClickEvent;
import org.eclipse.jface.viewers.IBaseLabelProvider;
import org.eclipse.jface.viewers.IDoubleClickListener;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.TreePath;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.dnd.DND;
import org.eclipse.swt.dnd.Transfer;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Label;
import org.eclipse.ui.ISelectionListener;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.part.ViewPart;

import alma.acs.tmcdb.AcsService;
import alma.acs.tmcdb.BACIProperty;
import alma.acs.tmcdb.ChannelMapping;
import alma.acs.tmcdb.Component;
import alma.acs.tmcdb.Computer;
import alma.acs.tmcdb.Configuration;
import alma.acs.tmcdb.Container;
import alma.acs.tmcdb.ContainerStartupOption;
import alma.acs.tmcdb.DefaultCanAddress;
import alma.acs.tmcdb.DomainsMapping;
import alma.acs.tmcdb.Manager;
import alma.acs.tmcdb.NotificationServiceMapping;
import alma.obops.tmcdbgui.domain.IModelChangeListener;
import alma.obops.tmcdbgui.handlers.CopyComponentAction;
import alma.obops.tmcdbgui.handlers.DeleteSwDeploymentObjectAction;
import alma.obops.tmcdbgui.handlers.DeleteWrappedSwDeploymentObjectAction;
import alma.obops.tmcdbgui.handlers.DeployAcsServiceAction;
import alma.obops.tmcdbgui.handlers.DeployComponentAction;
import alma.obops.tmcdbgui.handlers.DeployContainerAction;
import alma.obops.tmcdbgui.handlers.DeployWrappedAcsServiceAction;
import alma.obops.tmcdbgui.handlers.EditAcsServiceAction;
import alma.obops.tmcdbgui.handlers.EditAcsServiceWrapperAction;
import alma.obops.tmcdbgui.handlers.EditBACIPropertyAction;
import alma.obops.tmcdbgui.handlers.EditChannelMappingAction;
import alma.obops.tmcdbgui.handlers.EditComponentAction;
import alma.obops.tmcdbgui.handlers.EditComputerAction;
import alma.obops.tmcdbgui.handlers.EditContainerAction;
import alma.obops.tmcdbgui.handlers.EditContainerStartupOptionAction;
import alma.obops.tmcdbgui.handlers.EditDefaultCanAddressAction;
import alma.obops.tmcdbgui.handlers.EditDomainsMappingAction;
import alma.obops.tmcdbgui.handlers.EditManagerAction;
import alma.obops.tmcdbgui.handlers.NewAcsServiceAction;
import alma.obops.tmcdbgui.handlers.NewAcsServiceWhenWrapperSelectedAction;
import alma.obops.tmcdbgui.handlers.NewBACIPropertyAction;
import alma.obops.tmcdbgui.handlers.NewChannelMappingAction;
import alma.obops.tmcdbgui.handlers.NewComponentAction;
import alma.obops.tmcdbgui.handlers.NewComputerAction;
import alma.obops.tmcdbgui.handlers.NewContainerAction;
import alma.obops.tmcdbgui.handlers.NewContainerStartupOptionAction;
import alma.obops.tmcdbgui.handlers.NewDefaultCanAddressAction;
import alma.obops.tmcdbgui.handlers.NewDomainsMappingAction;
import alma.obops.tmcdbgui.handlers.NewNotificationServiceMappingAction;
import alma.obops.tmcdbgui.handlers.UndeployComponentAction;
import alma.obops.tmcdbgui.handlers.UndeployContainerAction;
import alma.obops.tmcdbgui.handlers.CopyComponentAction.CopyOperation;
import alma.obops.tmcdbgui.rcp.ApplicationActionBarAdvisor;
import alma.obops.tmcdbgui.rcp.ApplicationWorkbenchWindowAdvisor;
import alma.obops.tmcdbgui.utils.conversation.DefaultCanAddressConversationUtils;
import alma.obops.tmcdbgui.views.dnd.SwDeploymentDragListener;
import alma.obops.tmcdbgui.views.dnd.SwDeploymentTreeDropAdapter;
import alma.obops.tmcdbgui.views.dnd.TmcdbObjectTransfer;
import alma.obops.tmcdbgui.views.providers.SwDeploymentTreeContentsProvider;
import alma.obops.tmcdbgui.views.providers.SwDeploymentTreeLabelProvider;
import alma.obops.tmcdbgui.views.providers.SwDeploymentTreeSorter;
import alma.obops.tmcdbgui.views.support.AcsServiceWrapper;
import alma.tmcdb.domain.HwConfiguration;

/**
 * This view contains the deployment information of the SW tables
 * for a selected configuration
 * 
 * @author rtobar, Feb 19, 2010
 * 
 */



public class SoftwareDeploymentView extends ViewPart implements IModelChangeListener {

    public static final String ID = "sw-deployment.view";
    private TreeViewer deploymentViewer;
    private Label configurationNameText;

    private ApplicationActionBarAdvisor applicationActionBarAdvisor;
    
    // Actions
    private IAction cloneComponent;
    private IAction copyComponent;
    private IAction editContainer;
    private IAction editContainerStartupOption;
    private IAction editBACIProperty;
    private IAction deployAcsService;
    private IAction deployWrappedAcsService;
    private IAction editAcsService;
    private IAction editAcsServiceWrapper;
    private IAction editChannelMapping;
    private IAction editComponent;
    private IAction editComputer;
    private IAction editDCA;
    private IAction editDomainsMapping;
    private IAction editManager;
    private IAction newDCA;
    private IAction undeployComponent;
    private IAction deployComponent;
    private IAction undeployContainer;
    private IAction deployContainer;
    private DeleteSwDeploymentObjectAction  deleteObject;
    private DeleteWrappedSwDeploymentObjectAction  deleteWrappedObject;
    private HwConfiguration hwConfiguration;
    private NewComputerAction newComputer;
    private NewContainerAction newContainer;
    private NewContainerStartupOptionAction newContainerStartupOption;
    private NewAcsServiceAction newAcsService;
    private NewAcsServiceWhenWrapperSelectedAction newAcsServiceWrapper;
    private NewComponentAction newComponent;
    private NewBACIPropertyAction newBACIProperty;
    private NewDomainsMappingAction newDomainsMapping;
    private NewChannelMappingAction newChannelMapping;
    private NewNotificationServiceMappingAction newNotificationServiceMapping;
    
    public HwConfiguration getHwConfiguration() {
    	return this.hwConfiguration;
    }
    
    public void createPartControl( Composite parent ) {

    	Composite c = new Composite(parent, SWT.NONE);
    	GridLayout gridLayout = new GridLayout();
    	gridLayout.marginWidth = 0;
    	gridLayout.marginHeight = 0;
    	gridLayout.numColumns = 1;
    	c.setLayout(gridLayout);

    	GridData gd = new GridData(GridData.FILL, GridData.BEGINNING, true, false);
    	gd.horizontalSpan = 0;
    	gd.horizontalIndent = 2;
    	configurationNameText = new Label(c, SWT.NONE);
    	configurationNameText.setText("[No configuration]");
    	configurationNameText.setLayoutData(gd);
    	configurationNameText.setAlignment(SWT.RIGHT);

    	IBaseLabelProvider lp = new DecoratingStyledCellLabelProvider(new SwDeploymentTreeLabelProvider(), getSite().getWorkbenchWindow().getWorkbench().getDecoratorManager(), null);
    	gd = new GridData(GridData.FILL, GridData.FILL, true, true);
    	gd.horizontalSpan = 0;
    	deploymentViewer = new TreeViewer(c, SWT.BORDER | SWT.FULL_SELECTION | SWT.MULTI);
    	deploymentViewer.setContentProvider( new SwDeploymentTreeContentsProvider() );
    	deploymentViewer.setLabelProvider( lp );
    	deploymentViewer.setSorter( new SwDeploymentTreeSorter() );
    	deploymentViewer.getControl().setLayoutData(gd);

    	// Register our viewer as a source of selections
        getSite().setSelectionProvider( deploymentViewer );

    	// Add support for dragging and dropping TmcdbObjectTransfer transfers into and from the deployment tree
    	Transfer[] types = new Transfer[] { TmcdbObjectTransfer.getInstance() };
		deploymentViewer.addDragSupport( DND.DROP_MOVE | DND.DROP_COPY, types, new SwDeploymentDragListener(deploymentViewer));
		deploymentViewer.addDropSupport( DND.DROP_MOVE, types, new SwDeploymentTreeDropAdapter(deploymentViewer));

		makeActions();
		makeContextMenu();
    }

    private void makeActions() {

    	IWorkbenchWindow win = getSite().getWorkbenchWindow();
    	Configuration config = (hwConfiguration == null) ? null : hwConfiguration.getSwConfiguration();
    	newContainer     = new NewContainerAction(win, config);
    	newContainerStartupOption = new NewContainerStartupOptionAction(win);
    	newComponent     =  new NewComponentAction(win, config);
    	newComputer      =  new NewComputerAction(win, config);
    	newBACIProperty  = new NewBACIPropertyAction(win, config);
    	newAcsService = new NewAcsServiceAction(win, config);
    	newAcsServiceWrapper = new NewAcsServiceWhenWrapperSelectedAction(win, config);
    	newDomainsMapping = new NewDomainsMappingAction(win);
    	newChannelMapping = new NewChannelMappingAction(win);
    	newNotificationServiceMapping = new NewNotificationServiceMappingAction(win, config);
    	
    	undeployComponent = new UndeployComponentAction( win );
    	deployComponent   = new DeployComponentAction( win );
    	undeployContainer = new UndeployContainerAction( win );
    	deployContainer   = new DeployContainerAction( win );

    	cloneComponent   = new CopyComponentAction(CopyOperation.CLONE, win, this);
    	copyComponent    = new CopyComponentAction(CopyOperation.COPY, win, this);
    	editChannelMapping  = new EditChannelMappingAction( win );
    	editContainer    = new EditContainerAction( win );
    	editContainerStartupOption   = new EditContainerStartupOptionAction( win );
    	editDomainsMapping  = new EditDomainsMappingAction( win );
    	editManager = new EditManagerAction( win );
    	deployAcsService = new DeployAcsServiceAction( win );
    	deployWrappedAcsService = new DeployWrappedAcsServiceAction( win );
    	editAcsService   = new EditAcsServiceAction( win );
    	editAcsServiceWrapper = new EditAcsServiceWrapperAction( win );
    	editComponent    = new EditComponentAction( win );
    	editComputer     = new EditComputerAction( win );
    	editBACIProperty = new EditBACIPropertyAction(win);
    	editDCA          = new EditDefaultCanAddressAction( win );
    	newDCA           = new NewDefaultCanAddressAction( win );
    	
    	deleteObject= new DeleteSwDeploymentObjectAction( win );
    	deleteObject.addModelChangeListener(this);
    	deleteWrappedObject= new DeleteWrappedSwDeploymentObjectAction( win );
    	deleteWrappedObject.addModelChangeListener(this);

    	// Bind our deleteObject action with the global "delete" retargetable action
    	getViewSite().getActionBars().setGlobalActionHandler("delete", deleteObject);

    	// Double-click support for opening the editors
    	deploymentViewer.addDoubleClickListener(new IDoubleClickListener() {
			public void doubleClick(DoubleClickEvent event) {
				if( event.getSelection() instanceof IStructuredSelection ) {
					IStructuredSelection selection = (IStructuredSelection)event.getSelection();
					if( selection.getFirstElement() instanceof Container ) {
						((EditContainerAction)editContainer).selectionChanged(SoftwareDeploymentView.this, selection);
						editContainer.run();
					}
					else if( selection.getFirstElement() instanceof ContainerStartupOption ) {
						((EditContainerStartupOptionAction)editContainerStartupOption).selectionChanged(SoftwareDeploymentView.this, selection);
						editContainerStartupOption.run();
					}
					else if( selection.getFirstElement() instanceof AcsService ) {
						((EditAcsServiceAction)editAcsService).selectionChanged(SoftwareDeploymentView.this, selection);
						editAcsService.run();
					}
					else if( selection.getFirstElement() instanceof AcsServiceWrapper ) {
						((EditAcsServiceWrapperAction)editAcsServiceWrapper).selectionChanged(SoftwareDeploymentView.this, selection);
						editAcsServiceWrapper.run();
					}
					else if ( selection.getFirstElement() instanceof Component ) {
						((EditComponentAction)editComponent).selectionChanged(SoftwareDeploymentView.this, selection);
						editComponent.run();
					}
					else if ( selection.getFirstElement() instanceof Computer ) {
						((EditComputerAction)editComputer).selectionChanged(SoftwareDeploymentView.this, selection);
						editComputer.run();
					}
					else if ( selection.getFirstElement() instanceof ChannelMapping ) {
						((EditChannelMappingAction)editChannelMapping).selectionChanged(SoftwareDeploymentView.this, selection);
						editChannelMapping.run();
					}
					else if ( selection.getFirstElement() instanceof DomainsMapping ) {
						((EditDomainsMappingAction)editDomainsMapping).selectionChanged(SoftwareDeploymentView.this, selection);
						editDomainsMapping.run();
					}
					else if ( selection.getFirstElement() instanceof Manager ) {
						((EditManagerAction)editManager).selectionChanged(SoftwareDeploymentView.this, selection);
						editManager.run();
					}
					else if ( selection.getFirstElement() instanceof BACIProperty ) {
						((EditBACIPropertyAction)editBACIProperty).selectionChanged(SoftwareDeploymentView.this, selection);
						editBACIProperty.run();
					}
				}
			}
		});

    }

	private void makeContextMenu() {
		final MenuManager mgr = new MenuManager();
        mgr.setRemoveAllWhenShown(true);
        mgr.addMenuListener( new IMenuListener() {
			public void menuAboutToShow(IMenuManager manager) {
				fillContextMenu(manager);
			}
		});
        Control ctrl = deploymentViewer.getControl();
        ctrl.setMenu( mgr.createContextMenu( ctrl ));
	}

	private void fillContextMenu(IMenuManager manager) {
		
		ISelection selection = deploymentViewer.getSelection();

        if( selection instanceof IStructuredSelection ) {
            IStructuredSelection sselection = (IStructuredSelection) selection;
            Object selNode = sselection.getFirstElement();

            if( selNode instanceof Component ) {
            	if(((Component)selNode).getContainer() != null) {
            		manager.add(undeployComponent);
            	} else {
            		manager.add(deployComponent);
            	}
            	manager.add(newBACIProperty);
            	manager.add( new Separator() );
            	manager.add(copyComponent);
            	manager.add(cloneComponent);
            	manager.add( new Separator() );
            	manager.add(deleteObject);
            	manager.add( new Separator() );

            	DefaultCanAddress dca = null;
            	try {
					dca = DefaultCanAddressConversationUtils.getInstance().findForComponent((Component)selNode);
	            	((ISelectionListener)editDCA).selectionChanged(this, new StructuredSelection(dca));
	            	manager.add(editDCA);
				} catch (Exception e) {
					/* make sure editDCA action is disabled */
					((ISelectionListener)newDCA).selectionChanged(this, new StructuredSelection(selNode));
					manager.add(newDCA);
				}
				
            }
            else if( selNode instanceof Component[] ) {
            	manager.add(newComponent);
            }
            else if( selNode instanceof Container ) {
            	if(((Container)selNode).getComputer() != null) {
            			manager.add(undeployContainer);
            	} else {
            		manager.add(deployContainer);
            	}
            	manager.add(newComponent);
            	manager.add(newContainerStartupOption);
            	manager.add( new Separator() );
            	manager.add(deleteObject);
            }
            else if( selNode instanceof Container[] ) {
            	manager.add(newContainer);
            }
            else if( selNode instanceof ContainerStartupOption ) {
            	manager.add(editContainerStartupOption);
            	manager.add(deleteObject);
            }
            else if( selNode instanceof ContainerStartupOption[] ) {
            	manager.add(newContainerStartupOption);
            }
            else if( selNode instanceof DomainsMapping[]) {
            	manager.add(newDomainsMapping);
            }
            else if( selNode instanceof ChannelMapping[]) {
            	manager.add(newChannelMapping);
            }
            else if( selNode instanceof NotificationServiceMapping[]) {
            	manager.add(newNotificationServiceMapping);
            }
            else if( selNode instanceof NotificationServiceMapping) {
            	manager.add(newChannelMapping);
            	manager.add(newDomainsMapping);
            }
            else if( selNode instanceof ChannelMapping) {
            	manager.add(editChannelMapping);
            	manager.add(deleteObject);
            }
            else if( selNode instanceof DomainsMapping) {
            	manager.add(editDomainsMapping);
            	manager.add(deleteObject);
            }
            else if( selNode instanceof Manager) {
            	manager.add(editManager);
            }
            else if( selNode instanceof AcsService ) {
            	manager.add(deployAcsService);
            	manager.add(deleteObject);
            }
            else if( selNode instanceof AcsServiceWrapper ) {
            	manager.add(deployWrappedAcsService);
            	manager.add(deleteWrappedObject);
            }
            else if( selNode instanceof AcsService[] ) {
            	manager.add(newAcsService);
            }
            else if( selNode instanceof AcsServiceWrapper[] ) {
            	manager.add(newAcsServiceWrapper);
            }
            else if( selNode instanceof Computer ) {
            	manager.add(newContainer);
            	manager.add(newAcsService);
            	manager.add( new Separator() );
            	manager.add(deleteObject);
            }
            else if( selNode instanceof Computer[] ) {
            	manager.add(newComputer);
            }
            else if( selNode instanceof BACIProperty ) {
            	manager.add(deleteObject);
            }
            else if( selNode == null ) {
            	manager.add(newComputer);
            	manager.add(newContainer);
            	manager.add(newComponent);
            }
        }
	}

	@Override
	public void setFocus() {
		deploymentViewer.getControl().setFocus();
	}

	public TreeViewer getDeploymentViewer() {
		return deploymentViewer;
	}

	@Override
	public void externalModelChange() {
		
	}

	@Override
	public void internalModelChange() {
		TreePath[] paths = deploymentViewer.getExpandedTreePaths();
		deploymentViewer.refresh();
		deploymentViewer.setExpandedTreePaths(paths);
	}

	public void setInput(HwConfiguration conf) {
		this.configurationNameText.setText('[' + conf.getSwConfiguration().getConfigurationName() + ']');
		this.deploymentViewer.setInput(conf);
		this.hwConfiguration = conf;
		this.newBACIProperty.setConfiguration(conf.getSwConfiguration());
		this.newComponent.setConfiguration(conf.getSwConfiguration());
		this.newContainer.setConfiguration(conf.getSwConfiguration());
		this.newComputer.setConfiguration(conf.getSwConfiguration());
		this.newAcsService.setConfiguration(conf.getSwConfiguration());
		this.newAcsServiceWrapper.setConfiguration(conf.getSwConfiguration());
		this.newNotificationServiceMapping.setConfiguration(conf.getSwConfiguration());
		this.applicationActionBarAdvisor = ApplicationWorkbenchWindowAdvisor.getAdvisor();
		this.applicationActionBarAdvisor.configurationChanged(conf.getSwConfiguration());
	}

	public void addSwDeploymentViewListener(
			ApplicationActionBarAdvisor appActionBarAdvisor) 
	{
		this.applicationActionBarAdvisor = appActionBarAdvisor;
	}
}
