/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdbgui.views;

import java.io.IOException;
import java.util.logging.Handler;
import java.util.logging.Level;
import java.util.logging.LogRecord;

import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.IPartListener;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.console.IConsoleConstants;
import org.eclipse.ui.console.MessageConsole;
import org.eclipse.ui.console.MessageConsoleStream;

import alma.acs.logging.AcsLogger;
import alma.acs.logging.ClientLogManager;
import alma.acs.logging.config.LogConfig;
import alma.acs.logging.level.AcsLogLevelDefinition;
import alma.obops.tmcdb.alarms.ui.utils.RcpUtils;
import alma.obops.tmcdbgui.TmcdbGui;

public class SQLLogConsole {

	private AcsLogger hibernateSQLLogger = ClientLogManager.getAcsLogManager().getLoggerForCorba("hibernateSQL", true);
	private Handler consoleLogHandler;
	private IPartListener partListener;
	private MessageConsole console;

	public SQLLogConsole(IWorkbenchWindow window) {

		console = new MessageConsole("Hibernate SQL Log", RcpUtils.getImageDescriptor("icons/sql.gif"), true);

		// Create the stream
		final MessageConsoleStream stream = console.newMessageStream();
		stream.setActivateOnWrite(false);
		stream.setColor(Display.getCurrent().getSystemColor(SWT.COLOR_BLUE)); // Choose other color if you prefer

		// Custom handler that will be attached/detached from the hibernateSQLLogger
		// and prints the logs in the Eclipse Console View
		consoleLogHandler = new Handler() {

			public void publish(LogRecord record) {
				stream.print( getFormatter().format(record) );
			}

			public void flush() {
				try {
					stream.flush();
				} catch (IOException e) { }
			}
			
			public void close() throws SecurityException {
				try {
					stream.close();
				} catch (IOException e) { }
			}
		};
		consoleLogHandler.setFormatter( hibernateSQLLogger.getHandlers()[0].getFormatter() );

		// Register a partListener, so we turn on/off the ALL logging when the Console view is opened/closed
		partListener = new IPartListener() {

			@Override
			public void partOpened(IWorkbenchPart part) {
				if( part.getSite().getId().equals(IConsoleConstants.ID_CONSOLE_VIEW) )
					addConsoleHandlerToHibernateLogger();	
			}

			@Override
			public void partClosed(IWorkbenchPart part) {
				if( part.getSite().getId().equals(IConsoleConstants.ID_CONSOLE_VIEW) )
					removeConsoleHandlerFromHibernateLogger();
			}

			@Override
			public void partDeactivated(IWorkbenchPart part) {
				if( part.getSite().getId().equals(IConsoleConstants.ID_CONSOLE_VIEW) &&
				    !part.getSite().getPage().isPartVisible(part) )
					removeConsoleHandlerFromHibernateLogger();
			}

			@Override
			public void partActivated(IWorkbenchPart part) {
				if( part.getSite().getId().equals(IConsoleConstants.ID_CONSOLE_VIEW) )
					addConsoleHandlerToHibernateLogger();
			}

			// no-op
			public void partBroughtToTop(IWorkbenchPart part) {
				//System.out.println("top");
			}
		};
		window.getPartService().addPartListener(partListener);
	}

	public void dispose() {
		removeConsoleHandlerFromHibernateLogger();
	}

	public MessageConsole getConsole() {
		return console;
	}

	private void addConsoleHandlerToHibernateLogger() {
		if( !customConsoleHandlerIsAttached() ) {

			// Let the AcsLogger log all messages down to TRACE
			LogConfig logConfigAll = new LogConfig();
			logConfigAll.setDefaultMinLogLevel(AcsLogLevelDefinition.TRACE);
			logConfigAll.setDefaultMinLogLevelLocal(AcsLogLevelDefinition.TRACE);

			hibernateSQLLogger.configureLogging(logConfigAll);
			hibernateSQLLogger.addHandler(consoleLogHandler);
			consoleLogHandler.setLevel(Level.ALL);

			TmcdbGui.getLogger().info("Attached SQL Logger to Console View");
		}
	}

	private void removeConsoleHandlerFromHibernateLogger() {
		if( customConsoleHandlerIsAttached() ) {

			LogConfig logConfigOff = new LogConfig();
			logConfigOff.setDefaultMinLogLevel(AcsLogLevelDefinition.OFF);
			logConfigOff.setDefaultMinLogLevelLocal(AcsLogLevelDefinition.OFF);

			hibernateSQLLogger.configureLogging(logConfigOff);
			hibernateSQLLogger.removeHandler(consoleLogHandler);
			consoleLogHandler.setLevel(Level.OFF);

			TmcdbGui.getLogger().info("Dettached SQL Logger from Console View");
		}
	}

	private boolean customConsoleHandlerIsAttached() {

		for(Handler h: hibernateSQLLogger.getHandlers())
			if( h == consoleLogHandler )
				return true;

		return false;
	}
}
