/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
/**
 * RawDataView.java
 *
 * Copyright European Southern Observatory 2008
 */

package alma.obops.tmcdbgui.views;

import java.util.List;

import org.eclipse.jface.action.GroupMarker;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.action.IMenuListener;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.action.Separator;
import org.eclipse.jface.viewers.DoubleClickEvent;
import org.eclipse.jface.viewers.IDoubleClickListener;
import org.eclipse.jface.viewers.IElementComparer;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.TreePath;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.viewers.TreeViewerColumn;
import org.eclipse.swt.SWT;
import org.eclipse.swt.dnd.DND;
import org.eclipse.swt.dnd.DragSourceListener;
import org.eclipse.swt.dnd.Transfer;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Tree;
import org.eclipse.ui.IPropertyListener;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.part.ViewPart;
import org.hibernate.criterion.MatchMode;

import alma.obops.tmcdb.alarms.ui.tree.helpers.ThreeColumnDomainObjectHelper;
import alma.obops.tmcdb.alarms.ui.utils.RcpUtils;
import alma.obops.tmcdbgui.domain.IModelChangeListener;
import alma.obops.tmcdbgui.editors.ConfigurationEditor;
import alma.obops.tmcdbgui.editors.inputs.ConfigurationEditorInput;
import alma.obops.tmcdbgui.handlers.AddAntennaAction;
import alma.obops.tmcdbgui.handlers.AddAssemblyAction;
import alma.obops.tmcdbgui.handlers.AddFrontendAction;
import alma.obops.tmcdbgui.handlers.AddHolographyTowerAction;
import alma.obops.tmcdbgui.handlers.AddPadAction;
import alma.obops.tmcdbgui.handlers.AddWeatherStationAction;
import alma.obops.tmcdbgui.handlers.AssignAntennaToPadAction;
import alma.obops.tmcdbgui.handlers.AssignPadToHolographyTowerAction;
import alma.obops.tmcdbgui.handlers.BulkBACIPropertiesChangesByAssemblyTypeAction;
import alma.obops.tmcdbgui.handlers.BulkBACIPropertiesChangesByHwConfigurationAction;
import alma.obops.tmcdbgui.handlers.BulkBACIPropertiesCreationByAssemblyTypeAction;
import alma.obops.tmcdbgui.handlers.BulkBACIPropertiesCreationByHwConfigurationAction;
import alma.obops.tmcdbgui.handlers.BulkBACIPropertiesDeletionByAssemblyTypeAction;
import alma.obops.tmcdbgui.handlers.BulkBACIPropertiesDeletionByHwConfigurationAction;
import alma.obops.tmcdbgui.handlers.BulkComponentChangesByHwConfigurationAction;
import alma.obops.tmcdbgui.handlers.CheckForDuplicatedLoOffsetsAction;
import alma.obops.tmcdbgui.handlers.CheckForDuplicatedWalshSequencesAction;
import alma.obops.tmcdbgui.handlers.CloneAntennaAction;
import alma.obops.tmcdbgui.handlers.CloneConfigurationAction;
import alma.obops.tmcdbgui.handlers.CloseAntennaToPadAssignmentAction;
import alma.obops.tmcdbgui.handlers.CopyAntennaAction;
import alma.obops.tmcdbgui.handlers.DeleteAntennaAction;
import alma.obops.tmcdbgui.handlers.DeleteFrontendAction;
import alma.obops.tmcdbgui.handlers.DeleteWeatherStationAction;
import alma.obops.tmcdbgui.handlers.EditAcaCorrDelaysAction;
import alma.obops.tmcdbgui.handlers.EditAntennaAction;
import alma.obops.tmcdbgui.handlers.EditAntennaToPadAction;
import alma.obops.tmcdbgui.handlers.EditAssemblyAction;
import alma.obops.tmcdbgui.handlers.EditConfigurationAction;
import alma.obops.tmcdbgui.handlers.EditDelayModelAction;
import alma.obops.tmcdbgui.handlers.EditFocusModelAction;
import alma.obops.tmcdbgui.handlers.EditFrontEndAction;
import alma.obops.tmcdbgui.handlers.EditHolographyTowerAction;
import alma.obops.tmcdbgui.handlers.EditPadAction;
import alma.obops.tmcdbgui.handlers.EditPadToHolographyTowerAction;
import alma.obops.tmcdbgui.handlers.EditPointingModelAction;
import alma.obops.tmcdbgui.handlers.EditWeatherStationControllerAction;
import alma.obops.tmcdbgui.handlers.EditXpDelaysAction;
import alma.obops.tmcdbgui.handlers.ExportConfigurationAction;
import alma.obops.tmcdbgui.handlers.ImportConfigurationAction;
import alma.obops.tmcdbgui.handlers.QueryConfigurationsAction;
import alma.obops.tmcdbgui.handlers.RenameAssemblyAction;
import alma.obops.tmcdbgui.handlers.ShowAlarmCategoriesAction;
import alma.obops.tmcdbgui.handlers.ShowDefaultCanAddressesAction;
import alma.obops.tmcdbgui.handlers.ShowNotificationChannelsAction;
import alma.obops.tmcdbgui.handlers.ShowSwDeploymentAction;
import alma.obops.tmcdbgui.handlers.UnassignPadFromHolographyTowerAction;
import alma.obops.tmcdbgui.handlers.conversation.IConversationalAction;
import alma.obops.tmcdbgui.utils.GuiUtils;
import alma.obops.tmcdbgui.views.dnd.ConfigurationsDragListener;
import alma.obops.tmcdbgui.views.dnd.HwConfigurationBaseElementDropAdapter;
import alma.obops.tmcdbgui.views.dnd.TmcdbObjectTransfer;
import alma.obops.tmcdbgui.views.providers.ConfigurationTreeContentsProvider;
import alma.obops.tmcdbgui.views.providers.ConfigurationTreeLabelProvider;
import alma.obops.tmcdbgui.views.providers.ConfigurationTreeSorter;
import alma.obops.tmcdbgui.views.providers.helpers.config.CentralRackList;
import alma.obops.tmcdbgui.views.providers.helpers.config.ConfigHelperFactory;
import alma.obops.tmcdbgui.views.providers.helpers.config.DelayModel;
import alma.obops.tmcdbgui.views.providers.helpers.config.HolographyTowerList;
import alma.obops.tmcdbgui.views.providers.helpers.config.MasterClockList;
import alma.obops.tmcdbgui.views.providers.helpers.config.WeatherStationList;
import alma.obops.tmcdbgui.views.providers.helpers.config.XpDelaysModel;
import alma.obops.tmcdbgui.views.providers.helpers.startup.StartupHelperFactory;
import alma.obops.tmcdbgui.views.providers.typedlists.AntennaList;
import alma.obops.tmcdbgui.views.providers.typedlists.AssemblyList;
import alma.obops.tmcdbgui.views.providers.typedlists.BaseElementList;
import alma.obops.tmcdbgui.views.providers.typedlists.FrontEndList;
import alma.obops.tmcdbgui.views.providers.typedlists.PadList;
import alma.obops.tmcdbgui.views.providers.typedlists.TypeList;
import alma.tmcdb.domain.AcaCorrDelays;
import alma.tmcdb.domain.Antenna;
import alma.tmcdb.domain.AntennaToPad;
import alma.tmcdb.domain.Assembly;
import alma.tmcdb.domain.BaseElement;
import alma.tmcdb.domain.FocusModel;
import alma.tmcdb.domain.FrontEnd;
import alma.tmcdb.domain.HolographyTower;
import alma.tmcdb.domain.HolographyTowerToPad;
import alma.tmcdb.domain.HwConfiguration;
import alma.tmcdb.domain.Pad;
import alma.tmcdb.domain.PointingModel;
import alma.tmcdb.domain.WeatherStationController;

/**
 * Shows a list of configuration trees
 *
 * @author amchavan, Sep 3, 2008
 * 
 */



public class ConfigurationsView extends ViewPart implements IPropertyListener, IModelChangeListener, IConfigurationSearcher {

    public static final String ID = "configuration.view";
    private TreeViewer confTree;
    private String searchCriteriaName;
    private boolean searchCriteriaActive;
    private boolean searchCriteriaAllActiveStates;
    private MatchMode searchCriteriaMatchMode;

    /* Our sw actions */
    private IAction exportConfigurationAction;
	private IAction showNotificationChannelsAction;
	private IAction showSwDeployAction;
	private IAction showAlarmCategoriesAction;
	private IAction showDCAAction;
	private IAction bulkComponentChangesByHwConfigAction;
	private IAction bulkBACIChangesByHwConfigAction;
	private IAction bulkBACIChangesByAssemblyTypeAction;
	private IAction bulkBACIDeletionByHwConfigAction;
	private IAction bulkBACIDeletionByAssemblyTypeAction;
	private IAction bulkBACICreationByHwConfigAction;
	private IAction bulkBACICreationByAssemblyTypeAction;
    private ImportConfigurationAction importConfigurationAction;

	/* Our hw actions */
	private IConversationalAction addAntennaAction;
	private IConversationalAction addFrontendAction;
	private IAction addHolographyTowerAction;
	private IAction addWeatherStationAction;
    private IConversationalAction addPadAction;
    private IConversationalAction addAssemblyAction;
    private IConversationalAction renameAssemblyAction;
    private IConversationalAction cloneAntennaAction;
    private IConversationalAction assignAntennaToPadAction;
    private IConversationalAction assignPadToHolographyTowerAction;
    private IConversationalAction copyAntennaAction;
    private IConversationalAction cloneConfigurationAction;
    private IConversationalAction deleteAntennaAction;
    // TODO: private IConversationalAction deleteConfigurationAction;
    private IConversationalAction deleteFrontendAction;
    private IConversationalAction deleteWeatherStationAction;

    /* Our hw edition actions */
    private EditConfigurationAction editHwConfigurationAction;
    private EditAntennaToPadAction editAntennaToPadAction;
    private EditPadToHolographyTowerAction editPadToHolographyTowerAction;
    private EditFrontEndAction editFrontEndAction;
    private EditWeatherStationControllerAction editWeatherStationAction;
    private EditHolographyTowerAction editHolographyTowerAction;
    private EditAntennaAction editAntennaAction;
    private EditAcaCorrDelaysAction editAcaCorrDelaysAction;
    private EditAssemblyAction editAssemblyAction;
    private EditPadAction editPadAction;
    private EditPointingModelAction editPointingModelAction;
    private EditFocusModelAction editFocusModelAction;
    private EditDelayModelAction editDelayModelAction;
    private EditXpDelaysAction editXpDelaysAction;
    private CloseAntennaToPadAssignmentAction closeAntennaToPadAssignmentAction;
    private UnassignPadFromHolographyTowerAction unassignPadFromHolographyTowerAction;
    private CheckForDuplicatedLoOffsetsAction checkForDuplicatedLoOffsetsAction;
    private CheckForDuplicatedWalshSequencesAction checkForDuplicatedWalshSequencesAction;

    /**
     * @see org.eclipse.ui.part.WorkbenchPart#createPartControl(org.eclipse.swt.widgets.Composite)
     */
    @Override
    public void createPartControl( Composite parent ) 
    {
        int style = SWT.BORDER | SWT.FULL_SELECTION ;

        confTree = new TreeViewer( parent, style );
       	confTree.setContentProvider( new ConfigurationTreeContentsProvider(ConfigHelperFactory.getInstance()));
        confTree.setSorter(new ConfigurationTreeSorter());
        
        confTree.setComparer(new IElementComparer() {

			@Override
			public boolean equals(Object obj1, Object obj2) 
			{
				boolean retVal = false;
				if(obj1 instanceof HwConfiguration && obj2 instanceof HwConfiguration) 
				{
					HwConfiguration config1 = (HwConfiguration) obj1;
					HwConfiguration config2 = (HwConfiguration) obj2;
					if(config1.getId().equals(config2.getId())) {
						retVal = true;
					} 
				} 
				else if(obj1 instanceof HwConfiguration[] && obj2 instanceof HwConfiguration[]) 
				{
					retVal = true;
				} 
				else if(obj1 instanceof Assembly && obj2 instanceof Assembly) 
				{
					Assembly item1 = (Assembly) obj1;
					Assembly item2 = (Assembly) obj2;
					if(item1.getId().equals(item2.getId())) {
						retVal = true;
					}
				}
				else if(obj1 instanceof BaseElement && obj2 instanceof BaseElement) 
				{
					BaseElement item1 = (BaseElement) obj1;
					BaseElement item2 = (BaseElement) obj2;
					if(item1.getId().equals(item2.getId())) {
						retVal = true;
					}
				}
				else if(obj1 instanceof AssemblyList && obj2 instanceof AssemblyList) 
				{
					AssemblyList item1 = (AssemblyList) obj1;
					AssemblyList item2 = (AssemblyList) obj2;
					if(item1.getHwConfiguration().getId().equals(item2.getHwConfiguration().getId())) {
						retVal = true;
					}
				}
				else if(obj1 instanceof TypeList && obj2 instanceof TypeList) 
				{
					TypeList item1 = (TypeList) obj1;
					TypeList item2 = (TypeList) obj2;
					if(item1.getHwConfiguration().getId().equals(item2.getHwConfiguration().getId())) {
						retVal = true;
					}
				}
				else if(obj1 instanceof AntennaList && obj2 instanceof AntennaList) 
				{
					AntennaList item1 = (AntennaList) obj1;
					AntennaList item2 = (AntennaList) obj2;
					if(item1.getHwConfiguration().getId().equals(item2.getHwConfiguration().getId())) {
						retVal = true;
					}
				}
				else if(obj1 instanceof PadList && obj2 instanceof PadList) 
				{
					PadList item1 = (PadList) obj1;
					PadList item2 = (PadList) obj2;
					if(item1.getHwConfiguration().getId().equals(item2.getHwConfiguration().getId())) {
						retVal = true;
					}
				}
				else if(obj1 instanceof FrontEndList && obj2 instanceof FrontEndList) 
				{
					FrontEndList item1 = (FrontEndList) obj1;
					FrontEndList item2 = (FrontEndList) obj2;
					if(item1.getHwConfiguration().getId().equals(item2.getHwConfiguration().getId())) {
						retVal = true;
					}
				}
				else if(obj1 instanceof MasterClockList && obj2 instanceof MasterClockList) 
				{
					MasterClockList item1 = (MasterClockList) obj1;
					MasterClockList item2 = (MasterClockList) obj2;
					if(item1.getHwConfiguration().getId().equals(item2.getHwConfiguration().getId())) {
						retVal = true;
					}
				}
				else if(obj1 instanceof CentralRackList && obj2 instanceof CentralRackList) 
				{
					CentralRackList item1 = (CentralRackList) obj1;
					CentralRackList item2 = (CentralRackList) obj2;
					if(item1.getHwConfiguration().getId().equals(item2.getHwConfiguration().getId())) {
						retVal = true;
					}
				}
				
				return retVal;
			}

			@Override
			public int hashCode(Object obj) {
				int retVal = obj.hashCode();
				return retVal;
			}
        	
        });
        
        confTree.getTree().setLinesVisible( true );
        confTree.getTree().setHeaderVisible( true );
        
        // First column -- name and icon for all tree nodes
        TreeViewerColumn col0 = new TreeViewerColumn( confTree, SWT.NONE );
        col0.getColumn().setWidth( 150 );
        col0.getColumn().setMoveable( false );
        col0.getColumn().setText( "Name" );
        col0.setLabelProvider( new ConfigurationTreeLabelProvider( 0, ConfigHelperFactory.getInstance() ));

        // Creation actions and context menu
        makeActions();
        makeContextMenu();

        // Allow other views to subscribe to the selection changed events
        // originating on our viewer
        getSite().setSelectionProvider( confTree );

        // Hook-up to the drag and drop infrastructure
        Transfer []types = { TmcdbObjectTransfer.getInstance() };
        DragSourceListener dragListener = new ConfigurationsDragListener(confTree);
        HwConfigurationBaseElementDropAdapter dropAdapter = new HwConfigurationBaseElementDropAdapter( confTree );
        dropAdapter.addPropertyListener(this);

        confTree.addDragSupport(DND.DROP_COPY | DND.DROP_MOVE, types, dragListener);
        confTree.addDropSupport(DND.DROP_COPY | DND.DROP_MOVE, types, dropAdapter);
    }

	/**
     * @see org.eclipse.ui.part.WorkbenchPart#setFocus()
     */
    @Override
    public void setFocus() {
        // no-op
    }
    
    public HwConfiguration[] getInput() {
    	return (HwConfiguration[]) this.confTree.getInput();
    }

    public void setInput( HwConfiguration[] conf ) 
    {
    	try {
    		confTree.getTree().setRedraw(false);

    		// Save the expanded elements
    		//Object[] expandedElements = confTree.getExpandedElements();

    		// store information which can be used to restore 
    		// the current selection, if any
    		//Long id = computeActiveHwConfigurationId();
    		//ISelection selection = confTree.getSelection();

    		// update our treeviewer with updated info
    		confTree.setInput( conf );

    		// restore the current selection, if any
    		//restorePreviousSelectionInHwConfiguration(id, selection, expandedElements);

    		// notify the startup scenarios view that it needs to refresh its info  
    		// so as to remain in sync with the configurations view update(s)
    		updateStartupScenariosView();
    	}
    	finally {
    		confTree.getTree().setRedraw(true);
    	}
    }

    /**
     * This method gets invoked when some editor signals they are done with
     * editing -- we refresh our tree.
     *
     * @see org.eclipse.ui.IPropertyListener#propertyChanged(java.lang.Object, int)
     */
    @Override
    public void propertyChanged( Object source, int propId ) {
    	if(source instanceof AntennaToPad) {
    		ConfigHelperFactory.clearCaches();
    		StartupHelperFactory.getInstance().clearCaches();
    		reloadConfigs();
    		internalModelChange();
    		expandAntennas( ((AntennaToPad)source).getAntenna().getConfiguration() );
    		expandPads( ((AntennaToPad)source).getAntenna().getConfiguration() );
     	} 
    	else if(propId == GuiUtils.DROP_ANTENNA) {
    		this.internalModelChange();
    	}
    }
    
    private void expandAntennas(HwConfiguration configuration) 
    {
    	Object input = confTree.getInput();
    	if(input instanceof HwConfiguration[]) 
    	{
    		HwConfiguration[] configs = (HwConfiguration[]) input;
    		for(HwConfiguration config: configs) 
    		{
    			if(config.getId().equals(configuration.getId())) 
    			{
    				ThreeColumnDomainObjectHelper helper = ConfigHelperFactory.getInstance().getHelper(config);
    				for(Object obj: helper.getChildren()) 
    				{
    					if(obj instanceof AntennaList) 
    					{
    						confTree.setExpandedState(obj, true);
    					}
    				}
    				
    			}
    		}
    	}
    }
    
    public void expandPads(HwConfiguration configuration)
    {
       Object input = confTree.getInput();
       if(input instanceof HwConfiguration[]) 
       {
    	   HwConfiguration[] configs = (HwConfiguration[]) input;
    	   for(HwConfiguration config: configs) 
    	   {
    		   if(config.getId().equals(configuration.getId())) 
    		   {
    			   ThreeColumnDomainObjectHelper helper = ConfigHelperFactory.getInstance().getHelper(config);
    			   for(Object obj: helper.getChildren()) 
    			   {
    				   if(obj instanceof PadList) 
    				   {
    					   confTree.setExpandedState(obj, true);
    				   }
    			   }

    		   }
    	   }
       }
    }
    
    @Override
    public void setSearchCriteria(String configurationName, boolean queryAllActiveStates, boolean activeOnly, MatchMode matchMode) {
    	this.searchCriteriaName = configurationName;
    	this.searchCriteriaActive = activeOnly;
    	this.searchCriteriaMatchMode = matchMode;
    	this.searchCriteriaAllActiveStates = queryAllActiveStates;
    }

	 /**
     * Used to reload configs after a "hibernate-significant" change; this is a hack, but seems to be 
     * necessary in some cases.
     */
    private void reloadConfigs() 
    {
    	Shell ourShell = this.getSite().getShell();
    	ourShell.setCursor(ourShell.getDisplay().getSystemCursor(SWT.CURSOR_WAIT));
    	
    	List<HwConfiguration> newlyQueriedConfigList = 
    		QueryConfigurationsAction.queryConfigsByName(ourShell, searchCriteriaName, searchCriteriaAllActiveStates, searchCriteriaActive, searchCriteriaMatchMode);
    	HwConfiguration[] newlyQueriedConfigs = newlyQueriedConfigList.toArray(new HwConfiguration[0]); 
    	
   		setInput(newlyQueriedConfigs);
   		
   		// (yet another) hack to keep any open configuration editors in sync
   		for(ConfigurationEditor ed: ConfigurationEditor.getOpenEditors()) {
   			HwConfiguration config = ed.getConfiguration();
   			for(HwConfiguration configIterated : newlyQueriedConfigs) {
   				if(configIterated.getId().equals(config.getId())) {
   					ConfigurationEditorInput confEdInp = new ConfigurationEditorInput(configIterated, this);  
   					ed.setInput(confEdInp);
   				}
   			}
   		}
    	
   		ourShell.setCursor(null);
    }
    
    private void refreshTreeAndMaintainSelection()
    {
    	Object[] elements = confTree.getExpandedElements();
    	TreePath[] treePaths = confTree.getExpandedTreePaths();
    	confTree.refresh();
    	confTree.setExpandedElements(elements);
    	confTree.setExpandedTreePaths(treePaths);
    }
    
    /**
     * @return The Tree which represents our set of configurations
     */
    public Tree getConfigurationsTree() {
        return confTree.getTree();
    }
    
    public TreeViewer getConfigurationsTreeViewer() {
    	return confTree;
    }

	@Override
	public void internalModelChange()
	{
		refreshTreeAndMaintainSelection();
	}

	@Override
	public void externalModelChange()
	{
		Object[] elements = confTree.getExpandedElements();
    	TreePath[] treePaths = confTree.getExpandedTreePaths();
		
		// clear helper caches, so that they don't contain stale data/objects
		// which can cause various hibernate problems!
		ConfigHelperFactory.clearCaches();
		StartupHelperFactory.getInstance().clearCaches();

		// reload the configurations, so that our list is properly refreshed with
		reloadConfigs();
		
		confTree.refresh();
    	confTree.setExpandedElements(elements);
    	confTree.setExpandedTreePaths(treePaths);
	}

	/**
	 * Because the configuration view & startup scenario view are tightly coupled, when we have a 'significant' 
	 * change in the configurations view, the startup scenarios view must be notified so that we avoid things
	 * like stale hibernate objects, etc. This method performs the update, when needed.
	 * 
	 * TODO: think about changing the coupling of these views?!
	 */
	private void updateStartupScenariosView() 
	{
		if(!GuiUtils.isGodUser())
		{
			return;
		}
		
		StartupScenariosView startupView = (StartupScenariosView) RcpUtils.findView( StartupScenariosView.ID );

		if( this.confTree.getSelection() instanceof IStructuredSelection )
		{
			IStructuredSelection selection = (IStructuredSelection) this.confTree.getSelection();
			if( selection.size() == 1 &&
					   (selection.getFirstElement() instanceof HwConfiguration) )
			{
				startupView.setInput((HwConfiguration)selection.getFirstElement());
			} 
			else if(selection.size() == 1 &&
					   (selection.getFirstElement() instanceof BaseElement)) 
			{
				startupView.setInput(((BaseElement)selection.getFirstElement()).getConfiguration());
			}
			else if(selection.size() == 1 &&
					   (selection.getFirstElement() instanceof Assembly))
			{
				startupView.setInput(((Assembly)selection.getFirstElement()).getConfiguration());
			}
			else if(selection.size() == 1 &&
					   (selection.getFirstElement() instanceof BaseElementList))
			{
				startupView.setInput(((BaseElementList)selection.getFirstElement()).getHwConfiguration());
			}
			else if(selection.size() == 1 &&
					   (selection.getFirstElement() instanceof PadList))
			{
				startupView.setInput(((PadList)selection.getFirstElement()).getHwConfiguration());
			}
			else if(selection.size() == 1 &&
					   (selection.getFirstElement() instanceof AntennaList))
			{
				startupView.setInput(((AntennaList)selection.getFirstElement()).getHwConfiguration());
			}
			else if(selection.size() == 1 &&
					   (selection.getFirstElement() instanceof FrontEndList))
			{
				startupView.setInput(((FrontEndList)selection.getFirstElement()).getHwConfiguration());
			}
			else if(selection.size() == 1 &&
					   (selection.getFirstElement() instanceof MasterClockList))
			{
				startupView.setInput(((MasterClockList)selection.getFirstElement()).getHwConfiguration());
			}
			else if(selection.size() == 1 &&
					   (selection.getFirstElement() instanceof CentralRackList))
			{
				startupView.setInput(((CentralRackList)selection.getFirstElement()).getHwConfiguration());
			}
			else if(selection.size() == 1 &&
					   (selection.getFirstElement() instanceof TypeList))
			{
				startupView.setInput(((TypeList)selection.getFirstElement()).getHwConfiguration());
			}
			else if(selection.size() == 1 &&
					   (selection.getFirstElement() instanceof AssemblyList))
			{
				startupView.setInput(((AssemblyList)selection.getFirstElement()).getHwConfiguration());
			}
			else {
				resetStartupViewForNewInput(startupView);
			}
		} else {
			resetStartupViewForNewInput(startupView);
		}
	}

	private void resetStartupViewForNewInput(StartupScenariosView startupView) {
		if(null != this.confTree.getInput()) {
			HwConfiguration[] currentConfigs = (HwConfiguration[])this.confTree.getInput();
			this.confTree.getSorter().sort(this.confTree, currentConfigs);
			if(currentConfigs.length > 0) {
				startupView.setInput(currentConfigs[0]);
			}
			else {
				startupView.setInput(null);
			}
		} else {
			startupView.setInput(null);
		}
	}

    private void makeActions()
	{
		IWorkbenchWindow win = getSite().getWorkbenchWindow();

		// the following actions are available _only_ to 'god' user (almamgr)
		if(GuiUtils.isGodUser())
		{
			// software actions
			exportConfigurationAction = new ExportConfigurationAction( win );
			importConfigurationAction = new ImportConfigurationAction( win, this );
			importConfigurationAction.addModelChangeListener(this);
			showNotificationChannelsAction = new ShowNotificationChannelsAction( win );
			showSwDeployAction = new ShowSwDeploymentAction( win );
			showAlarmCategoriesAction = new ShowAlarmCategoriesAction( win );
			showDCAAction = new ShowDefaultCanAddressesAction( win );

			// bulk actions
			bulkComponentChangesByHwConfigAction = new BulkComponentChangesByHwConfigurationAction(win);
			bulkBACIChangesByHwConfigAction      = new BulkBACIPropertiesChangesByHwConfigurationAction( win );
			bulkBACIChangesByAssemblyTypeAction  = new BulkBACIPropertiesChangesByAssemblyTypeAction( win );
			bulkBACIDeletionByHwConfigAction     = new BulkBACIPropertiesDeletionByHwConfigurationAction( win );
			bulkBACIDeletionByAssemblyTypeAction = new BulkBACIPropertiesDeletionByAssemblyTypeAction( win );
			bulkBACICreationByHwConfigAction     = new BulkBACIPropertiesCreationByHwConfigurationAction( win );
			bulkBACICreationByAssemblyTypeAction = new BulkBACIPropertiesCreationByAssemblyTypeAction( win );

			// hardware actions
			addAntennaAction = new AddAntennaAction( win, this );
			addFrontendAction = new AddFrontendAction( win, this );
			addWeatherStationAction = new AddWeatherStationAction( win );
			addPadAction = new AddPadAction( win, this );
			addHolographyTowerAction = new AddHolographyTowerAction( win );
			addAssemblyAction = new AddAssemblyAction( win, this );
			renameAssemblyAction = new RenameAssemblyAction( win, this );
			cloneAntennaAction = new CloneAntennaAction( win, this );
			copyAntennaAction = new CopyAntennaAction( win, this );
			cloneConfigurationAction = new CloneConfigurationAction( win, this );
			assignAntennaToPadAction = new AssignAntennaToPadAction( win, this );
			assignPadToHolographyTowerAction = new AssignPadToHolographyTowerAction( win, this );
			deleteAntennaAction = new DeleteAntennaAction( win );
			// TODO: deleteConfigurationAction = new DeleteConfigurationAction( win );
			deleteFrontendAction = new DeleteFrontendAction( win );
			deleteWeatherStationAction = new DeleteWeatherStationAction( win );

			// Hw edit actions
			editFrontEndAction = new EditFrontEndAction( win, this );
			editWeatherStationAction = new EditWeatherStationControllerAction( win, this );

			// Misc actions
			closeAntennaToPadAssignmentAction = new CloseAntennaToPadAssignmentAction(win, this);
		}

		// the following actions are partially available to any user (some portions
		// of these editors may be disabled for certain users)
		editHwConfigurationAction = new EditConfigurationAction( win, this );
		editAntennaAction = new EditAntennaAction( win, this );
		
		// the following actions, in their entirety, are available to any user
		unassignPadFromHolographyTowerAction = new UnassignPadFromHolographyTowerAction(win, this);
		editPadToHolographyTowerAction = new EditPadToHolographyTowerAction( win );
		editHolographyTowerAction = new EditHolographyTowerAction( win, this );
		editAssemblyAction = new EditAssemblyAction(win);
		editPadAction = new EditPadAction(win, this);
		editAntennaToPadAction = new EditAntennaToPadAction( win );
    	editXpDelaysAction = new EditXpDelaysAction( win );
    	editAntennaToPadAction = new EditAntennaToPadAction( win );
    	editPointingModelAction = new EditPointingModelAction(win, this);
    	editFocusModelAction = new EditFocusModelAction(win, this);
    	editDelayModelAction = new EditDelayModelAction(win, this);
    	editAcaCorrDelaysAction = new EditAcaCorrDelaysAction(win, this);
    	checkForDuplicatedWalshSequencesAction = new CheckForDuplicatedWalshSequencesAction(win, this);
    	checkForDuplicatedLoOffsetsAction = new CheckForDuplicatedLoOffsetsAction(win, this);
    	
    	// Double-click support for opening the editors
    	confTree.addDoubleClickListener(new IDoubleClickListener() {
			public void doubleClick(DoubleClickEvent event) {

				if( event.getSelection() instanceof IStructuredSelection ) 
				{
					IStructuredSelection selection = (IStructuredSelection)event.getSelection();
					if( selection.getFirstElement() instanceof FrontEnd && GuiUtils.isGodUser()) {
						editFrontEndAction.selectionChanged(ConfigurationsView.this, selection);
						editFrontEndAction.run();
					} else if( selection.getFirstElement() instanceof Antenna) {
						editAntennaAction.selectionChanged(ConfigurationsView.this, selection);
						editAntennaAction.run();
					} else if( selection.getFirstElement() instanceof AcaCorrDelays) {
						editAcaCorrDelaysAction.selectionChanged(ConfigurationsView.this, selection);
						editAcaCorrDelaysAction.run();
					} else if( selection.getFirstElement() instanceof Pad) {
						editPadAction.selectionChanged(ConfigurationsView.this, selection);
						editPadAction.run();
					} else if( selection.getFirstElement() instanceof WeatherStationController && GuiUtils.isGodUser()) {
						editWeatherStationAction.selectionChanged(ConfigurationsView.this, selection);
						editWeatherStationAction.run();
					} 
					else if( selection.getFirstElement() instanceof Assembly) 
					{
						editAssemblyAction.selectionChanged(ConfigurationsView.this, selection);
						editAssemblyAction.run();
					} else if( selection.getFirstElement() instanceof HwConfiguration ) {
						editHwConfigurationAction.selectionChanged(ConfigurationsView.this, selection);
						editHwConfigurationAction.run();
					} else if( selection.getFirstElement() instanceof AntennaToPad) {
						editAntennaToPadAction.selectionChanged(ConfigurationsView.this, selection);
						editAntennaToPadAction.run();
					} else if( selection.getFirstElement() instanceof HolographyTowerToPad) {
						editPadToHolographyTowerAction.selectionChanged(ConfigurationsView.this, selection);
						editPadToHolographyTowerAction.run();
					} else if( selection.getFirstElement() instanceof HolographyTower) {
						editHolographyTowerAction.selectionChanged(ConfigurationsView.this, selection);
						editHolographyTowerAction.run();
					}
					else if( selection.getFirstElement() instanceof PointingModel ) {
						editPointingModelAction.selectionChanged(ConfigurationsView.this, selection);
						editPointingModelAction.run();
					}
					else if( selection.getFirstElement() instanceof FocusModel ) {
						editFocusModelAction.selectionChanged(ConfigurationsView.this, selection);
						editFocusModelAction.run();
					}
					else if( selection.getFirstElement() instanceof DelayModel ) {
						editDelayModelAction.selectionChanged(ConfigurationsView.this, selection);
						editDelayModelAction.run();
					}
					else if( selection.getFirstElement() instanceof XpDelaysModel ) {
						editXpDelaysAction.selectionChanged(ConfigurationsView.this, selection);
						editXpDelaysAction.run();
					} 
					
					confTree.setSelection(selection, true);
					confTree.reveal(selection);
				}
			}
		});
	}

	private void makeContextMenu() {
		final MenuManager mgr = new MenuManager("configurationsPopup");
        mgr.setRemoveAllWhenShown(true);
        mgr.addMenuListener( new IMenuListener() {
			public void menuAboutToShow(IMenuManager manager) {
				fillContextMenu(manager);
			}
		});
        Control ctrl = confTree.getControl();
        ctrl.setMenu( mgr.createContextMenu( ctrl ));
        getSite().registerContextMenu(mgr, confTree);
	}

	private void fillContextMenu(IMenuManager manager) 
	{
		ISelection selection = confTree.getSelection();
        if( selection.isEmpty() ) {
        	manager.add( importConfigurationAction );
            return;
        }

        if( selection instanceof IStructuredSelection ) {
            IStructuredSelection sselection = (IStructuredSelection) selection;
            Object selNode = sselection.getFirstElement();

            if( selNode instanceof HwConfiguration && GuiUtils.isGodUser()) {
            	MenuManager newMenu = new MenuManager("&New", "new");
            	newMenu.add(addAntennaAction);
            	newMenu.add(addAssemblyAction);
            	newMenu.add(addFrontendAction);
            	newMenu.add(addHolographyTowerAction);
            	newMenu.add(addPadAction);
            	newMenu.add(addWeatherStationAction);
            	newMenu.add( new GroupMarker("additions") );

            	MenuManager showMenu = new MenuManager("&Show", "show");
            	showMenu.add(showSwDeployAction);
            	showMenu.add(showAlarmCategoriesAction);
            	showMenu.add(showDCAAction);
            	showMenu.add(showNotificationChannelsAction);

            	MenuManager bulkMenu = new MenuManager("&Bulk changes", "bulk");
            	bulkMenu.add(bulkComponentChangesByHwConfigAction);
            	bulkMenu.add(bulkBACICreationByHwConfigAction);
            	bulkMenu.add(bulkBACIChangesByHwConfigAction);
            	bulkMenu.add(bulkBACIDeletionByHwConfigAction);

            	manager.add( newMenu );
            	manager.add( showMenu );
            	manager.add( new Separator() );
            	manager.add( cloneConfigurationAction );
            	// TODO: manager.add( deleteConfigurationAction );
            	manager.add( new Separator() );
            	manager.add( bulkMenu );
            	manager.add( new Separator() );
            	manager.add( exportConfigurationAction );
            	manager.add( importConfigurationAction );
            	manager.add( new Separator() );
            	manager.add( checkForDuplicatedWalshSequencesAction );
            	manager.add( checkForDuplicatedLoOffsetsAction );
            } 
            else if (selNode instanceof HwConfiguration ) {
            	manager.add( checkForDuplicatedWalshSequencesAction );
            	manager.add( checkForDuplicatedLoOffsetsAction );
            }
            else if( selNode instanceof AntennaList && GuiUtils.isGodUser()) {
            	manager.add( addAntennaAction );
            } else if( selNode instanceof PadList && GuiUtils.isGodUser()) {
            	manager.add( addPadAction );
            } else if( selNode instanceof FrontEndList && GuiUtils.isGodUser()) {
            	manager.add( addFrontendAction );
            } else if( selNode instanceof HolographyTowerList && GuiUtils.isGodUser()) {
            	manager.add( addHolographyTowerAction );
            } else if( selNode instanceof WeatherStationList && GuiUtils.isGodUser()) {
            	manager.add( addWeatherStationAction );
            } 
            else if( selNode instanceof Antenna && GuiUtils.isGodUser()) {
            	manager.add( cloneAntennaAction );
            	manager.add(assignAntennaToPadAction);
            	manager.add(copyAntennaAction);
            	manager.add(deleteAntennaAction);
            } else if( selNode instanceof FrontEnd && GuiUtils.isGodUser()) {
            	manager.add( deleteFrontendAction );
            } else if( selNode instanceof Pad ) {
            	manager.add(assignPadToHolographyTowerAction);
            }
            else if( selNode instanceof WeatherStationController && GuiUtils.isGodUser()) {
            	manager.add( deleteWeatherStationAction );
            } 
            else if( selNode instanceof TypeList && GuiUtils.isGodUser()) {
            	manager.add(addAssemblyAction);
            }
            else if( selNode instanceof AssemblyList && GuiUtils.isGodUser()) {
            	manager.add(addAssemblyAction);
            	manager.add(new Separator());
            	MenuManager bulkMenu = new MenuManager("&Bulk changes", "bulk");
            	bulkMenu.add( bulkBACICreationByAssemblyTypeAction );
            	bulkMenu.add( bulkBACIChangesByAssemblyTypeAction );
            	bulkMenu.add( bulkBACIDeletionByAssemblyTypeAction );
            	manager.add(bulkMenu);
            } 
            else if( selNode instanceof Assembly && GuiUtils.isGodUser()) {
            	manager.add(editAssemblyAction);
            	manager.add(renameAssemblyAction);
            }
            else if( selNode instanceof AntennaToPad && GuiUtils.isGodUser()) {
            	manager.add( closeAntennaToPadAssignmentAction );
            }
            else if( selNode instanceof HolographyTowerToPad ) {
            	manager.add( unassignPadFromHolographyTowerAction );
            }
        }
	}
}
