/*******************************************************************************
  * ALMA - Atacama Large Millimeter Array
  * Copyright (c) NRAO - National Radio Astronomy Observatory, 2012
  * (in the framework of the ALMA collaboration).
  * All rights reserved.
  *
  * This library is free software; you can redistribute it and/or
  * modify it under the terms of the GNU Lesser General Public
  * License as published by the Free Software Foundation; either
  * version 2.1 of the License, or (at your option) any later version.
  *
  * This library is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  * Lesser General Public License for more details.
  *
  * You should have received a copy of the GNU Lesser General Public
  * License along with this library; if not, write to the Free Software
  * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 
*******************************************************************************/
package alma.obops.tmcdbgui.utils.conversation;

import java.lang.reflect.Method;
import java.util.List;

import alma.acs.tmcdb.Configuration;
import alma.acs.tmcdb.EventChannel;
import alma.obops.dam.config.TmcdbContextFactory;
import alma.obops.dam.tmcdb.service.EventChannelService;
import alma.obops.dam.tmcdb.service.SwConfigurationService;
import alma.obops.dam.utils.ConversationInterceptor;
import alma.obops.dam.utils.ConversationTokenProvider;
import alma.obops.dam.utils.ConversationTokenProviderAdapter;
import alma.obops.dam.utils.ConversationTokenProvider.ConversationToken;

/**
 * @author sharring
 *
 */
public class SwConfigurationConversationUtils 
{
	private static SwConfigurationConversationUtils singletonInstance;
	
	private SwConfigurationConversationUtils() 
	{
	}
	
	public static synchronized SwConfigurationConversationUtils getInstance()
	{
		if(null == singletonInstance)
		{
			singletonInstance = new SwConfigurationConversationUtils();
		}
		
		return singletonInstance;
	}
	
	public void saveOrUpdateSwConfiguration(Configuration configuration, boolean conversationCompleted) throws Exception 
	{
		Method methodToInvoke = SwConfigurationConversationUtils.class.getMethod("privateSaveOrUpdateSwConfiguration", Configuration.class, Boolean.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[2];
		args[0] = configuration;
		args[1] = conversationCompleted;
		conversationInterceptor.invoke(methodToInvoke, this, args);
	}
	
	public ConversationTokenProvider privateSaveOrUpdateSwConfiguration(Configuration configuration, Boolean conversationCompleted) 
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_PENDING);
		if(conversationCompleted) {
			retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		}
		SwConfigurationService service = TmcdbContextFactory.INSTANCE.getSwConfigurationService();
		configuration = (Configuration) service.read(configuration.getConfigurationId());
		service.update(configuration);
		return retVal;
	}
	
	public ConversationTokenProvider privateDeleteConfiguration(Configuration config, ConversationToken token) {
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(token);
		SwConfigurationService swconfigService = TmcdbContextFactory.INSTANCE.getSwConfigurationService();
		swconfigService.delete(config);
		return retVal;
	}
	
	public void hydrateManagers(Configuration swConfiguration) 
	throws Exception 
	{
		Method methodToInvoke = SwConfigurationConversationUtils.class.getMethod("privateHydrateManagers", Configuration.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[1];
		args[0] = swConfiguration;
		conversationInterceptor.invoke(methodToInvoke, this, args);
	}

	public ConversationTokenProvider privateHydrateManagers(Configuration config)
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		SwConfigurationService configService = TmcdbContextFactory.INSTANCE.getSwConfigurationService();
		configService.hydrateManagers(config);
		return retVal;
	}
	
	public void hydrateNotificationServiceMappings(Configuration swConfiguration) 
	throws Exception 
	{
		Method methodToInvoke = SwConfigurationConversationUtils.class.getMethod("privateHydrateNotificationServiceMappings", Configuration.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[1];
		args[0] = swConfiguration;
		conversationInterceptor.invoke(methodToInvoke, this, args);
	}
	
	public ConversationTokenProvider privateHydrateNotificationServiceMappings(Configuration config)
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		SwConfigurationService configService = TmcdbContextFactory.INSTANCE.getSwConfigurationService();
		configService.hydrateNotificationServiceMappings(config);
		return retVal;
	}
	
	public void hydrateEventChannels(Configuration configuration) 
	   throws Exception 
	{
		Method methodToInvoke = SwConfigurationConversationUtils.class.getMethod("privateHydrateEventChannels", Configuration.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[1];
		args[0] = configuration;
		conversationInterceptor.invoke(methodToInvoke, this, args);
	}
	
	public ConversationTokenProvider privateHydrateEventChannels(Configuration config)
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		SwConfigurationService configService = TmcdbContextFactory.INSTANCE.getSwConfigurationService();
		configService.hydrateEventChannels(config);
		return retVal;
	}
		
	public EventChannel[] findEventChannelsByRegEx(String regExp, Configuration config) throws Exception 
	{
		Method methodToInvoke = SwConfigurationConversationUtils.class.getMethod("privateFindEventChannelsByRegEx", String.class, Configuration.class, EventChannelsHolder.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		EventChannelsHolder holder = new EventChannelsHolder();
		Object[] args = new Object[3];
		args[0] = regExp;
		args[1] = config;
		args[2] = holder;
		conversationInterceptor.invoke(methodToInvoke, this, args);
		return holder.getEventChannels().toArray(new EventChannel[0]);
	}
	
	public ConversationTokenProvider privateFindEventChannelsByRegEx(String regEx, Configuration config, EventChannelsHolder holder)
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_PENDING);
		EventChannelService service = TmcdbContextFactory.INSTANCE.getEventChannelService();
		holder.setEventChannels( service.findEventChannelsByRegExInConfig(regEx, config));
		return retVal;
	}
	
	private class EventChannelsHolder 
	{
		private List<EventChannel> eventChannels;
		private List<EventChannel> getEventChannels() { return eventChannels; }
		private void setEventChannels(List<EventChannel> eventChannels ) { this.eventChannels = eventChannels; }
	}

	/**
	 * @param swConfiguration
	 */
	public void hydrateNetworkDevices(Configuration swConfiguration) 
	throws Exception 
	{
		Method methodToInvoke = SwConfigurationConversationUtils.class.getMethod("privateHydrateNetworkDevices", Configuration.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[1];
		args[0] = swConfiguration;
		conversationInterceptor.invoke(methodToInvoke, this, args);
	}
	
	public ConversationTokenProvider privateHydrateNetworkDevices(Configuration config)
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		SwConfigurationService configService = TmcdbContextFactory.INSTANCE.getSwConfigurationService();
		configService.hydrateNetworkDevices(config);
		return retVal;
	}

	/**
	 * @param criteria
	 * @param object
	 * @return
	 */
	public List<?> find(List<Object> searchCriteria, List<Object> orderCriteria) throws Exception 
	{
		Method methodToInvoke = SwConfigurationConversationUtils.class.getMethod("privateFind", List.class, List.class, ConfigurationListHolder.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[3];
		args[0] = searchCriteria;
		args[1] = orderCriteria;
		ConfigurationListHolder resultHolder = new ConfigurationListHolder();
		args[2] = resultHolder;
		conversationInterceptor.invoke(methodToInvoke, this, args);
		return resultHolder.getConfigurations();
	}

	@SuppressWarnings("unchecked")
	public ConversationTokenProvider privateFind(List searchCriteria, List orderCriteria, ConfigurationListHolder resultHolder)
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		SwConfigurationService service = TmcdbContextFactory.INSTANCE.getSwConfigurationService();
		resultHolder.setConfigurations((List<Configuration>) service.find(searchCriteria, orderCriteria));
		return retVal;
	}
	
	private class ConfigurationListHolder 
	{
		private List<Configuration> configs;
		public List<Configuration> getConfigurations() { return this.configs; }
		public void setConfigurations(List<Configuration> configs) { this.configs = configs; }
	}
	
}
