/*******************************************************************************
  * ALMA - Atacama Large Millimeter Array
  * Copyright (c) NRAO - National Radio Astronomy Observatory, 2012
  * (in the framework of the ALMA collaboration).
  * All rights reserved.
  *
  * This library is free software; you can redistribute it and/or
  * modify it under the terms of the GNU Lesser General Public
  * License as published by the Free Software Foundation; either
  * version 2.1 of the License, or (at your option) any later version.
  *
  * This library is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  * Lesser General Public License for more details.
  *
  * You should have received a copy of the GNU Lesser General Public
  * License along with this library; if not, write to the Free Software
  * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 
*******************************************************************************/
package alma.obops.tmcdbgui.utils.conversation;

import java.lang.reflect.Method;
import java.util.HashSet;
import java.util.Set;

import alma.obops.dam.config.TmcdbContextFactory;
import alma.obops.dam.tmcdb.service.AssemblyStartupService;
import alma.obops.dam.tmcdb.service.AssemblyTypeService;
import alma.obops.dam.tmcdb.service.BaseElementStartupService;
import alma.obops.dam.tmcdb.service.ComponentTypeService;
import alma.obops.dam.tmcdb.service.LruTypeService;
import alma.obops.dam.tmcdb.service.StartupScenarioService;
import alma.obops.dam.utils.ConversationInterceptor;
import alma.obops.dam.utils.ConversationTokenProvider;
import alma.obops.dam.utils.ConversationTokenProviderAdapter;
import alma.obops.dam.utils.ConversationTokenProvider.ConversationToken;
import alma.obops.tmcdbgui.utils.DomainObjectUtils;
import alma.tmcdb.domain.AssemblyStartup;
import alma.tmcdb.domain.AssemblyType;
import alma.tmcdb.domain.BaseElement;
import alma.tmcdb.domain.BaseElementStartup;
import alma.tmcdb.domain.BaseElementStartupType;
import alma.tmcdb.domain.BaseElementType;
import alma.tmcdb.domain.HwConfiguration;
import alma.tmcdb.domain.LruType;
import alma.tmcdb.domain.StartupScenario;

/**
 * @author sharring
 *
 */
public class StartupScenarioConversationUtils 
{
	private static StartupScenarioConversationUtils singletonInstance;

	private StartupScenarioConversationUtils() 
	{
	}

	public static synchronized StartupScenarioConversationUtils getInstance()
	{
		if(null == singletonInstance)
		{
			singletonInstance = new StartupScenarioConversationUtils();
		}

		return singletonInstance;
	}
	
	public BaseElementStartup addBaseElementToStartupScenario(BaseElement elementToAdd, StartupScenario startupToWhichToAdd) 
	  throws Exception
	{
		BaseElementStartup retVal = null;
		
		Method methodToInvoke = StartupScenarioConversationUtils.class.getMethod("privateAddBaseElementToStartupScenario", 
				BaseElement.class, StartupScenario.class, BaseElementStartupHolder.class);
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[3];
		args[0] = elementToAdd;
		args[1] = startupToWhichToAdd;
		BaseElementStartupHolder resultHolder = new BaseElementStartupHolder();
		args[2] = resultHolder;
		conversationInterceptor.invoke(methodToInvoke, this, args);
      retVal = resultHolder.getBaseElementStartup();
      
      return retVal;
	}
	
	public ConversationTokenProvider privateAddBaseElementToStartupScenario(BaseElement elementToAdd, 
			StartupScenario startupToWhichToAdd, BaseElementStartupHolder resultHolder)
	{
		BaseElementStartup result = null;
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		StartupScenarioService s =  TmcdbContextFactory.INSTANCE.getStartupScenarioService();
		
		// Look for that BaseElement, do we have it already?
		for( BaseElementStartup bes : startupToWhichToAdd.getBaseElementStartups() )
		{
			BaseElement be = bes.getBaseElement();
			if( null != be && be.equals( elementToAdd )) {
				resultHolder.setBaseElementStartup(null);
				return retVal;   // YES, we do have it, nothing to do
			}
		}

		// NO, we don't have it and can add it
		BaseElementStartup bes = new BaseElementStartup();
		bes.setBaseElement(elementToAdd);
		bes.setType(DomainObjectUtils.getBaseElementStartupTypeFromBaseElementType(elementToAdd.getType()));
		bes.setStartup(startupToWhichToAdd);
		startupToWhichToAdd.addBaseElementStartup(bes);
		bes.setGeneric("false");
		bes.setSimulated(false);
		bes.setParent(null);
		
		// have the service create the object
		s.create(bes);
		result = bes;
		
		// special logic to handle antenna & centralrack in a special way
		switch(elementToAdd.getType()) {
		case Antenna:
			// for the antenna, we always also add a frontend baseelementstartup nested inside the antenna
			BaseElementStartup frontendBaseElementStartup = new BaseElementStartup(BaseElementStartupType.FrontEnd);
			frontendBaseElementStartup.setSimulated(false);
			frontendBaseElementStartup.setParent(result);
			result.getChildren().add(frontendBaseElementStartup);
			s.update(result);
			break;
		case CentralLO:
			// for the centralrack, we also add 6 photonicreference baseelementstartups nested inside the centralrack
			BaseElementStartup photonicRef1 = new BaseElementStartup(BaseElementStartupType.PhotonicReference1);
			photonicRef1.setParent(result);
			photonicRef1.setSimulated(false);
			result.getChildren().add(photonicRef1);

			BaseElementStartup photonicRef2 = new BaseElementStartup(BaseElementStartupType.PhotonicReference2);
			photonicRef2.setParent(result);
			photonicRef2.setSimulated(false);
			result.getChildren().add(photonicRef2);
			
			BaseElementStartup photonicRef3 = new BaseElementStartup(BaseElementStartupType.PhotonicReference3);
			photonicRef3.setParent(result);
			photonicRef3.setSimulated(false);
			result.getChildren().add(photonicRef3);
			
			BaseElementStartup photonicRef4 = new BaseElementStartup(BaseElementStartupType.PhotonicReference4);
			photonicRef4.setParent(result);
			photonicRef4.setSimulated(false);
			result.getChildren().add(photonicRef4);
			
			BaseElementStartup photonicRef5 = new BaseElementStartup(BaseElementStartupType.PhotonicReference5);
			photonicRef5.setParent(result);
			photonicRef5.setSimulated(false);
			result.getChildren().add(photonicRef5);
			
			BaseElementStartup photonicRef6 = new BaseElementStartup(BaseElementStartupType.PhotonicReference6);
			photonicRef6.setParent(result);
			photonicRef6.setSimulated(false);
			result.getChildren().add(photonicRef6);
			
			s.update(result);
			break;
		default:
			// nothing special for other baseelement types
			break;
		}
		s.update(startupToWhichToAdd);
		resultHolder.setBaseElementStartup(result);
		return retVal;
	}
	
	public static BaseElementType getBaseElementTypeFromBaseElementStartupType(BaseElementStartupType bes)
	{
		BaseElementType retVal = null;
		switch(bes)
		{
		case Antenna:
			retVal = BaseElementType.Antenna;
			break;
		case PhotonicReference1:
		case PhotonicReference2:
		case PhotonicReference3:
		case PhotonicReference4:
		case PhotonicReference5:
		case PhotonicReference6:
			retVal = BaseElementType.PhotonicReference;
			break;
		case Pad:
			retVal = BaseElementType.Pad;
			break;
		case AOSTiming:
			retVal = BaseElementType.AOSTiming;
			break;
		case CentralLO:
			retVal = BaseElementType.CentralLO;
			break;
		case FrontEnd:
			retVal = BaseElementType.FrontEnd;
			break;
		case WeatherStationController:
			retVal = BaseElementType.WeatherStationController;
			break;
		case Array:
			retVal = BaseElementType.Array;
			break;
		case HolographyTower:
			retVal = BaseElementType.HolographyTower;
			break;
		}
		return retVal;
	}
	
	private class BaseElementStartupHolder {
	   private BaseElementStartup baseElementStartup;
	   
	   public BaseElementStartup getBaseElementStartup() {
		   return this.baseElementStartup;
	   }
	   
	   public void setBaseElementStartup(BaseElementStartup beStartup) {
		   this.baseElementStartup = beStartup;
	   }
	}
	
	public StartupScenario findStartupScenario(HwConfiguration configuration, String name) 
	  throws Exception 
	{
		Method methodToInvoke = StartupScenarioConversationUtils.class.getMethod("privateFindStartupScenario", 
				HwConfiguration.class, String.class, StartupScenarioHolder.class);
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[3];
		args[0] = configuration;
		args[1] = name;
		StartupScenarioHolder resultHolder = new StartupScenarioHolder();
		args[2] = resultHolder;
		conversationInterceptor.invoke(methodToInvoke, this, args);
		return resultHolder.getStartupScenario();
	}
	
	public ConversationTokenProvider privateFindStartupScenario(HwConfiguration configuration, 
			String name, StartupScenarioHolder resultHolder)
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		resultHolder.setStartupScenario(null);
		StartupScenarioService service = TmcdbContextFactory.INSTANCE.getStartupScenarioService();
		resultHolder.setStartupScenario(service.findByNameWithinConfiguration(configuration, name));
		return retVal;
	}
	
	private class StartupScenarioHolder
	{
		private StartupScenario startupScenario;
		public StartupScenario getStartupScenario() { return startupScenario; }
		public void setStartupScenario(StartupScenario scenario) { this.startupScenario = scenario; }
	}

	public void cloneStartupScenario(StartupScenario scenario, String name) 
	   throws Exception 
	{
		Method methodToInvoke = StartupScenarioConversationUtils.class.getMethod("privateCloneStartupScenario", 
			 StartupScenario.class, String.class);
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[2];
		args[0] = scenario;
		args[1] = name;
		conversationInterceptor.invoke(methodToInvoke, this, args);
	}
	
	public ConversationTokenProvider privateCloneStartupScenario(StartupScenario scenarioToClone, String clonedScenarioName)
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		StartupScenarioService startupService = TmcdbContextFactory.INSTANCE.getStartupScenarioService();
		startupService.cloneStartupScenario(scenarioToClone, clonedScenarioName);
		return retVal;	
	}
	
	public BaseElementStartup addFrontEndStartupToAntennaStartup(BaseElementStartup antennaStartup) throws Exception 
	{
		Method methodToInvoke = StartupScenarioConversationUtils.class.getMethod("privateAddFrontEndStartupToAntennaStartup", 
				BaseElementStartup.class, BaseElementStartupHolder.class);
		
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[2];
		args[0] = antennaStartup;
		BaseElementStartupHolder resultHolder = new BaseElementStartupHolder();
		args[1] = resultHolder;
		conversationInterceptor.invoke(methodToInvoke, this, args);
    	BaseElementStartup retVal = resultHolder.getBaseElementStartup(); 
    	return retVal;
	}
	
	public ConversationTokenProvider privateAddFrontEndStartupToAntennaStartup(
			BaseElementStartup antennaStartup, BaseElementStartupHolder resultHolder)
	{			
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		if(!antennaStartup.getType().equals(BaseElementStartupType.Antenna)) {
			retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_FAILED);
		} 
		else 
		{
			BaseElementStartupService baseElementStartupService = TmcdbContextFactory.INSTANCE.getBaseElementStartupService();	   
			baseElementStartupService.hydrateToBaseElementChildren(antennaStartup);
		
			boolean alreadyHasFrontEndStartup = false;
			for(BaseElementStartup childStartup : antennaStartup.getChildren()) {
				if(childStartup.getType().equals(BaseElementStartupType.FrontEnd)) {
					alreadyHasFrontEndStartup = true;
					break;
				}
			}
			if(!alreadyHasFrontEndStartup) 
			{
				BaseElementStartup toAdd = new BaseElementStartup(BaseElementStartupType.FrontEnd);
				//toAdd.setStartup(antennaStartup.getStartup());
				toAdd.setParent(antennaStartup);
				toAdd.setGeneric("true");
				antennaStartup.getChildren().add(toAdd);
				resultHolder.setBaseElementStartup(toAdd);
				baseElementStartupService.update(antennaStartup);
			}
		}
    	return retVal;
	}
	
	public void hydrateBaseElementStartupToChildren(BaseElementStartup baseElementStartup) throws Exception 
	{
		Method methodToInvoke = StartupScenarioConversationUtils.class.getMethod("privateHydrateBaseElementStartupToChildren", BaseElementStartup.class);
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[1];
		args[0] = baseElementStartup;
		conversationInterceptor.invoke(methodToInvoke, this, args);
	}
	
	public ConversationTokenProvider privateHydrateBaseElementStartupToChildren(BaseElementStartup beStartup)
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		BaseElementStartupService baseElementStartupService = TmcdbContextFactory.INSTANCE.getBaseElementStartupService();	   
		baseElementStartupService.hydrateToBaseElementChildren(beStartup);
		return retVal;
	}

	public LruType[] findLruTypesByBaseElementStartupType(BaseElementStartupType type) throws Exception
	{
		Method methodToInvoke = StartupScenarioConversationUtils.class.getMethod("privateFindLruTypesByBaseElementStartupType", 
				BaseElementStartupType.class, LruTypesHolder.class);
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[2];
		args[0] = type;
		LruTypesHolder resultHolder = new LruTypesHolder();
		args[1] = resultHolder;
		conversationInterceptor.invoke(methodToInvoke, this, args);
		return resultHolder.getLruTypes();
	}
	
	public ConversationTokenProvider privateFindLruTypesByBaseElementStartupType(BaseElementStartupType type, LruTypesHolder resultHolder)
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		LruTypeService service = TmcdbContextFactory.INSTANCE.getLruTypeService();	   
		resultHolder.setLruTypes(service.findByBaseElementStartupType(type));
		
		// now do some hydration, for convenience... should we make 
		// the caller do this explicitly as a separate step?
		for(LruType lruType : resultHolder.lruTypes) {
			service.hydrateToAssemblyTypes(lruType);
		}
		return retVal;
	}
	
	private class LruTypesHolder
	{
		private LruType[] lruTypes;
		public LruType[] getLruTypes() { return lruTypes; }
		public void setLruTypes(LruType[] types) { this.lruTypes = types; }
	}

	public void hydrateLruType(LruType lruType) throws Exception 
	{
		Method methodToInvoke = StartupScenarioConversationUtils.class.getMethod("privateHydrateLruType", LruType.class);
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[1];
		args[0] = lruType;
		conversationInterceptor.invoke(methodToInvoke, this, args);
	}
	
	public ConversationTokenProvider privateHydrateLruType(LruType lruType)
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		LruTypeService lruTypeService = TmcdbContextFactory.INSTANCE.getLruTypeService();
		lruTypeService.hydrate(lruType);
		AssemblyTypeService assemblyTypeService = TmcdbContextFactory.INSTANCE.getAssemblyTypeService();
		ComponentTypeService componentTypeService = TmcdbContextFactory.INSTANCE.getComponentTypeService();
		for(AssemblyType assemblyType : lruType.getAssemblyTypes()) {
			assemblyTypeService.hydrate(assemblyType);
			componentTypeService.hydrate(assemblyType.getComponentType());
		}
		return retVal;
	}
	
	public BaseElementStartup addPhotonicReferenceToCentralRackStartup(BaseElement dragged, 
			BaseElementStartup centralRackStartup) throws Exception 
	{
		Method methodToInvoke = StartupScenarioConversationUtils.class.getMethod("privateAddPhotonicReferenceStartupToCentralRackStartup", 
				BaseElement.class, BaseElementStartup.class, BaseElementStartupHolder.class);
		
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[3];
		args[0] = dragged;
		args[1] = centralRackStartup;
		BaseElementStartupHolder resultHolder = new BaseElementStartupHolder();
		args[2] = resultHolder;
		conversationInterceptor.invoke(methodToInvoke, this, args);
    	BaseElementStartup retVal = resultHolder.getBaseElementStartup(); 
    	return retVal;
	}
	
	public ConversationTokenProvider privateAddPhotonicReferenceStartupToCentralRackStartup(BaseElement dragged, 
			BaseElementStartup centralRackStartup, BaseElementStartupHolder resultHolder)
	{			
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		if(!centralRackStartup.getType().equals(BaseElementStartupType.CentralLO)) {
			retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_FAILED);
		} 
		else 
		{
			BaseElementStartupService baseElementStartupService = TmcdbContextFactory.INSTANCE.getBaseElementStartupService();	   
			baseElementStartupService.hydrateToBaseElementChildren(centralRackStartup);
		
			boolean alreadyHasPhotonicRefStartup = false;
			BaseElementStartupType bestartuptype = BaseElementStartupType.valueOf(dragged.getName());
			for(BaseElementStartup childStartup : centralRackStartup.getChildren()) {
				if(childStartup.getType().equals(bestartuptype)) {
					alreadyHasPhotonicRefStartup = true;
					break;
				}
			}
			if(!alreadyHasPhotonicRefStartup) 
			{
				BaseElementStartup toAdd = new BaseElementStartup(bestartuptype);
				//toAdd.setStartup(centralRackStartup.getStartup());
				toAdd.setParent(centralRackStartup);
				toAdd.setGeneric("true");
				toAdd.setSimulated(false);
				centralRackStartup.getChildren().add(toAdd);
				resultHolder.setBaseElementStartup(toAdd);
				baseElementStartupService.update(centralRackStartup);
			}
		}
    	return retVal;
	}
	
	public void removeBaseElementFromStartupScenario(BaseElementStartup beStartup, StartupScenario startup)
	  throws Exception
	{
		Method methodToInvoke = StartupScenarioConversationUtils.class.getMethod("privateRemoveBaseElementFromStartupScenario", 
				BaseElementStartup.class, StartupScenario.class);
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[2];
		args[0] = beStartup;
		args[1] = startup;
		conversationInterceptor.invoke(methodToInvoke, this, args);
	}
	
	public ConversationTokenProvider privateRemoveBaseElementFromStartupScenario(BaseElementStartup baseElementStartupToRemove, StartupScenario startup)
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		
		// first delete the baseelementstartup itself
		BaseElementStartupService s1 =  TmcdbContextFactory.INSTANCE.getBaseElementStartupService();
		boolean removed = startup.getBaseElementStartups().remove(baseElementStartupToRemove);
		s1.delete(baseElementStartupToRemove);
		
		// if it's a nested baseelementstartup, we must remove it from the parent's children collection
		// else hibernate cascades will attempt to resave it!
		if(baseElementStartupToRemove.getParent() != null) 
		{
			BaseElementStartup besParent = baseElementStartupToRemove.getParent();
			besParent.getChildren().remove(baseElementStartupToRemove);
		}
		
		// BEGIN HACK
		if(!removed)
		{
			Set<BaseElementStartup> startups = new HashSet<BaseElementStartup>();
			for(BaseElementStartup bes : startup.getBaseElementStartups())
			{
				if(!bes.getId().equals(baseElementStartupToRemove.getId()))
				{
					startups.add(bes);
				}
			}
			startup.setBaseElementStartups(startups);
		}
		// END HACK

		// and remove it from the collection of the startup scenario as well
		StartupScenarioService s2 = TmcdbContextFactory.INSTANCE.getStartupScenarioService();
		s2.update(startup);
		
		return retVal;
	}
	
	public ConversationTokenProvider privateDeleteAssemblyStartup(AssemblyStartup assStartup, ConversationToken token) {
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(token);
		AssemblyStartupService startupService = TmcdbContextFactory.INSTANCE.getAssemblyStartupService();
		startupService.delete(assStartup);
		return retVal;
	}
	
	public ConversationTokenProvider privateDeleteStartupScenario(StartupScenario startup, ConversationToken token) {
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(token);
		StartupScenarioService startupScenarioService = TmcdbContextFactory.INSTANCE.getStartupScenarioService();
		startupScenarioService.delete(startup);
		return retVal;
	}
	
	private void commonDeleteBaseElementStartup(BaseElementStartup beStartup)
	{
		StartupScenario startup = beStartup.getStartup() != null ? beStartup.getStartup() : beStartup.getParent().getStartup();
		boolean removed = false;
		if(null != beStartup.getParent()) 
		{
			removed = beStartup.getParent().getChildren().remove(beStartup);
			startup.getBaseElementStartups().remove(beStartup);

			// begin hack
			if(!removed) 
			{
				Set<BaseElementStartup> newSet = new HashSet<BaseElementStartup>();
				for(BaseElementStartup bes: beStartup.getParent().getChildren()) {
					if(bes.hashCode() != beStartup.hashCode() ||  !bes.equals(beStartup)) 
					{
						newSet.add(bes);
					}
				}
				int previous = beStartup.getParent().getChildren().size();
				beStartup.getParent().setChildren(newSet);
				int after = beStartup.getParent().getChildren().size();
				removed = previous != after;
			}
			// end hack
			else {
				beStartup.setParent(null);
				beStartup.setStartup(null);
			}
		}
		else 
		{
			removed = startup.getBaseElementStartups().remove(beStartup);

			// begin hack
			if(!removed) 
			{
				Set<BaseElementStartup> newSet = new HashSet<BaseElementStartup>();
				for(BaseElementStartup bes: startup.getBaseElementStartups()) {
					if(bes.hashCode() != beStartup.hashCode() ||  !bes.equals(beStartup)) 
					{
						newSet.add(bes);
					}
				}
				int previous = startup.getBaseElementStartups().size();
				startup.setBaseElementStartups(newSet);
				int after = startup.getBaseElementStartups().size();
				removed = previous != after;
			}
			// end hack
		}

		if(!removed) {
			// should never happen!
			throw new IllegalStateException("BaseElementStartup could not be removed from parent collection.");
		}
	}
	

	/**
	 * This method deletes a BaseElementStartup without immediately completing the conversation; it also
	 * uses the BaseElementStartupService to delete, rather than updating the owning StartupScenario;
	 * this prevents the update of the owning StartupScenario from cascading and attempting to 're-save' 
	 * a deleted BaseElementStartup, which causes hibernate exceptions. This problem arose when deleting
	 * StartupScenarios. The fix was to differentiate the "normal" deletion of a BaseElementStartup from the
	 * "special" case deletion of a BaseElementStartup during the deletion of a full StartupScenario. This
	 * method is used during deletion of a full StartupScenario.
	 *  
	 * @param beStartup the BaseElementStartup being deleted
	 * @param token a token indicating whether
	 * the conversation should be ended or continued.
	 * @return a ConversationToken indicating whether the conversation should be continued or completed.
	 */
	public ConversationTokenProvider privateDeleteBaseElementStartup(BaseElementStartup beStartup, ConversationToken token) 
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(token);
		commonDeleteBaseElementStartup(beStartup);
		BaseElementStartupService service = TmcdbContextFactory.INSTANCE.getBaseElementStartupService();
		service.delete(beStartup);
		return retVal;
	}
	
	/**
	 * This method deletes a BaseElementStartup and immediately completes the conversation; it 
	 * uses the StartupScenarioService to update the owning StartupScenario and do the delete via a cascade, 
	 * rather than calling the BaseElementStartupService delete method directly.
	 * This method is used during deletion of a simple/single BaseElementStartup and is not used
	 * when deleting a complete StartupScenario.
	 *  
	 * @param beStartup the BaseElementStartup being deleted
	 * @return a ConversationToken indicating that the conversation should be completed.
	 */
	public ConversationTokenProvider privateDeleteBaseElementStartup(BaseElementStartup beStartup) 
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		commonDeleteBaseElementStartup(beStartup);
		StartupScenarioService startupService = TmcdbContextFactory.INSTANCE.getStartupScenarioService();
		StartupScenario startup = beStartup.getStartup() != null ? beStartup.getStartup() : beStartup.getParent().getStartup();
		startupService.update(startup);
		return retVal;
	}
	
	public void hydrateBaseElementStartups(StartupScenario startup) 
	   throws Exception 
	{
		Method methodToInvoke = StartupScenarioConversationUtils.class.getMethod("privateHydrateBaseElementStartups", StartupScenario.class);
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[1];
		args[0] = startup;
		conversationInterceptor.invoke(methodToInvoke, this, args);
	}
	
	public ConversationTokenProvider privateHydrateBaseElementStartups(StartupScenario scenario) 
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		StartupScenarioService startupService =  TmcdbContextFactory.INSTANCE.getStartupScenarioService();
		startupService.hydrateBaseElementStartups(scenario);	
		return retVal;
	}
	
	public void hydrateAssemblyStartups(StartupScenario startup) 
	   throws Exception 
	{
		Method methodToInvoke = StartupScenarioConversationUtils.class.getMethod("privateHydrateAssemblyStartups", StartupScenario.class);
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[1];
		args[0] = startup;
		conversationInterceptor.invoke(methodToInvoke, this, args);
	}
	
	public ConversationTokenProvider privateHydrateAssemblyStartups(StartupScenario startup) 
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		StartupScenarioService startupService =  TmcdbContextFactory.INSTANCE.getStartupScenarioService();
		startupService.hydrateAssemblyStartups(startup);	
		return retVal;
	}

	public void saveOrUpdateBaseElementStartup(BaseElementStartup baseElementStartup, ConversationToken token) throws Exception {
		Method methodToInvoke = StartupScenarioConversationUtils.class.getMethod("privateSaveOrUpdateBaseElementStartup", BaseElementStartup.class, ConversationToken.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[2];
		args[0] = baseElementStartup;
		args[1] = token;
		conversationInterceptor.invoke(methodToInvoke, this, args);
	}
	
	
	public ConversationTokenProvider privateSaveOrUpdateBaseElementStartup(BaseElementStartup baseElementStartup, ConversationToken token) {
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(token);
		BaseElementStartupService beService = TmcdbContextFactory.INSTANCE.getBaseElementStartupService();
		beService.update(baseElementStartup);
		return retVal;
	}
	
	public void saveOrUpdateStartupScenario(StartupScenario startup) throws Exception {
		Method methodToInvoke = StartupScenarioConversationUtils.class.getMethod("privateSaveOrUpdateStartupScenario", StartupScenario.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[1];
		args[0] = startup;
		conversationInterceptor.invoke(methodToInvoke, this, args);
	}
	
	
	public ConversationTokenProvider privateSaveOrUpdateStartupScenario(StartupScenario startup) {
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		StartupScenarioService service = TmcdbContextFactory.INSTANCE.getStartupScenarioService();
		service.update(startup);
		return retVal;
	}
	
	public void saveOrUpdateAssemblyStartup(AssemblyStartup assemblyStartup) throws Exception 
	{
		Method methodToInvoke = StartupScenarioConversationUtils.class.getMethod("privateSaveOrUpdateAssemblyStartup", AssemblyStartup.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[1];
		args[0] = assemblyStartup;
		conversationInterceptor.invoke(methodToInvoke, this, args);
	}

	public ConversationTokenProvider privateSaveOrUpdateAssemblyStartup(AssemblyStartup assemblyStartup) 
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		AssemblyStartupService service = TmcdbContextFactory.INSTANCE.getAssemblyStartupService();
		service.update(assemblyStartup);
		return retVal;
	}
}
