/*******************************************************************************
  * ALMA - Atacama Large Millimeter Array
  * Copyright (c) NRAO - National Radio Astronomy Observatory, 2012
  * (in the framework of the ALMA collaboration).
  * All rights reserved.
  *
  * This library is free software; you can redistribute it and/or
  * modify it under the terms of the GNU Lesser General Public
  * License as published by the Free Software Foundation; either
  * version 2.1 of the License, or (at your option) any later version.
  *
  * This library is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  * Lesser General Public License for more details.
  *
  * You should have received a copy of the GNU Lesser General Public
  * License along with this library; if not, write to the Free Software
  * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 
*******************************************************************************/
package alma.obops.tmcdbgui.utils.conversation;

import java.lang.reflect.Method;
import java.util.List;

import alma.obops.dam.config.TmcdbContextFactory;
import alma.obops.dam.tmcdb.service.PointingModelService;
import alma.obops.dam.utils.ConversationInterceptor;
import alma.obops.dam.utils.ConversationTokenProvider;
import alma.obops.dam.utils.ConversationTokenProviderAdapter;
import alma.obops.dam.utils.ConversationTokenProvider.ConversationToken;
import alma.tmcdb.domain.PointingModel;
import alma.tmcdb.domain.PointingModelCoeff;
import alma.tmcdb.history.HistoryRecord;

/**
 * @author sharring
 *
 */
public class PointingModelConversationUtils 
{
	private static PointingModelConversationUtils singletonInstance;

	private PointingModelConversationUtils() 
	{
	}

	public static synchronized PointingModelConversationUtils getInstance()
	{
		if(null == singletonInstance)
		{
			singletonInstance = new PointingModelConversationUtils();
		}

		return singletonInstance;
	}	
	
	public ConversationTokenProvider privateDeletePointingModelCoeff(PointingModelCoeff coeff, ConversationToken token) {
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(token);
		PointingModelService pointingModelService = TmcdbContextFactory.INSTANCE.getPointingModelService();
		pointingModelService.delete(coeff);
		return retVal;
	}
	
	public void saveOrUpdatePointingModel(PointingModel pm) throws Exception 
	{
		saveOrUpdatePointingModel(pm, ConversationToken.CONVERSATION_COMPLETED);
	}

	/**
	 * @param pointingModel
	 * @param conversationPending
	 */
	public void saveOrUpdatePointingModel(PointingModel pm, ConversationToken token) throws Exception 
	{
		Method methodToInvoke = PointingModelConversationUtils.class.getMethod("privateSaveOrUpdatePointingModel", PointingModel.class, ConversationToken.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[2];
		args[0] = pm;
		args[1] = token;
		// note: at present, the tmcdb database only has a single constraint that is deferrable (baseelsaltkey); 
		// so that invoking this method with deferred constraints is ultimately equivalent to invoking it w/ immediate
		// constraints. However, it might be nice to defer additional constraints and use it here, just for odd situations
		// such as might occur if someone deletes, then re-adds the same item in a single edit. leaving this call
		// here, in case we do indeed change the db constraints to be deferrable; SLH (02-03-2012)
		conversationInterceptor.invokeWithDeferredConstraints(methodToInvoke, this, args);
	}

	public ConversationTokenProvider privateSaveOrUpdatePointingModel(PointingModel pm, ConversationToken token) 
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(token);
		PointingModelService pmService = TmcdbContextFactory.INSTANCE.getPointingModelService();
		pmService.update(pm);
		return retVal;
	}
	
	public List<HistoryRecord> getPointingModelHistory(PointingModel pointingModel) 
	   throws Exception 
	{
		Method methodToInvoke = PointingModelConversationUtils.class.getMethod("privateGetPointingModelHistory", PointingModel.class, HistoryRecordListHolder.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		HistoryRecordListHolder holder = new HistoryRecordListHolder();
		Object[] args = new Object[2];
		args[0] = pointingModel;
		args[1] = holder;
		conversationInterceptor.invoke(methodToInvoke, this, args);
		return holder.getHistoryRecords();
		
	}
	
	public ConversationTokenProvider privateGetPointingModelHistory(PointingModel pm, HistoryRecordListHolder resultHolder) 
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		PointingModelService service = TmcdbContextFactory.INSTANCE.getPointingModelService();
		List<HistoryRecord> results = service.getHistory(pm);
		resultHolder.setHistoryRecords(results);
		return retVal;
	}

	public PointingModel getHistoricalPointingModel(PointingModel pointingModel, HistoryRecord record) throws Exception 
	{
		Method methodToInvoke = PointingModelConversationUtils.class.getMethod("privateGetHistoricalPointingModel", PointingModel.class, HistoryRecord.class, PointingModelHolder.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		PointingModelHolder holder = new PointingModelHolder();
		Object[] args = new Object[3];
		args[0] = pointingModel;
		args[1] = record;
		args[2] = holder;
		conversationInterceptor.invoke(methodToInvoke, this, args);
		PointingModel retVal = holder.getPointingModel();
		retVal.setAntenna(pointingModel.getAntenna());
		return retVal;
	}
	
	public ConversationTokenProvider privateGetHistoricalPointingModel(PointingModel pm, HistoryRecord record, PointingModelHolder resultHolder) 
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		PointingModelService service = TmcdbContextFactory.INSTANCE.getPointingModelService();
		PointingModel historicalPm = service.getHistoricalPointingModel(pm, record.getVersion());
		resultHolder.setPointingModel(historicalPm);
		return retVal;
	}
	
	private class PointingModelHolder
	{
		private PointingModel pointingModel;
		
		public PointingModel getPointingModel() {
			return this.pointingModel;
		}
		
		public void setPointingModel(PointingModel pm) {
			this.pointingModel = pm;
		}
	}
	
	public boolean preparePointingModelSave(PointingModel pointingModel, String who, String description) throws Exception 
	{
		Method methodToInvoke = PointingModelConversationUtils.class.getMethod("privatePreparePointingModelSave", 
				PointingModel.class, String.class, String.class, BooleanHolder.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		BooleanHolder resultholder = new BooleanHolder();
		Object[] args = new Object[4];
		args[0] = pointingModel;
		args[1] = who;
		args[2] = description;
		args[3] = resultholder;
		conversationInterceptor.invoke(methodToInvoke, this, args);
		boolean retVal = resultholder.getBooleanValue();
		return retVal;
	}
	
	public ConversationTokenProvider privatePreparePointingModelSave(PointingModel pm, 
			String who, String description, BooleanHolder resultHolder) 
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		PointingModelService service = TmcdbContextFactory.INSTANCE.getPointingModelService();
		boolean successful = service.prepareSave(pm, who, description);
		resultHolder.setBooleanValue(successful);
		return retVal;
	}

	public void endPointingModelSave(PointingModel pointingModel) throws Exception 
	{
		Method methodToInvoke = PointingModelConversationUtils.class.getMethod("privateEndPointingModelSave", 
				PointingModel.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[1];
		args[0] = pointingModel;
		conversationInterceptor.invoke(methodToInvoke, this, args);
	}
	
	public ConversationTokenProvider privateEndPointingModelSave(PointingModel pm) 
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		PointingModelService service = TmcdbContextFactory.INSTANCE.getPointingModelService();
		service.endSave(pm);
		return retVal;
	}

}
