/*******************************************************************************
  * ALMA - Atacama Large Millimeter Array
  * Copyright (c) NRAO - National Radio Astronomy Observatory, 2012
  * (in the framework of the ALMA collaboration).
  * All rights reserved.
  *
  * This library is free software; you can redistribute it and/or
  * modify it under the terms of the GNU Lesser General Public
  * License as published by the Free Software Foundation; either
  * version 2.1 of the License, or (at your option) any later version.
  *
  * This library is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  * Lesser General Public License for more details.
  *
  * You should have received a copy of the GNU Lesser General Public
  * License along with this library; if not, write to the Free Software
  * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 
*******************************************************************************/
package alma.obops.tmcdbgui.utils.conversation;

import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.hibernate.criterion.MatchMode;

import alma.acs.tmcdb.AlarmDefinition;
import alma.acs.tmcdb.Component;
import alma.acs.tmcdb.Container;
import alma.acs.tmcdb.Schemas;
import alma.obops.dam.config.TmcdbContextFactory;
import alma.obops.dam.tmcdb.domain.TMCDBExport;
import alma.obops.dam.tmcdb.service.AlarmDefinitionService;
import alma.obops.dam.tmcdb.service.AntennaService;
import alma.obops.dam.tmcdb.service.BaseElementService;
import alma.obops.dam.tmcdb.service.ConfigurationService;
import alma.obops.dam.tmcdb.service.SchemasService;
import alma.obops.dam.utils.ConversationInterceptor;
import alma.obops.dam.utils.ConversationTokenProvider;
import alma.obops.dam.utils.ConversationTokenProviderAdapter;
import alma.obops.dam.utils.ConversationTokenProvider.ConversationToken;
import alma.tmcdb.cloning.CloningTestUtils;
import alma.tmcdb.domain.Antenna;
import alma.tmcdb.domain.AssemblyStartup;
import alma.tmcdb.domain.BaseElement;
import alma.tmcdb.domain.BaseElementStartup;
import alma.tmcdb.domain.HwConfiguration;
import alma.tmcdb.domain.StartupScenario;
import alma.tmcdb.domain.XPDelay;
import alma.tmcdb.history.HistoryRecord;

/**
 * @author sharring
 *
 */
public class HwConfigurationConversationUtils 
{
	private static HwConfigurationConversationUtils singletonInstance;

	private HwConfigurationConversationUtils() 
	{
	}

	public static synchronized HwConfigurationConversationUtils getInstance()
	{
		if(null == singletonInstance)
		{
			singletonInstance = new HwConfigurationConversationUtils();
		}

		return singletonInstance;
	}
	
	private class BaseElementArrayHolder 
	{
		BaseElement[] baseElements;
		
		public BaseElement[] getBaseElements() { return this.baseElements; }
		public void setBaseElements(BaseElement[] bes) { this.baseElements = bes; }
	}
	
	public BaseElement[] findTopLevelBaseElementsByConfiguration(HwConfiguration config) throws Exception
	{
		BaseElement[] retVal = null;

		Method methodToInvoke = HwConfigurationConversationUtils.class.getMethod("privateFindTopLevelBaseElementsByConfiguration", 
				HwConfiguration.class, BaseElementArrayHolder.class);
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[2];
		args[0] = config;
		BaseElementArrayHolder resultHolder = new BaseElementArrayHolder();
		args[1] = resultHolder;
		conversationInterceptor.invoke(methodToInvoke, this, args);
		retVal = resultHolder.getBaseElements();

		return retVal;
	}
	
	public ConversationTokenProvider privateFindTopLevelBaseElementsByConfiguration(HwConfiguration configuration, 
			BaseElementArrayHolder resultHolder)
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		BaseElementService service = TmcdbContextFactory.INSTANCE.getBaseElementService();
		resultHolder.setBaseElements(service.findTopLevelBaseElementsByConfiguration(configuration).toArray(new BaseElement[0]));
		return retVal;
	}
	
	// this is not yet working; needs to be completed
	public void removeEntireConfiguration(HwConfiguration configuration) throws Exception
	{
		configuration = HwConfigurationConversationUtils.getInstance().readConfigurationById(configuration.getId());
		configuration = HwConfigurationConversationUtils.getInstance().hydrateConfigurationForCloning(configuration);
		
		Set<StartupScenario> scenariosToDelete = new HashSet<StartupScenario>(configuration.getStartupScenarios());
		for(StartupScenario scenario : scenariosToDelete) {
			removeEntireStartupScenario(scenario, false);
		}
		BackendConversationUtils.getInstance().delete(configuration, ConversationToken.CONVERSATION_PENDING, true);
		
		configuration.getSwConfiguration().getAcsServices();
		configuration.getSwConfiguration().getAlarmCategories();
		
		for(Component component : configuration.getSwConfiguration().getComponents()) {
			BackendConversationUtils.getInstance().delete(component, ConversationToken.CONVERSATION_PENDING, true);
		}
		
		for(Container container : configuration.getSwConfiguration().getContainers()) {
			BackendConversationUtils.getInstance().delete(container, ConversationToken.CONVERSATION_PENDING, true);
		}
		
		configuration.getSwConfiguration().getEventChannels();
		configuration.getSwConfiguration().getFaultFamilies();
//		for(Manager manager : configuration.getSwConfiguration().getManagers()) {
//			delete(manager, ConversationToken.CONVERSATION_PENDING, true);
//		}
		
		configuration.getSwConfiguration().getNetworkDevices();
		configuration.getSwConfiguration().getNotificationServiceMappings();
		configuration.getSwConfiguration().getReductionLinks();
		configuration.getSwConfiguration().getReductionThresholds();
		
		for(Schemas schemas : configuration.getSchemas()) {
			BackendConversationUtils.getInstance().delete(schemas, ConversationToken.CONVERSATION_PENDING, true);
		}

		BackendConversationUtils.getInstance().delete(configuration.getSwConfiguration(), ConversationToken.CONVERSATION_COMPLETED, true);
	}

	public void removeEntireStartupScenario(StartupScenario startupScenario, boolean commitAtEnd) throws Exception
	{
		StartupScenarioConversationUtils.getInstance().hydrateBaseElementStartups(startupScenario);
		
		// make a copy of collection, so as to avoid ConcurrentModificationExceptions
		Set<BaseElementStartup> besCopy = new HashSet<BaseElementStartup>();
		besCopy.addAll(startupScenario.getBaseElementStartups());
		
		// iterate over the baseelementstartups, deleting their assemblystartups
		for(BaseElementStartup bes : besCopy) 
		{
			Set<AssemblyStartup> assStartupsCopy = new HashSet<AssemblyStartup>();
			assStartupsCopy.addAll(bes.getAssemblyStartups());
			for(AssemblyStartup assStartup : assStartupsCopy) {
				// delete the assembly startup
				BackendConversationUtils.getInstance().delete(assStartup, ConversationToken.CONVERSATION_PENDING, true);
				bes.getAssemblyStartups().remove(assStartup);
			}
			
			// make a copy of collection, so as to avoid ConcurrentModificationExceptions
			Set<BaseElementStartup> besCopy2 = new HashSet<BaseElementStartup>();
			besCopy2.addAll(bes.getChildren());
			
			// TODO: START - recurse multiple levels of children?? 
			for(BaseElementStartup besCheck : bes.getChildren()) {
				if(besCheck.getChildren() != null && besCheck.getChildren().size() > 0) {
					throw new IllegalStateException("SLH didn't expect nested children here!");
				}
			}
			// TODO: END - recurse multiple levels of children??
			
			// iterate over the nested baseelementstartups, if any, deleting their assemblystartups
			for(BaseElementStartup bes2 : besCopy2) 
			{
				Set<AssemblyStartup> assStartupsCopy2 = new HashSet<AssemblyStartup>();
				assStartupsCopy.addAll(bes2.getAssemblyStartups());
				for(AssemblyStartup assStartup2 : assStartupsCopy2) {
					// delete the nested assembly startup
					BackendConversationUtils.getInstance().delete(assStartup2, ConversationToken.CONVERSATION_PENDING, true);
					bes2.getAssemblyStartups().remove(assStartup2);
				}
				
				// delete the nested baseelementstartup
				BackendConversationUtils.getInstance().delete(bes2, ConversationToken.CONVERSATION_PENDING, true);
				bes.getChildren().remove(bes2);
			}
			
			// delete the baseelementstartup
			BackendConversationUtils.getInstance().delete(bes, ConversationToken.CONVERSATION_PENDING, true);
			startupScenario.getBaseElementStartups().remove(bes);
		}
		
		// finally delete the startup scenario
		ConversationToken token = ConversationToken.CONVERSATION_FAILED;
		if(commitAtEnd) {
			token = ConversationToken.CONVERSATION_COMPLETED;
			startupScenario.getConfiguration().getStartupScenarios().remove(startupScenario);
		} else {
			 token = ConversationToken.CONVERSATION_PENDING;
		}
		BackendConversationUtils.getInstance().delete(startupScenario, token, true);
	}
	
	public ConversationTokenProvider privateDeleteHwConfiguration(HwConfiguration config, ConversationToken token) {
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(token);
		ConfigurationService configService = TmcdbContextFactory.INSTANCE.getConfigurationService();
		configService.delete(config);
		configService.delete(config.getSwConfiguration());
		return retVal;
	}
	
	public List<AlarmDefinition> findAlarmDefinitionsForConfiguration(HwConfiguration conf)  throws Exception
	{
		Method methodToInvoke = HwConfigurationConversationUtils.class.
		   getMethod("privateFindAlarmDefinitionsForConfiguration", HwConfiguration.class, AlarmDefinitionListHolder.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[2];
		args[0] = conf;
		args[1] = new AlarmDefinitionListHolder();
		AlarmDefinitionListHolder resultHolder = (AlarmDefinitionListHolder)args[1];
		conversationInterceptor.invoke(methodToInvoke, this, args);
		
		List<AlarmDefinition> resultList = resultHolder.getAlarmDefinitions();
		return resultList;
	}
	
	public ConversationTokenProvider privateFindAlarmDefinitionsForConfiguration(HwConfiguration conf, AlarmDefinitionListHolder resultHolder) 
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_PENDING);
		AlarmDefinitionService service = TmcdbContextFactory.INSTANCE.getAlarmDefinitionService();
		resultHolder.setAlarmDefinitions(service.findAllInConfiguration(conf.getSwConfiguration()));
		return retVal;
	}
	
	private class AlarmDefinitionListHolder
	{
		private List<AlarmDefinition> alarmDefinitions;
		
		public List<AlarmDefinition> getAlarmDefinitions() {
			return alarmDefinitions;
		}
		
		public void setAlarmDefinitions(List<AlarmDefinition> alarmDefinitions) {
			this.alarmDefinitions = alarmDefinitions;
		}
	}
	public void hydrateConfigurationHashCode(HwConfiguration configuration) throws Exception 
	{
		Method methodToInvoke = HwConfigurationConversationUtils.class.getMethod("privateHydrateConfigurationHashCode", 
				HwConfiguration.class);
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[1];
		args[0] = configuration;
		conversationInterceptor.invoke(methodToInvoke, this, args);
	}

	public ConversationTokenProvider privateHydrateConfigurationHashCode(HwConfiguration configuration)
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		ConfigurationService configService = TmcdbContextFactory.INSTANCE.getConfigurationService();
		configService.hydrateConfigurationHashCode(configuration);
		return retVal;
	}

	public HwConfiguration hydrateConfigurationForCloning(HwConfiguration configuration) 
	   throws Exception 
	{
		Method methodToInvoke = HwConfigurationConversationUtils.class.getMethod("privateHydrateConfigurationForCloning", 
				HwConfiguration.class, ConfigurationHolder.class);
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[2];
		args[0] = configuration;
		ConfigurationHolder holder = new ConfigurationHolder();
		args[1] = holder;

		conversationInterceptor.invoke(methodToInvoke, this, args);
		return holder.getConfiguration();
	}
	
	public ConversationTokenProvider privateHydrateConfigurationForCloning(HwConfiguration configuration, ConfigurationHolder holder)
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		ConfigurationService configService = TmcdbContextFactory.INSTANCE.getConfigurationService();
		holder.setConfiguration(configService.hydrateConfigurationForCloning(configuration));		
		return retVal;
	}

	public HwConfiguration reHydrateConfigurationSimple(HwConfiguration configuration) 
	   throws Exception 
	{
		Method methodToInvoke = 
			HwConfigurationConversationUtils.class.getMethod("privateReHydrateConfigurationSimple", 
				HwConfiguration.class, ConfigurationHolder.class);
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[2];
		args[0] = configuration;
		ConfigurationHolder holder = new ConfigurationHolder();
		args[1] = holder;

		conversationInterceptor.invoke(methodToInvoke, this, args);
		return holder.getConfiguration();
	}
	
	public ConversationTokenProvider privateReHydrateConfigurationSimple(HwConfiguration configuration, ConfigurationHolder holder)
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		ConfigurationService configService = TmcdbContextFactory.INSTANCE.getConfigurationService();
		holder.setConfiguration(configService.reHydrateSimple(configuration));		
		return retVal;
	}
	
	public HwConfiguration hydrateConfigurationForExport(HwConfiguration configuration) 
	   throws Exception 
	{
		Method methodToInvoke = HwConfigurationConversationUtils.class.getMethod("privateHydrateConfigurationForExport", 
				HwConfiguration.class, ConfigurationHolder.class);
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		ConfigurationHolder holder = new ConfigurationHolder();
		Object[] args = new Object[2];
		args[0] = configuration;
		args[1] = holder;
		conversationInterceptor.invoke(methodToInvoke, this, args);
		return holder.getConfiguration();
	}
	
	public ConversationTokenProvider privateHydrateConfigurationForExport(HwConfiguration configuration, ConfigurationHolder holder)
	{
		// Keep the conversation open (pending) so that the serialization will be able to use the same session
		// and avoid LazyInitializationExceptions for empty collections; see comments in COMP-4943.
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_PENDING);

		ConfigurationService configService = TmcdbContextFactory.INSTANCE.getConfigurationService();
		holder.setConfiguration(configService.hydrateConfigurationForExport(configuration));		
		return retVal;
	}

	public void hydrateConfiguration(HwConfiguration configuration) 
	   throws Exception 
	{
		Method methodToInvoke = HwConfigurationConversationUtils.class.getMethod("privateHydrateConfiguration", 
				HwConfiguration.class);
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[1];
		args[0] = configuration;
		conversationInterceptor.invoke(methodToInvoke, this, args);
	}
	
	public ConversationTokenProvider privateHydrateConfiguration(HwConfiguration configuration)
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		ConfigurationService configService = TmcdbContextFactory.INSTANCE.getConfigurationService();
		configService.hydrate(configuration);		
		return retVal;
	}
	
	public List<HwConfiguration> findConfigurationsByName(String name, boolean active, MatchMode matchMode) throws Exception
	{
		Method methodToInvoke = HwConfigurationConversationUtils.class.getMethod("privateFindConfigurationsByNameActiveAndMatchModeFlags", 
				String.class, Boolean.class, MatchMode.class, ConfigurationsHolder.class);
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[4];
		args[0] = name;
		args[1] = active;
		args[2] = matchMode;
		ConfigurationsHolder resultHolder = new ConfigurationsHolder();
		args[3] = resultHolder;
		conversationInterceptor.invoke(methodToInvoke, this, args);
		List<HwConfiguration> retVal = resultHolder.getConfigurations();
		return retVal;
	}
	
	public ConversationTokenProvider privateFindConfigurationsByNameActiveAndMatchModeFlags(String name, Boolean active, MatchMode matchMode, ConfigurationsHolder resultHolder) 
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		ConfigurationService cService =  TmcdbContextFactory.INSTANCE.getConfigurationService();			
		resultHolder.setConfigurations (cService.findByName(name, active, matchMode));		
		return retVal;
	}
	
	public List<HwConfiguration> findConfigurationsByName(String name) throws Exception
	{
		Method methodToInvoke = HwConfigurationConversationUtils.class.getMethod("privateFindConfigurationsByName", String.class, ConfigurationsHolder.class);
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[2];
		args[0] = name;
		ConfigurationsHolder resultHolder = new ConfigurationsHolder();
		args[1] = resultHolder;
		conversationInterceptor.invoke(methodToInvoke, this, args);
		List<HwConfiguration> retVal = resultHolder.getConfigurations();
		return retVal;
	}
	
	@SuppressWarnings("unchecked")
	public ConversationTokenProvider privateFindConfigurationsByName(String name, ConfigurationsHolder resultHolder) 
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		ConfigurationService cService =  TmcdbContextFactory.INSTANCE.getConfigurationService();			
		resultHolder.setConfigurations (((List<HwConfiguration>) cService.findByName( name )));		
		return retVal;
	}

	public List<String> getConfigurationNames() throws Exception 
	{
		Method methodToInvoke = HwConfigurationConversationUtils.class.getMethod("privateGetConfigurationNames", Boolean.class, StringsHolder.class);
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[2];
		args[0] = null;
		StringsHolder resultHolder = new StringsHolder();
		args[1] = resultHolder;
		conversationInterceptor.invoke(methodToInvoke, this, args);
		List<String> retVal = resultHolder.getStrings();
		return retVal;
	}
	
	public List<String> getConfigurationNames(boolean activeFlag) throws Exception
	{
		Method methodToInvoke = HwConfigurationConversationUtils.class.getMethod("privateGetConfigurationNames", Boolean.class, StringsHolder.class);
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[2];
		args[0] = activeFlag;
		StringsHolder resultHolder = new StringsHolder();
		args[1] = resultHolder;
		conversationInterceptor.invoke(methodToInvoke, this, args);
		List<String> retVal = resultHolder.getStrings();
		return retVal;
	}
	
	public ConversationTokenProvider privateGetConfigurationNames(Boolean activeFlag, StringsHolder resultHolder) 
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		ConfigurationService cService =  TmcdbContextFactory.INSTANCE.getConfigurationService();			
		resultHolder.setStrings (cService.getConfigurationNames( activeFlag ));		
		return retVal;
	}
	
	private class StringsHolder
	{
		private List<String> strings = new ArrayList<String>();
		
		public List<String> getStrings() {
			return strings;
		}
		
		public void setStrings(List<String> strs) {
			this.strings = strs;
		}
	}
	
	public List<HwConfiguration> findConfigurationsByName(String name, MatchMode matchMode) throws Exception
	{
		Method methodToInvoke = HwConfigurationConversationUtils.class.getMethod("privateFindConfigurationsByName", String.class, MatchMode.class, ConfigurationsHolder.class);
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[3];
		args[0] = name;
		args[1] = matchMode;
		ConfigurationsHolder resultHolder = new ConfigurationsHolder();
		args[2] = resultHolder;
		conversationInterceptor.invoke(methodToInvoke, this, args);
		return resultHolder.getConfigurations();
	}
	
	public ConversationTokenProvider privateFindConfigurationsByName(String name, MatchMode matchMode, ConfigurationsHolder resultHolder) 
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		ConfigurationService cService =  TmcdbContextFactory.INSTANCE.getConfigurationService();			
		resultHolder.setConfigurations (cService.findByName( name, matchMode ));		
		return retVal;
	}

	public HwConfiguration findConfigurationById(Long id) throws Exception
	{
		Method methodToInvoke = HwConfigurationConversationUtils.class.getMethod("privateFindConfigurationById", Long.class, ConfigurationHolder.class);
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[2];
		args[0] = id;
		ConfigurationHolder resultHolder = new ConfigurationHolder();
		args[1] = resultHolder;
		conversationInterceptor.invoke(methodToInvoke, this, args);
		return resultHolder.getConfiguration();
	}
	
	public ConversationTokenProvider privateFindConfigurationById(Long id, ConfigurationHolder resultHolder) 
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		ConfigurationService cService =  TmcdbContextFactory.INSTANCE.getConfigurationService();			
		HwConfiguration conf = (HwConfiguration)cService.read(id);	
		cService.hydrateSwConfiguration(conf);
		resultHolder.setConfiguration(conf);
		return retVal;
	}

	private class ConfigurationsHolder
	{
		private List<HwConfiguration> configurations = new ArrayList<HwConfiguration>();
		
		public List<HwConfiguration> getConfigurations() {
			return configurations;
		}
		
		public void setConfigurations(List<HwConfiguration> configs) {
			this.configurations = configs;
		}
	}
	
	public HwConfiguration readConfigurationById(Long id) 
	   throws Exception 
	{
		Method methodToInvoke = HwConfigurationConversationUtils.class.getMethod("privateReadConfigurationById", Long.class, ConfigurationHolder.class);
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[2];
		args[0] = id;
		ConfigurationHolder result = new ConfigurationHolder();
		args[1] = result;
		conversationInterceptor.invoke(methodToInvoke, this, args);
		HwConfiguration retVal = result.getConfiguration();
		return retVal;
	}
	
	public ConversationTokenProvider privateReadConfigurationById(Long id, ConfigurationHolder resultHolder) 
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		ConfigurationService configService =  TmcdbContextFactory.INSTANCE.getConfigurationService();
		HwConfiguration config = (HwConfiguration)configService.read(id);
		resultHolder.setConfiguration(config);
		return retVal;
	}
	
	public HwConfiguration cloneConfiguration(HwConfiguration originalConfig, String newName) 
	   throws Exception 
	{
		Method methodToInvoke = HwConfigurationConversationUtils.class.getMethod("privateCloneConfiguration", 
				HwConfiguration.class, String.class, ConfigurationHolder.class);
		
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[3];
		args[0] = originalConfig;
		args[1] = newName;
		ConfigurationHolder result = new ConfigurationHolder();
		args[2] = result;
		conversationInterceptor.invokeWithDeferredConstraints(methodToInvoke, this, args);
		HwConfiguration retVal = result.getConfiguration();
		return retVal;
	}
	
	public ConversationTokenProvider privateCloneConfiguration(HwConfiguration originalConfig, String newName, ConfigurationHolder resultHolder) 
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		ConfigurationService configService =  TmcdbContextFactory.INSTANCE.getConfigurationService();
		resultHolder.setConfiguration(configService.cloneConfiguration(originalConfig, newName));
		return retVal;
	}

	public HwConfiguration cloneImportedConfiguration(TMCDBExport originalConfig, String newName)
	   throws Exception
	{
		Method methodToInvoke = HwConfigurationConversationUtils.class.getMethod("privateCloneImportedConfiguration",
				TMCDBExport.class, String.class, ConfigurationHolder.class);

		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[3];
		args[0] = originalConfig;
		args[1] = newName;
		ConfigurationHolder result = new ConfigurationHolder();
		args[2] = result;
		conversationInterceptor.invoke(methodToInvoke, this, args);
		HwConfiguration retVal = result.getConfiguration();
		return retVal;
	}

	public ConversationTokenProvider privateCloneImportedConfiguration(TMCDBExport originalConfig, String newName, ConfigurationHolder resultHolder)
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		ConfigurationService configService =  TmcdbContextFactory.INSTANCE.getConfigurationService();
		resultHolder.setConfiguration(configService.cloneImportedConfiguration(originalConfig, newName));
		return retVal;
	}
	public void compareConfigurations(HwConfiguration config1, HwConfiguration config2) throws Exception
	{
		Method methodToInvoke = HwConfigurationConversationUtils.class.getMethod("privateCompareConfigurations", HwConfiguration.class, HwConfiguration.class);
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object [] args = new Object[2];
		args[0] = config1;
		args[1] = config2;
		conversationInterceptor.invoke(methodToInvoke, this, args);
	}
	
	public ConversationTokenProvider privateCompareConfigurations(HwConfiguration originalConfig, HwConfiguration clonedConfig) 
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		ConfigurationService configService =  TmcdbContextFactory.INSTANCE.getConfigurationService();
		
//   // don't think this is needed:
//		configService.update(originalConfig);
//		configService.update(clonedConfig);
//		configService.update(clonedConfig.getSwConfiguration());
//		configService.update(originalConfig.getSwConfiguration());
		
//		System.out.println("original config is: " + originalConfig.toString());
//		System.out.println("************************");
//		System.out.println("cloned config is: " + clonedConfig.toString());
		
		originalConfig = configService.hydrateConfigurationForCloning(originalConfig);
		clonedConfig = configService.hydrateConfigurationForCloning(clonedConfig);
		CloningTestUtils.compareConfigurations(originalConfig, clonedConfig);
		return retVal;
	}

	public String exportConfigurationAsXml(HwConfiguration conf) 
	   throws Exception 
	{
		Method methodToInvoke = HwConfigurationConversationUtils.class.getMethod("privateExportConfigurationAsXml",
				HwConfiguration.class, StringHolder.class);
		
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		StringHolder result = new StringHolder();
		Object[] args = new Object[2];
		args[0] = conf;
		args[1] = result;
		conversationInterceptor.invoke(methodToInvoke, this, args);
		String retVal = result.getString();
		return retVal;
	}
	
	public ConversationTokenProvider privateExportConfigurationAsXml(HwConfiguration conf, StringHolder resultHolder) 
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		ConfigurationService configService =  TmcdbContextFactory.INSTANCE.getConfigurationService();
		resultHolder.setString(configService.exportConfigurationAsXml(conf));
		return retVal;
	}

	public TMCDBExport importConfigurationFromXml(String xml)
	   throws Exception
	{
		Method methodToInvoke = HwConfigurationConversationUtils.class.getMethod("privateImportConfigurationFromXml",
				String.class, TMCDBExportHolder.class);

		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[2];
		args[0] = xml;
		TMCDBExportHolder result = new TMCDBExportHolder();
		args[1] = result;
		conversationInterceptor.invoke(methodToInvoke, this, args);
		TMCDBExport retVal = result.getExport();
		return retVal;
	}

	public ConversationTokenProvider privateImportConfigurationFromXml(String xml, TMCDBExportHolder resultHolder)
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		ConfigurationService configService =  TmcdbContextFactory.INSTANCE.getConfigurationService();
		resultHolder.setExport(configService.importConfigurationFromXml(xml));
		return retVal;
	}
	
	private class TMCDBExportHolder
	{
		private TMCDBExport export;

		public TMCDBExport getExport() {
			return export;
		}

		public void setExport(TMCDBExport export) {
			this.export = export;
		}
	}
	
	 private class StringHolder
		{
			private String str;
			
			public String getString() {
				return str;
			}
			
			public void setString(String str) {
				this.str = str;
			}
		}
	 public void addBaseElement(HwConfiguration configuration, BaseElement newBaseElement) 
	   throws Exception 
	{
		Method methodToInvoke = HwConfigurationConversationUtils.class.getMethod("privateAddBaseElement", BaseElement.class, HwConfiguration.class);
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[2];
		args[0] = newBaseElement;
		args[1] = configuration;
		conversationInterceptor.invoke(methodToInvoke, this, args);
	}
	
	public ConversationTokenProvider privateAddBaseElement(BaseElement newBaseElement, HwConfiguration config)
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		ConfigurationService configService = TmcdbContextFactory.INSTANCE.getConfigurationService();

		newBaseElement.setConfiguration(config);	
		configService.hydrateComponents(config);
		config.addBaseElement(newBaseElement);
		configService.update( config);

		return retVal;
	}
	
	public void hydrateBaseElements(HwConfiguration config) 
	   throws Exception 
	{
		Method methodToInvoke = HwConfigurationConversationUtils.class.getMethod("privateHydrateBaseElements", HwConfiguration.class);
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[1];
		args[0] = config;
		conversationInterceptor.invoke(methodToInvoke, this, args);
	}
	
	public ConversationTokenProvider privateHydrateBaseElements(HwConfiguration config) 
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		ConfigurationService configService =  TmcdbContextFactory.INSTANCE.getConfigurationService();
		configService.hydrateBaseElements(config);	
		return retVal;
	}
	
	 /**
     * Save the input configuration back to the database
     * 
     * @param configuration
     * @throws Exception 
     */
    public void updateConfiguration( HwConfiguration config ) throws Exception 
    {
    	Method methodToInvoke = this.getClass().getMethod("privateUpdateConfiguration", HwConfiguration.class);
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[1];
		args[0] = config;
		conversationInterceptor.invoke(methodToInvoke, this, args);
    }
    
    public ConversationTokenProvider privateUpdateConfiguration(HwConfiguration config) 
    {
        ConfigurationService service =  TmcdbContextFactory.INSTANCE.getConfigurationService();
        service.update( config );
        ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
        return retVal;
    }
    
    public boolean prepareHwConfigurationSave(HwConfiguration config, String userId, String description) throws Exception
	{
		Method methodToInvoke = HwConfigurationConversationUtils.class.getMethod("privatePrepareHwConfigurationSave", 
				HwConfiguration.class, String.class, String.class, BooleanHolder.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		BooleanHolder resultholder = new BooleanHolder();
		Object[] args = new Object[4];
		args[0] = config;
		args[1] = userId;
		args[2] = description;
		args[3] = resultholder;
		conversationInterceptor.invoke(methodToInvoke, this, args);
		boolean retVal = resultholder.getBooleanValue();
		return retVal;
	}
	
	public ConversationTokenProvider privatePrepareHwConfigurationSave(HwConfiguration config, String who, String description, BooleanHolder resultHolder) 
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		ConfigurationService service = TmcdbContextFactory.INSTANCE.getConfigurationService();
		boolean successful = service.prepareSave(config, who, description);
		resultHolder.setBooleanValue(successful);
		return retVal;
	}
	
	public void endHwConfigurationSave(HwConfiguration config) throws Exception 
	{
		Method methodToInvoke = HwConfigurationConversationUtils.class.getMethod("privateEndHwConfigurationSave", 
				HwConfiguration.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[1];
		args[0] = config;
		conversationInterceptor.invoke(methodToInvoke, this, args);
	}
	
	public ConversationTokenProvider privateEndHwConfigurationSave(HwConfiguration config) 
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		ConfigurationService service = TmcdbContextFactory.INSTANCE.getConfigurationService();
		service.endSave(config);
		return retVal;
	}
	
	private class AntennaListHolder
	{
		private List<Antenna> antennas;
		
		public List<Antenna> getAntennas() {
			return antennas;
		}
		
		public void setAntennas(List<Antenna> ants) {
			this.antennas = ants;
		}
	}

	public Antenna[] findAntennaByLoOffsetInConfig(Integer loOffset, HwConfiguration configuration) 
	   throws Exception 
		{
			Method methodToInvoke = HwConfigurationConversationUtils.class.getMethod("privateFindAntennasByLoOffsetInConfiguration", Integer.class, HwConfiguration.class, AntennaListHolder.class);
			ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
			Object[] args = new Object[3];
			args[0] = loOffset;
			args[1] = configuration;
			AntennaListHolder result = new AntennaListHolder();
			args[2] = result;
			conversationInterceptor.invoke(methodToInvoke, this, args);
			Antenna[] retVal = result.getAntennas().toArray(new Antenna[0]);
			return retVal;
		}
	
	public ConversationTokenProvider privateFindAntennasByLoOffsetInConfiguration(Integer loOffset, HwConfiguration config, AntennaListHolder resultHolder) 
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		
		AntennaService service =  TmcdbContextFactory.INSTANCE.getAntennaService();
		List<Antenna> antennas = service.findByLoOffsetInConfiguration(loOffset, config);
		resultHolder.setAntennas(antennas);
		
		return retVal;
	}

	public Antenna[] findAntennaByWalshFunctionInConfig(Integer walshSequence, HwConfiguration configuration) throws Exception
	{
		Method methodToInvoke = HwConfigurationConversationUtils.class.getMethod("privateFindAntennasByWalshFunctionInConfiguration", Integer.class, HwConfiguration.class, AntennaListHolder.class);
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[3];
		args[0] = walshSequence;
		args[1] = configuration;
		AntennaListHolder result = new AntennaListHolder();
		args[2] = result;
		conversationInterceptor.invoke(methodToInvoke, this, args);
		Antenna[] retVal = result.getAntennas().toArray(new Antenna[0]);
		return retVal;
	}

	public ConversationTokenProvider privateFindAntennasByWalshFunctionInConfiguration(Integer walshSeq, HwConfiguration config, AntennaListHolder resultHolder) 
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		AntennaService service =  TmcdbContextFactory.INSTANCE.getAntennaService();
		List<Antenna> antennas = service.findByWalshSequenceInConfiguration(walshSeq, config);
		resultHolder.setAntennas(antennas);
	
		return retVal;
	}

	public List<HistoryRecord> getXpDelayHistory(HwConfiguration config) throws Exception 
	{
		Method methodToInvoke = HwConfigurationConversationUtils.class.getMethod("privateGetXpDelayHistory", HwConfiguration.class, HistoryRecordListHolder.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		HistoryRecordListHolder holder = new HistoryRecordListHolder();
		Object[] args = new Object[2];
		args[0] = config;
		args[1] = holder;
		conversationInterceptor.invoke(methodToInvoke, this, args);
		return holder.getHistoryRecords();
		
	}
	
	public ConversationTokenProvider privateGetXpDelayHistory(HwConfiguration config, HistoryRecordListHolder resultHolder) 
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		ConfigurationService service = TmcdbContextFactory.INSTANCE.getConfigurationService();
		List<HistoryRecord> results = service.getHistory(config);
		resultHolder.setHistoryRecords(results);
		return retVal;
	}
	
	public Set<XPDelay> getHistoricalXpDelays(HwConfiguration config, HistoryRecord clickedRecord) throws Exception 
	{
		Method methodToInvoke = HwConfigurationConversationUtils.class.getMethod("privateGetHistoricalXpDelays", HwConfiguration.class, HistoryRecord.class, ConfigurationHolder.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		ConfigurationHolder holder = new ConfigurationHolder();
		Object[] args = new Object[3];
		args[0] = config;
		args[1] = clickedRecord;
		args[2] = holder;
		conversationInterceptor.invoke(methodToInvoke, this, args);
		HwConfiguration configRetVal = holder.getConfiguration();
		Set<XPDelay> retVal = configRetVal.getCrossPolarizationDelays();
		return retVal;
	}
	
	public ConversationTokenProvider privateGetHistoricalXpDelays(HwConfiguration config, HistoryRecord record, ConfigurationHolder resultHolder) 
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		ConfigurationService service = TmcdbContextFactory.INSTANCE.getConfigurationService();
		HwConfiguration historicalConfig = service.getHistoricalConfiguration(config, record.getVersion());
		resultHolder.setConfiguration(historicalConfig);
		return retVal;
	}

	public void hydrateComponentsShallow(HwConfiguration configuration)
	   throws Exception
	{
		Method methodToInvoke = HwConfigurationConversationUtils.class.getMethod("privateHydrateComponentsShallow", HwConfiguration.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[1];
		args[0] = configuration;
		conversationInterceptor.invoke(methodToInvoke, this, args);
	}
	
	public ConversationTokenProvider privateHydrateComponentsShallow(HwConfiguration config)
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		ConfigurationService configService = TmcdbContextFactory.INSTANCE.getConfigurationService();
		configService.hydrateComponentsShallow(config);
		return retVal;
	}
	
	public void hydrateComponents(HwConfiguration config) 
	   throws Exception
	{
		Method methodToInvoke = HwConfigurationConversationUtils.class.getMethod("privateHydrateComponents", 
				HwConfiguration.class);
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[1];
		args[0] = config;
		conversationInterceptor.invoke(methodToInvoke, this, args);
	}
	
	public ConversationTokenProvider privateHydrateComponents(HwConfiguration configuration)
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		ConfigurationService configService = TmcdbContextFactory.INSTANCE.getConfigurationService();
		configService.hydrateComponents(configuration);		
		return retVal;
	}
	
	public ConversationTokenProvider privateDeleteSchemas(Schemas schemas, ConversationToken token) {
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(token);
		SchemasService schemasService = TmcdbContextFactory.INSTANCE.getSchemasService();
		schemasService.delete(schemas);
		return retVal;
	}	
}
