/*******************************************************************************
  * ALMA - Atacama Large Millimeter Array
  * Copyright (c) NRAO - National Radio Astronomy Observatory, 2012
  * (in the framework of the ALMA collaboration).
  * All rights reserved.
  *
  * This library is free software; you can redistribute it and/or
  * modify it under the terms of the GNU Lesser General Public
  * License as published by the Free Software Foundation; either
  * version 2.1 of the License, or (at your option) any later version.
  *
  * This library is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  * Lesser General Public License for more details.
  *
  * You should have received a copy of the GNU Lesser General Public
  * License along with this library; if not, write to the Free Software
  * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 
*******************************************************************************/
package alma.obops.tmcdbgui.utils.conversation;

import java.lang.reflect.Method;
import java.util.List;

import alma.obops.dam.config.TmcdbContextFactory;
import alma.obops.dam.tmcdb.service.FocusModelService;
import alma.obops.dam.utils.ConversationInterceptor;
import alma.obops.dam.utils.ConversationTokenProvider;
import alma.obops.dam.utils.ConversationTokenProviderAdapter;
import alma.obops.dam.utils.ConversationTokenProvider.ConversationToken;
import alma.tmcdb.domain.FocusModel;
import alma.tmcdb.domain.FocusModelCoeff;
import alma.tmcdb.history.HistoryRecord;

/**
 * @author sharring
 *
 */
public class FocusModelConversationUtils 
{
	private static FocusModelConversationUtils singletonInstance;

	private FocusModelConversationUtils() 
	{
	}

	public static synchronized FocusModelConversationUtils getInstance()
	{
		if(null == singletonInstance)
		{
			singletonInstance = new FocusModelConversationUtils();
		}

		return singletonInstance;
	}	
	
	public ConversationTokenProvider privateDeleteFocusModelCoeff(FocusModelCoeff coeff, ConversationToken token) {
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(token);
		FocusModelService focusModelService = TmcdbContextFactory.INSTANCE.getFocusModelService();
		focusModelService.delete(coeff);
		return retVal;
	}
	
	public void saveOrUpdateFocusModel(FocusModel fm) throws Exception 
	{
		saveOrUpdateFocusModel(fm, ConversationToken.CONVERSATION_COMPLETED);
	}
	
	public void saveOrUpdateFocusModel(FocusModel fm, ConversationToken token) throws Exception 
	{
		Method methodToInvoke = FocusModelConversationUtils.class.getMethod("privateSaveOrUpdateFocusModel", FocusModel.class, ConversationToken.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[2];
		args[0] = fm;
		args[1] = token;
		conversationInterceptor.invoke(methodToInvoke, this, args);
	}

	public ConversationTokenProvider privateSaveOrUpdateFocusModel(FocusModel fm, ConversationToken token) 
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(token);
		FocusModelService fmService = TmcdbContextFactory.INSTANCE.getFocusModelService();
		fmService.update(fm);
		return retVal;
	}
	
	public List<HistoryRecord> getFocusModelHistory(FocusModel focusModel) 
	   throws Exception 
	{
		Method methodToInvoke = FocusModelConversationUtils.class.getMethod("privateGetFocusModelHistory", FocusModel.class, HistoryRecordListHolder.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		HistoryRecordListHolder holder = new HistoryRecordListHolder();
		Object[] args = new Object[2];
		args[0] = focusModel;
		args[1] = holder;
		conversationInterceptor.invoke(methodToInvoke, this, args);
		return holder.getHistoryRecords();
		
	}
	
	public ConversationTokenProvider privateGetFocusModelHistory(FocusModel fm, HistoryRecordListHolder resultHolder) 
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		FocusModelService service = TmcdbContextFactory.INSTANCE.getFocusModelService();
		List<HistoryRecord> results = service.getHistory(fm);
		resultHolder.setHistoryRecords(results);
		return retVal;
	}

	public FocusModel getHistoricalFocusModel(FocusModel focusModel, HistoryRecord record) throws Exception 
	{
		Method methodToInvoke = FocusModelConversationUtils.class.getMethod("privateGetHistoricalFocusModel", FocusModel.class, HistoryRecord.class, FocusModelHolder.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		FocusModelHolder holder = new FocusModelHolder();
		Object[] args = new Object[3];
		args[0] = focusModel;
		args[1] = record;
		args[2] = holder;
		conversationInterceptor.invoke(methodToInvoke, this, args);
		FocusModel retVal = holder.getFocusModel();
		retVal.setAntenna(focusModel.getAntenna());
		return retVal;
	}
	
	public ConversationTokenProvider privateGetHistoricalFocusModel(FocusModel fm, HistoryRecord record, FocusModelHolder resultHolder) 
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		FocusModelService service = TmcdbContextFactory.INSTANCE.getFocusModelService();
		FocusModel historicalFm = service.getHistoricalFocusModel(fm, record.getVersion());
		resultHolder.setFocusModel(historicalFm);
		return retVal;
	}
	
	private class FocusModelHolder
	{
		private FocusModel focusModel;
		
		public FocusModel getFocusModel() {
			return this.focusModel;
		}
		
		public void setFocusModel(FocusModel fm) {
			this.focusModel = fm;
		}
	}
	
	public boolean prepareFocusModelSave(FocusModel focusModel, String who, String description) throws Exception 
	{
		Method methodToInvoke = FocusModelConversationUtils.class.getMethod("privatePrepareFocusModelSave", 
				FocusModel.class, String.class, String.class, BooleanHolder.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		BooleanHolder resultholder = new BooleanHolder();
		Object[] args = new Object[4];
		args[0] = focusModel;
		args[1] = who;
		args[2] = description;
		args[3] = resultholder;
		conversationInterceptor.invoke(methodToInvoke, this, args);
		boolean retVal = resultholder.getBooleanValue();
		return retVal;
	}
	
	public ConversationTokenProvider privatePrepareFocusModelSave(FocusModel fm, 
			String who, String description, BooleanHolder resultHolder) 
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		FocusModelService service = TmcdbContextFactory.INSTANCE.getFocusModelService();
		boolean successful = service.prepareSave(fm, who, description);
		resultHolder.setBooleanValue(successful);
		return retVal;
	}
	
	public void endFocusModelSave(FocusModel focusModel) throws Exception 
	{
		Method methodToInvoke = FocusModelConversationUtils.class.getMethod("privateEndFocusModelSave", 
				FocusModel.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[1];
		args[0] = focusModel;
		conversationInterceptor.invoke(methodToInvoke, this, args);
	}
	
	public ConversationTokenProvider privateEndFocusModelSave(FocusModel fm) 
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		FocusModelService service = TmcdbContextFactory.INSTANCE.getFocusModelService();
		service.endSave(fm);
		return retVal;
	}
}
