/*******************************************************************************
  * ALMA - Atacama Large Millimeter Array
  * Copyright (c) NRAO - National Radio Astronomy Observatory, 2012
  * (in the framework of the ALMA collaboration).
  * All rights reserved.
  *
  * This library is free software; you can redistribute it and/or
  * modify it under the terms of the GNU Lesser General Public
  * License as published by the Free Software Foundation; either
  * version 2.1 of the License, or (at your option) any later version.
  *
  * This library is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  * Lesser General Public License for more details.
  *
  * You should have received a copy of the GNU Lesser General Public
  * License along with this library; if not, write to the Free Software
  * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 
*******************************************************************************/
package alma.obops.tmcdbgui.utils.conversation;

import java.lang.reflect.Method;
import java.util.List;

import alma.obops.dam.config.TmcdbContextFactory;
import alma.obops.dam.tmcdb.service.AntennaService;
import alma.obops.dam.tmcdb.service.XpDelayService;
import alma.obops.dam.utils.ConversationInterceptor;
import alma.obops.dam.utils.ConversationTokenProvider;
import alma.obops.dam.utils.ConversationTokenProviderAdapter;
import alma.obops.dam.utils.ConversationTokenProvider.ConversationToken;
import alma.obops.tmcdbgui.views.providers.helpers.config.DelayModel;
import alma.tmcdb.domain.Antenna;
import alma.tmcdb.domain.XPDelay;
import alma.tmcdb.history.HistoryRecord;

/**
 * @author sharring
 *
 */
public class DelaysConversationUtils 
{
	private static DelaysConversationUtils singletonInstance;
	
	private DelaysConversationUtils() 
	{
	}
	
	public static synchronized DelaysConversationUtils getInstance()
	{
		if(null == singletonInstance)
		{
			singletonInstance = new DelaysConversationUtils();
		}
		
		return singletonInstance;
	}
	
	public void saveOrUpdateXpDelay(XPDelay delay, ConversationToken token) throws Exception 
	{
		Method methodToInvoke = DelaysConversationUtils.class.getMethod("privateSaveOrUpdateXpDelay", XPDelay.class, ConversationToken.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[2];
		args[0] = delay;
		args[1] = token;
		conversationInterceptor.invoke(methodToInvoke, this, args);
	}

	public ConversationTokenProvider privateSaveOrUpdateXpDelay(XPDelay delay, ConversationToken token) {
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(token);
		XpDelayService service = TmcdbContextFactory.INSTANCE.getXpDelayService();
		if(delay.getId() != null)
		{
			service.update(delay);
		}
		else {
			service.create(delay);
		}
		return retVal;
	}

	public List<HistoryRecord> getDelayModelHistory(DelayModel delayModel) throws Exception 
	{
		Method methodToInvoke = DelaysConversationUtils.class.getMethod("privateGetDelayModelHistory", DelayModel.class, HistoryRecordListHolder.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		HistoryRecordListHolder holder = new HistoryRecordListHolder();
		Object[] args = new Object[2];
		args[0] = delayModel;
		args[1] = holder;
		conversationInterceptor.invoke(methodToInvoke, this, args);
		return holder.getHistoryRecords();

	}

	public ConversationTokenProvider privateGetDelayModelHistory(DelayModel dm, HistoryRecordListHolder resultHolder) 
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		AntennaService service = TmcdbContextFactory.INSTANCE.getAntennaService();
		List<HistoryRecord> results = service.getDelayModelHistory(dm.getAntenna());
		resultHolder.setHistoryRecords(results);
		return retVal;
	}
	
	private class DelayModelHolder
	{
		private DelayModel delayModel;
		
		public DelayModel getDelayModel() {
			return this.delayModel;
		}
		
		public void setDelayModel(DelayModel dm) {
			this.delayModel = dm;
		}
	}

	public DelayModel getHistoricalDelayModel(DelayModel delayModel,
			HistoryRecord referenceRecord) throws Exception 
	{
		Method methodToInvoke = DelaysConversationUtils.class.getMethod("privateGetHistoricalDelayModel", DelayModel.class, HistoryRecord.class, DelayModelHolder.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		DelayModelHolder holder = new DelayModelHolder();
		Object[] args = new Object[3];
		args[0] = delayModel;
		args[1] = referenceRecord;
		args[2] = holder;
		conversationInterceptor.invoke(methodToInvoke, this, args);
		DelayModel retVal = holder.getDelayModel();
		return retVal;
	}
	
	public ConversationTokenProvider privateGetHistoricalDelayModel(DelayModel dm, HistoryRecord record, DelayModelHolder resultHolder) 
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		AntennaService service = TmcdbContextFactory.INSTANCE.getAntennaService();
		Antenna historicalAntenna = service.getHistoricalDelayAntenna(dm.getAntenna(), record.getVersion());
		DelayModel historicalDm = new DelayModel(historicalAntenna, null);
		resultHolder.setDelayModel(historicalDm);
		return retVal;
	}
	
	public boolean prepareDelayModelSave(DelayModel delayModel, String who, String description)
	   throws Exception
	{
		Method methodToInvoke = DelaysConversationUtils.class.getMethod("privatePrepareDelayModelSave", 
				DelayModel.class, String.class, String.class, BooleanHolder.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		BooleanHolder resultholder = new BooleanHolder();
		Object[] args = new Object[4];
		args[0] = delayModel;
		args[1] = who;
		args[2] = description;
		args[3] = resultholder;
		conversationInterceptor.invoke(methodToInvoke, this, args);
		boolean retVal = resultholder.getBooleanValue();
		return retVal;
	}
	
	public ConversationTokenProvider privatePrepareDelayModelSave(DelayModel dm, 
			String who, String description, BooleanHolder resultHolder) 
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		AntennaService service = TmcdbContextFactory.INSTANCE.getAntennaService();
		boolean successful = service.prepareDelaySave(dm.getAntenna(), who, description);
		resultHolder.setBooleanValue(successful);
		return retVal;
	}
	
	public void endDelayModelSave(DelayModel delayModel)
	   throws Exception
	{
		Method methodToInvoke = DelaysConversationUtils.class.getMethod("privateEndDelayModelSave", 
				DelayModel.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[1];
		args[0] = delayModel;
		conversationInterceptor.invoke(methodToInvoke, this, args);
	}
	
	public ConversationTokenProvider privateEndDelayModelSave(DelayModel dm) 
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		AntennaService service = TmcdbContextFactory.INSTANCE.getAntennaService();
		service.endDelaySave(dm.getAntenna());
		return retVal;
	}
	
}
