/*******************************************************************************
  * ALMA - Atacama Large Millimeter Array
  * Copyright (c) NRAO - National Radio Astronomy Observatory, 2012
  * (in the framework of the ALMA collaboration).
  * All rights reserved.
  *
  * This library is free software; you can redistribute it and/or
  * modify it under the terms of the GNU Lesser General Public
  * License as published by the Free Software Foundation; either
  * version 2.1 of the License, or (at your option) any later version.
  *
  * This library is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  * Lesser General Public License for more details.
  *
  * You should have received a copy of the GNU Lesser General Public
  * License along with this library; if not, write to the Free Software
  * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 
*******************************************************************************/
package alma.obops.tmcdbgui.utils.conversation;

import java.lang.reflect.Method;
import java.util.List;

import alma.acs.tmcdb.Component;
import alma.acs.tmcdb.Configuration;
import alma.acs.tmcdb.DefaultCanAddress;
import alma.obops.dam.config.TmcdbContextFactory;
import alma.obops.dam.tmcdb.service.DefaultCanAddressService;
import alma.obops.dam.utils.ConversationInterceptor;
import alma.obops.dam.utils.ConversationTokenProvider;
import alma.obops.dam.utils.ConversationTokenProviderAdapter;
import alma.obops.dam.utils.ConversationTokenProvider.ConversationToken;

/**
 * @author sharring
 *
 */
public class DefaultCanAddressConversationUtils 
{
	private static DefaultCanAddressConversationUtils singletonInstance;

	private DefaultCanAddressConversationUtils() 
	{
	}

	public static synchronized DefaultCanAddressConversationUtils getInstance()
	{
		if(null == singletonInstance)
		{
			singletonInstance = new DefaultCanAddressConversationUtils();
		}

		return singletonInstance;
	}
	
	public void saveOrUpdate(DefaultCanAddress dca) throws Exception 
	{
		Method methodToInvoke = DefaultCanAddressConversationUtils.class.getMethod("privateSaveOrUpdate", DefaultCanAddress.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[1];
		args[0] = dca;
		conversationInterceptor.invoke(methodToInvoke, this, args);
	}

	public ConversationTokenProvider privateSaveOrUpdate(DefaultCanAddress dca) 
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		DefaultCanAddressService service = TmcdbContextFactory.INSTANCE.getDefaultCanAddressService();
		service.update(dca);
		return retVal;
	}

	/**
	 * @param swConfiguration the configuration for which we wish to find all the DefaultCanAddress objects
	 * @param resultHolder the holder object for returning the results of the search
	 * @return a list of the DefaultCanAddress items associated with the given configuration
	 * @throws Exception 
	 */
	public List<DefaultCanAddress> findAll(Configuration swConfiguration) throws Exception 
	{
		Method methodToInvoke = DefaultCanAddressConversationUtils.class.getMethod("privateFindAll", Configuration.class, DefaultCanAddressListHolder.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[2];
		args[0] = swConfiguration;
		DefaultCanAddressListHolder resultHolder = new DefaultCanAddressListHolder();
		args[1] = resultHolder;
		conversationInterceptor.invoke(methodToInvoke, this, args);
		return resultHolder.getDefaultCanAddresses();
	}
	
	@SuppressWarnings("cast")
	public ConversationTokenProvider privateFindAll(Configuration config, DefaultCanAddressListHolder resultHolder) 
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		DefaultCanAddressService service = TmcdbContextFactory.INSTANCE.getDefaultCanAddressService();
		resultHolder.setDefaultCanAddresses((List<DefaultCanAddress>) service.findAll(config));
		return retVal;
	}
	
	private class DefaultCanAddressListHolder
	{
		private List<DefaultCanAddress> dcas;
		public List<DefaultCanAddress> getDefaultCanAddresses() { return this.dcas; }
		public void setDefaultCanAddresses(List<DefaultCanAddress> dcas) { this.dcas = dcas; }
	}
	
	private class DefaultCanAddressHolder
	{
		private DefaultCanAddress dca;
		public DefaultCanAddress getDefaultCanAddress() { return this.dca; }
		public void setDefaultCanAddress(DefaultCanAddress dca) { this.dca = dca; }
	}

	/**
	 * @param component
	 * @return the defaultcanaddress for the given component
	 */
	public DefaultCanAddress findForComponent(Component component) throws Exception
	{
		Method methodToInvoke = DefaultCanAddressConversationUtils.class.getMethod("privateFindForComponent", Component.class, DefaultCanAddressHolder.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[2];
		args[0] = component;
		DefaultCanAddressHolder resultHolder = new DefaultCanAddressHolder();
		args[1] = resultHolder;
		conversationInterceptor.invoke(methodToInvoke, this, args);
		return resultHolder.getDefaultCanAddress();
	}
	
	@SuppressWarnings("cast")
	public ConversationTokenProvider privateFindForComponent(Component comp, DefaultCanAddressHolder resultHolder) 
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		DefaultCanAddressService service = TmcdbContextFactory.INSTANCE.getDefaultCanAddressService();
		resultHolder.setDefaultCanAddress((DefaultCanAddress) service.findForComponent(comp));
		return retVal;
	}
}
