/*******************************************************************************
  * ALMA - Atacama Large Millimeter Array
  * Copyright (c) NRAO - National Radio Astronomy Observatory, 2012
  * (in the framework of the ALMA collaboration).
  * All rights reserved.
  *
  * This library is free software; you can redistribute it and/or
  * modify it under the terms of the GNU Lesser General Public
  * License as published by the Free Software Foundation; either
  * version 2.1 of the License, or (at your option) any later version.
  *
  * This library is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  * Lesser General Public License for more details.
  *
  * You should have received a copy of the GNU Lesser General Public
  * License along with this library; if not, write to the Free Software
  * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 
*******************************************************************************/
package alma.obops.tmcdbgui.utils.conversation;

import java.lang.reflect.Method;
import java.util.List;

import alma.acs.tmcdb.Configuration;
import alma.acs.tmcdb.Container;
import alma.acs.tmcdb.ContainerStartupOption;
import alma.obops.dam.config.TmcdbContextFactory;
import alma.obops.dam.tmcdb.service.ContainerService;
import alma.obops.dam.tmcdb.service.ContainerStartupOptionService;
import alma.obops.dam.tmcdb.service.SwConfigurationService;
import alma.obops.dam.utils.ConversationInterceptor;
import alma.obops.dam.utils.ConversationTokenProvider;
import alma.obops.dam.utils.ConversationTokenProviderAdapter;
import alma.obops.dam.utils.ConversationTokenProvider.ConversationToken;

/**
 * @author sharring
 *
 */
public class ContainerConversationUtils 
{
	private static ContainerConversationUtils singletonInstance;
	
	private ContainerConversationUtils() 
	{
	}
	
	public static synchronized ContainerConversationUtils getInstance()
	{
		if(null == singletonInstance)
		{
			singletonInstance = new ContainerConversationUtils();
		}
		
		return singletonInstance;
	}
	
	public Container readContainerById(Integer id)
	throws Exception {
		Method methodToInvoke = ContainerConversationUtils.class.getMethod("privateReadContainerById", Integer.class, ContainerHolder.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		ContainerHolder holder = new ContainerHolder();
		Object[] args = new Object[2];
		args[0] = id;
		args[1] = holder;
		conversationInterceptor.invoke(methodToInvoke, this, args);
		return holder.getContainer();
	}

	public ConversationTokenProvider privateReadContainerById(Integer id, ContainerHolder holder)
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		ContainerService containerService = TmcdbContextFactory.INSTANCE.getContainerService();
		Container c = (Container)containerService.read(id);
		containerService.hydrateConfiguration(c);
		holder.setContainer( c );
		return retVal;
	}
	
	public void hydrateContainerStartupOptions(Container container)
	   throws Exception
	{
		Method methodToInvoke = ContainerConversationUtils.class.getMethod("privateHydrateContainerStartupOptions", Container.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[1];
		args[0] = container;
		conversationInterceptor.invoke(methodToInvoke, this, args);
	}

	public ConversationTokenProvider privateHydrateContainerStartupOptions(Container container)
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		ContainerService containerService = TmcdbContextFactory.INSTANCE.getContainerService();
		containerService.hydrateContainerStartupOptions(container);
		return retVal;
	}
	
	public void saveOrUpdateContainer(Container container, ConversationToken token)
	   throws Exception 
	{
		Method methodToInvoke = ContainerConversationUtils.class.getMethod("privateSaveOrUpdateContainer", Container.class, ConversationToken.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[2];
		args[0] = container;
		args[1] = token;
		conversationInterceptor.invoke(methodToInvoke, this, args);
	}
	
	public void saveOrUpdateContainer(Container container)
		throws Exception 
	{
		saveOrUpdateContainer(container, ConversationToken.CONVERSATION_COMPLETED);
	}

	public ConversationTokenProvider privateSaveOrUpdateContainer(Container container, ConversationToken token) {
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(token);
		ContainerService containerService = TmcdbContextFactory.INSTANCE.getContainerService();
		containerService.update(container);
		return retVal;
	}

	public ConversationTokenProvider privateDeleteContainer(Container container, ConversationToken token) {
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(token);
		ContainerService containerService = TmcdbContextFactory.INSTANCE.getContainerService();
		containerService.delete(container);
		return retVal;
	}
	
	public ConversationTokenProvider privateDeleteContainerStartupOption(ContainerStartupOption opt, ConversationToken token) {
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(token);
		ContainerStartupOptionService service = TmcdbContextFactory.INSTANCE.getContainerStartupOptionService();
		opt.getContainer().getContainerStartupOptions().remove(opt);
		service.update(opt.getContainer());
		service.delete(opt);
		return retVal;
	}
	
	public ConversationTokenProvider privateFindContainer(Container container, ContainerHolder holder)
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		ContainerService containerService = TmcdbContextFactory.INSTANCE.getContainerService();
		holder.setContainer( containerService.findContainer(container) );
		return retVal;
	}
	
	public void saveOrUpdateContainerStartupOption(ContainerStartupOption option) 
	throws Exception {
		Method methodToInvoke = ContainerConversationUtils.class.getMethod("privateSaveOrUpdateContainerStartupOption", ContainerStartupOption.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[1];
		args[0] = option;
		conversationInterceptor.invoke(methodToInvoke, this, args);
	}

	public ConversationTokenProvider privateSaveOrUpdateContainerStartupOption(ContainerStartupOption opt) {
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		ContainerStartupOptionService containerStartupOptionService = TmcdbContextFactory.INSTANCE.getContainerStartupOptionService();
		containerStartupOptionService.update(opt);
		return retVal;
	}

	public void hydrateContainers(Configuration configuration) 
	   throws Exception 
	{
		Method methodToInvoke = ContainerConversationUtils.class.getMethod("privateHydrateContainers", Configuration.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[1];
		args[0] = configuration;
		conversationInterceptor.invoke(methodToInvoke, this, args);
	}
	
	public ConversationTokenProvider privateHydrateContainers(Configuration config)
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		SwConfigurationService configService = TmcdbContextFactory.INSTANCE.getSwConfigurationService();
		configService.hydrateContainers(config);
		return retVal;
	}

	/**
	 * @param element the element for which we want to run in an attached mode
	 * @param runnable the runnable to run
	 */
	public void runWithAttachedObject(Object element, Runnable runnable) throws Exception 
	{
		Method methodToInvoke = ContainerConversationUtils.class.getMethod("privateRunWithAttachedObject", Object.class, Runnable.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[2];
		args[0] = element;
		args[1] = runnable;
		conversationInterceptor.invoke(methodToInvoke, this, args);
	}

	public ConversationTokenProvider privateRunWithAttachedObject(Object obj, Runnable runnable)
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_PENDING);
		ContainerService service = TmcdbContextFactory.INSTANCE.getContainerService();
		service.runWithAttachedObject(obj, runnable);
		return retVal;
	}

	/**
	 * @param criteria
	 * @param object
	 * @return
	 */
	public List<?> find(List<Object> searchCriteria, List<Object> orderCriteria) throws Exception 
	{
		Method methodToInvoke = ContainerConversationUtils.class.getMethod("privateFind", List.class, List.class, ContainerListHolder.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[3];
		args[0] = searchCriteria;
		args[1] = orderCriteria;
		ContainerListHolder resultHolder = new ContainerListHolder();
		args[2] = resultHolder;
		conversationInterceptor.invoke(methodToInvoke, this, args);
		return resultHolder.getContainers();
	}

	@SuppressWarnings("unchecked")
	public ConversationTokenProvider privateFind(List searchCriteria, List orderCriteria, ContainerListHolder resultHolder)
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		ContainerService service = TmcdbContextFactory.INSTANCE.getContainerService();
		resultHolder.setContainers((List<Container>) service.find(searchCriteria, orderCriteria));
		return retVal;
	}
	
	private class ContainerListHolder 
	{
		private List<Container> conts;
		public List<Container> getContainers() { return this.conts; }
		public void setContainers(List<Container> conts) { this.conts = conts; }
	}
}
