/*******************************************************************************
  * ALMA - Atacama Large Millimeter Array
  * Copyright (c) NRAO - National Radio Astronomy Observatory, 2012
  * (in the framework of the ALMA collaboration).
  * All rights reserved.
  *
  * This library is free software; you can redistribute it and/or
  * modify it under the terms of the GNU Lesser General Public
  * License as published by the Free Software Foundation; either
  * version 2.1 of the License, or (at your option) any later version.
  *
  * This library is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  * Lesser General Public License for more details.
  *
  * You should have received a copy of the GNU Lesser General Public
  * License along with this library; if not, write to the Free Software
  * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 
*******************************************************************************/
package alma.obops.tmcdbgui.utils.conversation;

import java.lang.reflect.Method;
import java.util.List;

import alma.acs.tmcdb.Computer;
import alma.acs.tmcdb.Configuration;
import alma.obops.dam.config.TmcdbContextFactory;
import alma.obops.dam.tmcdb.service.ComputerService;
import alma.obops.dam.tmcdb.service.SwConfigurationService;
import alma.obops.dam.utils.ConversationInterceptor;
import alma.obops.dam.utils.ConversationTokenProvider;
import alma.obops.dam.utils.ConversationTokenProviderAdapter;
import alma.obops.dam.utils.ConversationTokenProvider.ConversationToken;

/**
 * @author sharring
 *
 */
public class ComputerConversationUtils 
{
	private static ComputerConversationUtils singletonInstance;

	private ComputerConversationUtils() 
	{
	}

	public static synchronized ComputerConversationUtils getInstance()
	{
		if(null == singletonInstance)
		{
			singletonInstance = new ComputerConversationUtils();
		}

		return singletonInstance;
	}
	
	public void hydrateAcsServices(Computer computer) 
	   throws Exception 
	{
		Method methodToInvoke = ComputerConversationUtils.class.getMethod("privateHydrateAcsServices", Computer.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[1];
		args[0] = computer;
		conversationInterceptor.invoke(methodToInvoke, this, args);
	}
	
	public ConversationTokenProvider privateHydrateAcsServices(Computer computer)
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		ComputerService computerService = TmcdbContextFactory.INSTANCE.getComputerService();
		computerService.hydrateAcsServices(computer);
		return retVal;
	}
	
	public void hydrateComputer(Computer computer) throws Exception
	{
		Method methodToInvoke = ComputerConversationUtils.class.getMethod("privateHydrateComputer", Computer.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[1];
		args[0] = computer;
		conversationInterceptor.invoke(methodToInvoke, this, args);
	}

	public ConversationTokenProvider privateHydrateComputer(Computer computer)
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		ComputerService computerService = TmcdbContextFactory.INSTANCE.getComputerService();
		computerService.hydrate(computer);
		return retVal;
	}
	
	/************************** Computer related methods **************************************************************/

	public void hydrateContainers(Computer computer) 
	   throws Exception 
	{
		Method methodToInvoke = ComputerConversationUtils.class.getMethod("privateHydrateContainers", Computer.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[1];
		args[0] = computer;
		conversationInterceptor.invoke(methodToInvoke, this, args);
	}
	
	public ConversationTokenProvider privateHydrateContainers(Computer computer)
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		ComputerService computerService = TmcdbContextFactory.INSTANCE.getComputerService();
		computerService.hydrateContainers(computer);
		return retVal;
	}

	public Computer readComputerById(Integer id)
	throws Exception {
		Method methodToInvoke = ComputerConversationUtils.class.getMethod("privateReadComputerById", Integer.class, ComputerHolder.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		ComputerHolder holder = new ComputerHolder();
		Object[] args = new Object[2];
		args[0] = id;
		args[1] = holder;
		conversationInterceptor.invoke(methodToInvoke, this, args);
		return holder.getComputer();
	}

	public ConversationTokenProvider privateReadComputerById(Integer id, ComputerHolder holder)
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		ComputerService computerService = TmcdbContextFactory.INSTANCE.getComputerService();
		Computer c = (Computer)computerService.read(id);
		computerService.hydrateConfiguration(c);
		holder.setComputer( c );
		return retVal;
	}
	
	public void saveOrUpdateComputer(Computer comp)
	throws Exception {
		Method methodToInvoke = ComputerConversationUtils.class.getMethod("privateSaveOrUpdateComputer", Computer.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[1];
		args[0] = comp;
		conversationInterceptor.invoke(methodToInvoke, this, args);
	}

	public ConversationTokenProvider privateSaveOrUpdateComputer(Computer comp) {
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		ComputerService computerService = TmcdbContextFactory.INSTANCE.getComputerService();
		computerService.update(comp);
		return retVal;
	}

	public ConversationTokenProvider privateDeleteComputer(Computer computer, ConversationToken token)
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(token);
		ComputerService computerService = TmcdbContextFactory.INSTANCE.getComputerService();
		computerService.delete(computer);
		return retVal;
	}
	
	public void hydrateComputers(Configuration configuration) 
	   throws Exception 
	{
		Method methodToInvoke = ComputerConversationUtils.class.getMethod("privateHydrateComputers", Configuration.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[1];
		args[0] = configuration;
		conversationInterceptor.invoke(methodToInvoke, this, args);
	}
	
	public ConversationTokenProvider privateHydrateComputers(Configuration config)
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		SwConfigurationService configService = TmcdbContextFactory.INSTANCE.getSwConfigurationService();
		configService.hydrateComputers(config);
		return retVal;
	}
	
	public ConversationTokenProvider privateFindComputer(Computer computer, ComputerHolder holder)
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		ComputerService computerService = TmcdbContextFactory.INSTANCE.getComputerService();
		holder.setComputer( computerService.findComputer(computer) );
		return retVal;
	}

	/**
	 * @param element the element for which we want to run in an attached mode
	 * @param runnable the runnable to run
	 */
	public void runWithAttachedObject(Object element, Runnable runnable) throws Exception 
	{
		Method methodToInvoke = ComputerConversationUtils.class.getMethod("privateRunWithAttachedObject", Object.class, Runnable.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[2];
		args[0] = element;
		args[1] = runnable;
		conversationInterceptor.invoke(methodToInvoke, this, args);
	}

	public ConversationTokenProvider privateRunWithAttachedObject(Object obj, Runnable runnable)
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_PENDING);
		ComputerService service = TmcdbContextFactory.INSTANCE.getComputerService();
		service.runWithAttachedObject(obj, runnable);
		return retVal;
	}

	/**
	 * @param criteria
	 * @param object
	 * @return
	 */
	public List<?> find(List<Object> searchCriteria, List<Object> orderCriteria) throws Exception 
	{
		Method methodToInvoke = ComputerConversationUtils.class.getMethod("privateFind", List.class, List.class, ComputerListHolder.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[3];
		args[0] = searchCriteria;
		args[1] = orderCriteria;
		ComputerListHolder resultHolder = new ComputerListHolder();
		args[2] = resultHolder;
		conversationInterceptor.invoke(methodToInvoke, this, args);
		return resultHolder.getComputers();
	}

	@SuppressWarnings("unchecked")
	public ConversationTokenProvider privateFind(List searchCriteria, List orderCriteria, ComputerListHolder resultHolder)
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		ComputerService service = TmcdbContextFactory.INSTANCE.getComputerService();
		resultHolder.setComputers((List<Computer>) service.find(searchCriteria, orderCriteria));
		return retVal;
	}
	
	private class ComputerListHolder 
	{
		private List<Computer> comps;
		public List<Computer> getComputers() { return this.comps; }
		public void setComputers(List<Computer> comps) { this.comps = comps; }
	}
}
