/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) NRAO - National Radio Astronomy Observatory, 2012
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA

 *******************************************************************************/
package alma.obops.tmcdbgui.utils.conversation;

import java.lang.reflect.Method;
import java.util.List;

import alma.acs.tmcdb.Component;
import alma.acs.tmcdb.ComponentType;
import alma.acs.tmcdb.Configuration;
import alma.acs.tmcdb.Container;
import alma.obops.dam.config.TmcdbContextFactory;
import alma.obops.dam.tmcdb.service.ComponentService;
import alma.obops.dam.tmcdb.service.ContainerService;
import alma.obops.dam.tmcdb.service.SwConfigurationService;
import alma.obops.dam.utils.ConversationInterceptor;
import alma.obops.dam.utils.ConversationTokenProvider;
import alma.obops.dam.utils.ConversationTokenProviderAdapter;
import alma.obops.dam.utils.ProgressMonitor;
import alma.obops.dam.utils.ConversationTokenProvider.ConversationToken;
import alma.tmcdb.domain.HwConfiguration;

/**
 * @author sharring
 *
 */
public class ComponentConversationUtils 
{
	private static ComponentConversationUtils singletonInstance;

	private ComponentConversationUtils() 
	{
	}

	public static synchronized ComponentConversationUtils getInstance()
	{
		if(null == singletonInstance)
		{
			singletonInstance = new ComponentConversationUtils();
		}

		return singletonInstance;
	}

	public void cloneAndStoreComponent(Component component, Configuration targetConfig, String newName, String newPath) throws Exception {
		Method methodToInvoke = ComponentConversationUtils.class.getMethod("privateCloneAndStoreComponent", Component.class, Configuration.class, String.class, String.class);
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[4];
		args[0] = component;
		args[1] = targetConfig;
		args[2] = newName;
		args[3] = newPath;
		conversationInterceptor.invoke(methodToInvoke, this, args);
	}

	public ConversationTokenProvider privateCloneAndStoreComponent (Component component, Configuration targetConfig, String newName, String newPath) {
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		ComponentService componentService = TmcdbContextFactory.INSTANCE.getComponentService();
		componentService.cloneAndStoreComponentInConfiguration(component, targetConfig, newName, newPath);
		return retVal;
	}
	
	/**
	 * @param comp the component for which we wish to hydrate the associated/owning container
	 */
	public void hydrateContainer(Component comp) throws Exception  
	{		
		Method methodToInvoke = ComponentConversationUtils.class.getMethod("privateHydrateContainer", Component.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[1];
		args[0] = comp;
		conversationInterceptor.invoke(methodToInvoke, this, args);
	}
	
	public ConversationTokenProvider privateHydrateContainer(Component comp)
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		ComponentService service = TmcdbContextFactory.INSTANCE.getComponentService();
		service.hydrateContainer(comp);
		return retVal;
	}

	public void hydrateComponents(Container container) 
	throws Exception 
	{
		Method methodToInvoke = ComponentConversationUtils.class.getMethod("privateHydrateComponents", Container.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[1];
		args[0] = container;
		conversationInterceptor.invoke(methodToInvoke, this, args);
	}

	public ConversationTokenProvider privateHydrateComponents(Container container)
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		ContainerService containerService = TmcdbContextFactory.INSTANCE.getContainerService();
		containerService.hydrateComponents(container);
		return retVal;
	}

	static class ComponentTypesHolder {
		private List<ComponentType> _types;
		List<ComponentType> getComponentTypes() { return _types; }
		void setComponentTypes(List<ComponentType> types ) { _types = types; }
	}

	public void bulkUpdateComponents(Component[] selectedComponents,
			String[] objectProperties, Object[] values,
			ProgressMonitor eclipseMonitor) 
	throws Exception 
	{
		Method methodToInvoke = ComponentConversationUtils.class.getMethod("privateBulkUpdateComponents", Component[].class, String[].class, Object[].class, ProgressMonitor.class);
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[4];
		args[0] = selectedComponents;
		args[1] = objectProperties;
		args[2] = values;
		args[3] = eclipseMonitor;
		conversationInterceptor.invoke(methodToInvoke, this, args);
	}

	public ConversationTokenProvider privateBulkUpdateComponents(Component[] componentsToUpdate, String[] objectProperties, Object[] values, ProgressMonitor eclipseMonitor) {
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		ComponentService service = TmcdbContextFactory.INSTANCE.getComponentService();
		service.bulkUpdateComponents(componentsToUpdate, objectProperties, values, eclipseMonitor);
		return retVal;
	}

	public Component readComponentById(Integer id)
	throws Exception {
		Method methodToInvoke = ComponentConversationUtils.class.getMethod("privateReadComponentById", Integer.class, ComponentHolder.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		ComponentHolder holder = new ComponentHolder();
		Object[] args = new Object[2];
		args[0] = id;
		args[1] = holder;
		conversationInterceptor.invoke(methodToInvoke, this, args);
		return holder.getComponent();
	}

	public ConversationTokenProvider privateReadComponentById(Integer id, ComponentHolder holder)
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		ComponentService componentService = TmcdbContextFactory.INSTANCE.getComponentService();
		Component c = (Component)componentService.read(id);
		componentService.hydrateConfiguration(c);
		holder.setComponent( c );
		return retVal;
	}

	public void hydrateComponentType(Component comp)
	throws Exception {
		Method methodToInvoke = ComponentConversationUtils.class.getMethod("privateHydrateComponentType", Component.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[1];
		args[0] = comp;
		conversationInterceptor.invoke(methodToInvoke, this, args);
	}

	public ConversationTokenProvider privateHydrateComponentType(Component comp) {
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		ComponentService componentService = TmcdbContextFactory.INSTANCE.getComponentService();
		componentService.hydrateComponentType(comp);
		return retVal;
	}

	public void saveOrUpdateComponent(Component comp)
	throws Exception {
		Method methodToInvoke = ComponentConversationUtils.class.getMethod("privateSaveOrUpdateComponent", Component.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[1];
		args[0] = comp;
		conversationInterceptor.invoke(methodToInvoke, this, args);
	}

	public ConversationTokenProvider privateSaveOrUpdateComponent(Component comp) {
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		ComponentService componentService = TmcdbContextFactory.INSTANCE.getComponentService();
		componentService.update(comp);
		return retVal;
	}

	public ConversationTokenProvider privateDeleteComponent(Component comp, ConversationToken token) {
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(token);
		ComponentService componentService = TmcdbContextFactory.INSTANCE.getComponentService();
		componentService.delete(comp);
		return retVal;
	}

	public List<Component> findComponentByNamePrefixWithinConfiguration(String[] prefixes, Configuration swConfig)
	throws Exception 
	{
		Method methodToInvoke = ComponentConversationUtils.class.
		getMethod("privateComponentFindByNamePrefixWithinConfiguration", String[].class, Configuration.class, ComponentListHolder.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[3];
		args[0] = prefixes;
		args[1] = swConfig;
		args[2] = new ComponentListHolder();
		ComponentListHolder resultHolder = (ComponentListHolder)args[2];
		conversationInterceptor.invoke(methodToInvoke, this, args);
		List<Component> retVal = resultHolder.getComponents();	
		return retVal;
	}

	public ConversationTokenProvider privateComponentFindByNamePrefixWithinConfiguration(String[] prefixes, 
			Configuration swConfig, ComponentListHolder resultHolder) 
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		ComponentService componentService = TmcdbContextFactory.INSTANCE.getComponentService();
		resultHolder.setComponents(componentService.findByNamePrefixWithinConfiguration(prefixes, swConfig));
		return retVal;
	}

	public ConversationTokenProvider privateFindComponent(Component component, ComponentHolder holder)
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		ComponentService componentService = TmcdbContextFactory.INSTANCE.getComponentService();
		holder.setComponent( componentService.findComponent(component) );
		return retVal;
	}


	private class ComponentListHolder
	{
		private List<Component> components;

		public List<Component> getComponents() {
			return components;
		}

		public void setComponents(List<Component> components) {
			this.components = components;
		}
	}

	public void hydrateComponents(Configuration configuration) 
	throws Exception 
	{
		Method methodToInvoke = ComponentConversationUtils.class.getMethod("privateHydrateComponents", Configuration.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[1];
		args[0] = configuration;
		conversationInterceptor.invoke(methodToInvoke, this, args);
	}

	public ConversationTokenProvider privateHydrateComponents(Configuration config)
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_PENDING);
		SwConfigurationService configService = TmcdbContextFactory.INSTANCE.getSwConfigurationService();
		configService.hydrateComponents(config);
		return retVal;
	}
	
	public List<Component> findComponentsByComponentTypeId(ComponentType componentType, Configuration configuration, ProgressMonitor monitor)
	   throws Exception
	{
		Method methodToInvoke = ComponentConversationUtils.class.getMethod("privateFindComponentsByComponentTypeId", ComponentType.class, Configuration.class, ComponentListHolder.class, ProgressMonitor.class);
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[4];
		args[0] = componentType;
		args[1] = configuration;
		ComponentListHolder resultHolder = new ComponentListHolder();
		args[2] = resultHolder;
		args[3] = monitor;
		conversationInterceptor.invoke(methodToInvoke, this, args);
		return resultHolder.getComponents();
	}

	public ConversationTokenProvider privateFindComponentsByComponentTypeId(ComponentType type, Configuration configuration, ComponentListHolder resultHolder, ProgressMonitor monitor)
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		ComponentService cService =  TmcdbContextFactory.INSTANCE.getComponentService();
		List<Component> comps = cService.findByComponentTypeIdWithinConfiguration(type, configuration);
		monitor.beginTask("Getting components", comps.size());
		for (Component component : comps) {
			cService.hydrate(component);
			monitor.worked(1);
		}
		monitor.done();
		resultHolder.setComponents(comps);
		return retVal;
	}
	
	public List<Component> findComponentByPathAndNameWithinConfiguration(
			String path, String name, Configuration swConfiguration) 
			throws Exception
	{
		Method methodToInvoke = ComponentConversationUtils.class.getMethod("privateFindComponentByPathAndNameWithinConfiguration", String.class, String.class, Configuration.class, ComponentListHolder.class);
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[4];
		args[0] = path;
		args[1] = name;
		args[2] = swConfiguration;
		ComponentListHolder resultHolder = new ComponentListHolder();
		args[3] = resultHolder;
		conversationInterceptor.invoke(methodToInvoke, this, args);
		return resultHolder.getComponents();
	}
	
	public ConversationTokenProvider privateFindComponentByPathAndNameWithinConfiguration(String path, String name,  
			Configuration swConfig, ComponentListHolder resultHolder) 
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		ComponentService componentService = TmcdbContextFactory.INSTANCE.getComponentService();
		resultHolder.setComponents(componentService.findByParametersWithinConfiguration(new String[] { "path", "componentName" }, new String[] { path, name} , swConfig));
		return retVal;
	}

	public Component findComponentByComponentTypeId(ComponentType componentType, HwConfiguration configuration)
	   throws Exception
	{
		Method methodToInvoke = ComponentConversationUtils.class.getMethod("privateFindComponentByComponentTypeId", ComponentType.class, Configuration.class, ComponentHolder.class);
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[3];
		args[0] = componentType;
		args[1] = configuration.getSwConfiguration();
		ComponentHolder resultHolder = new ComponentHolder();
		args[2] = resultHolder;
		conversationInterceptor.invoke(methodToInvoke, this, args);
		return resultHolder.getComponent();
	}

	public ConversationTokenProvider privateFindComponentByComponentTypeId(ComponentType type, Configuration configuration, ComponentHolder resultHolder) 
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		ComponentService cService =  TmcdbContextFactory.INSTANCE.getComponentService();			
		List<Component> comps = cService.findByComponentTypeIdWithinConfiguration(type, configuration);	
		Component comp = (comps != null && comps.size() > 0) ? comps.get(0) : null;
		cService.hydrate(comp);
		resultHolder.setComponent(comp);
		return retVal;
	}

	public Component[] getComponentsByComponentType(Configuration configuration, ComponentType componentType) throws Exception 
	{
		Method methodToInvoke = ComponentConversationUtils.class.getMethod("privateGetComponentsByComponentType", Configuration.class, ComponentType.class, ComponentArrayHolder.class);
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		
		Object[] args = new Object[3];
		args[0] = configuration;
		args[1] = componentType;
		ComponentArrayHolder componentArrayHolder = new ComponentArrayHolder();
		args[2] = componentArrayHolder;
		
		conversationInterceptor.invoke(methodToInvoke, this, args);
		return componentArrayHolder.getComponents();
	}
	
	public ConversationTokenProvider privateGetComponentsByComponentType(Configuration config, ComponentType componentType, ComponentArrayHolder retValHolder)
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		ComponentService service = TmcdbContextFactory.INSTANCE.getComponentService();
		List<Component> components = service.findByComponentTypeIdWithinConfiguration(componentType, config);
		retValHolder.setComponents(components.toArray(new Component[0]));
		return retVal;
	}
	
	private class ComponentArrayHolder 
	{
		Component[] components;
		
		public Component[] getComponents() { return this.components; }
		public void setComponents(Component[] comps) { this.components = comps; }
	}

	/**
	 * @param element the element for which we want to run in an attached mode
	 * @param runnable the runnable to run
	 */
	public void runWithAttachedObject(Object element, Runnable runnable) throws Exception 
	{
		Method methodToInvoke = ComponentConversationUtils.class.getMethod("privateRunWithAttachedObject", Object.class, Runnable.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[2];
		args[0] = element;
		args[1] = runnable;
		conversationInterceptor.invoke(methodToInvoke, this, args);
	}

	public ConversationTokenProvider privateRunWithAttachedObject(Object obj, Runnable runnable)
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_PENDING);
		ComponentService service = TmcdbContextFactory.INSTANCE.getComponentService();
		service.runWithAttachedObject(obj, runnable);
		return retVal;
	}

	/**
	 * @param criteria
	 * @param object
	 * @return
	 */
	public List<?> find(List<Object> searchCriteria, List<Object> orderCriteria) throws Exception 
	{
		Method methodToInvoke = ComponentConversationUtils.class.getMethod("privateFind", List.class, List.class, ComponentListHolder.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[3];
		args[0] = searchCriteria;
		args[1] = orderCriteria;
		ComponentListHolder resultHolder = new ComponentListHolder();
		args[2] = resultHolder;
		conversationInterceptor.invoke(methodToInvoke, this, args);
		return resultHolder.getComponents();
	}

	@SuppressWarnings("unchecked")
	public ConversationTokenProvider privateFind(List searchCriteria, List orderCriteria, ComponentListHolder resultHolder)
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		ComponentService service = TmcdbContextFactory.INSTANCE.getComponentService();
		resultHolder.setComponents((List<Component>) service.find(searchCriteria, orderCriteria));
		return retVal;
	}
}
