/*******************************************************************************
  * ALMA - Atacama Large Millimeter Array
  * Copyright (c) NRAO - National Radio Astronomy Observatory, 2012
  * (in the framework of the ALMA collaboration).
  * All rights reserved.
  *
  * This library is free software; you can redistribute it and/or
  * modify it under the terms of the GNU Lesser General Public
  * License as published by the Free Software Foundation; either
  * version 2.1 of the License, or (at your option) any later version.
  *
  * This library is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  * Lesser General Public License for more details.
  *
  * You should have received a copy of the GNU Lesser General Public
  * License along with this library; if not, write to the Free Software
  * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 
*******************************************************************************/
package alma.obops.tmcdbgui.utils.conversation;

import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.List;
import java.util.Set;

import alma.obops.dam.config.TmcdbContextFactory;
import alma.obops.dam.tmcdb.service.AntennaService;
import alma.obops.dam.tmcdb.service.AntennaToPadService;
import alma.obops.dam.tmcdb.service.ConfigurationService;
import alma.obops.dam.tmcdb.service.FrontEndService;
import alma.obops.dam.tmcdb.service.HolographyTowerService;
import alma.obops.dam.tmcdb.service.HolographyTowerToPadService;
import alma.obops.dam.tmcdb.service.PadService;
import alma.obops.dam.tmcdb.service.WeatherStationControllerService;
import alma.obops.dam.utils.ConversationInterceptor;
import alma.obops.dam.utils.ConversationTokenProvider;
import alma.obops.dam.utils.ConversationTokenProviderAdapter;
import alma.obops.dam.utils.ConversationTokenProvider.ConversationToken;
import alma.tmcdb.domain.AOSTiming;
import alma.tmcdb.domain.Antenna;
import alma.tmcdb.domain.AntennaToPad;
import alma.tmcdb.domain.BaseElement;
import alma.tmcdb.domain.BaseElementType;
import alma.tmcdb.domain.CentralLO;
import alma.tmcdb.domain.FrontEnd;
import alma.tmcdb.domain.HolographyTower;
import alma.tmcdb.domain.HolographyTowerToPad;
import alma.tmcdb.domain.HwConfiguration;
import alma.tmcdb.domain.Pad;
import alma.tmcdb.domain.WeatherStationController;
import alma.tmcdb.history.HistoryRecord;

/**
 * @author sharring
 *
 */
public class BaseElementConversationUtils 
{
	private static BaseElementConversationUtils singletonInstance;

	private BaseElementConversationUtils() 
	{
	}

	public static synchronized BaseElementConversationUtils getInstance()
	{
		if(null == singletonInstance)
		{
			singletonInstance = new BaseElementConversationUtils();
		}

		return singletonInstance;
	}
	
	public void hydrateAntenna(Antenna antennaToCopy) throws Exception
	{
		Method methodToInvoke = BaseElementConversationUtils.class.getMethod("privateHydrateAntenna", Antenna.class);
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[1];
		args[0] = antennaToCopy;
		conversationInterceptor.invoke(methodToInvoke, this, args);
	}
	
	public ConversationTokenProvider privateHydrateAntenna(Antenna antenna)
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		AntennaService antService =  TmcdbContextFactory.INSTANCE.getAntennaService();	   
		antService.hydrate(antenna);
		return retVal;
	}
	
	public ConversationTokenProvider privateDeleteAntenna(Antenna ant, ConversationToken token) {
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(token);
		AntennaService antService = TmcdbContextFactory.INSTANCE.getAntennaService();
		antService.delete(ant);
		return retVal;
	}
	
	private class AntennaHolder
	{
		private Antenna antenna;
		
		public Antenna getAntenna() {
			return antenna;
		}
		
		public void setAntenna(Antenna antenna) {
			this.antenna = antenna;
		}
	}
	
	public Antenna findAntennaByName(Long configId, String name) 
	   throws Exception 
	{
		Method methodToInvoke = BaseElementConversationUtils.class.getMethod("privateFindAntennaByName", Long.class, String.class, AntennaHolder.class);
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[3];
		args[0] = configId;
		args[1] = name;
		AntennaHolder result = new AntennaHolder();
		args[2] = result;
		conversationInterceptor.invoke(methodToInvoke, this, args);
		Antenna retVal = result.getAntenna();
		return retVal;
	}
	
	public ConversationTokenProvider privateFindAntennaByName(Long configId, String name, AntennaHolder resultHolder) 
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);

		ConfigurationService configService =  TmcdbContextFactory.INSTANCE.getConfigurationService();
		HwConfiguration config = (HwConfiguration)configService.read(configId);
		Set<BaseElement> baseElements = config.getBaseElements();
		for(BaseElement baseElement: baseElements ) {
			if(baseElement.getType().equals(BaseElementType.Antenna) && baseElement.getName().equals(name)) {
				resultHolder.setAntenna((Antenna)baseElement);
				break;
			}
		}
		// TODO: add a service method to do the search, so we don't have to do a linear brute-force search?
		// But, this is currently only used in testing; so it probably doesn't matter (yet).
		return retVal;
	}
	
	public BaseElement copyAntenna(Antenna baseElement, String name, HwConfiguration addToConfiguration)
	   throws Exception 
	{
		Method methodToInvoke = BaseElementConversationUtils.class.getMethod("privateCopyBaseElement", 
				BaseElement.class, String.class, HwConfiguration.class, BaseElementHolder.class);
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		
		Object[] args = new Object[4];
		args[0] = baseElement;
		args[1] = name;
		args[2] = addToConfiguration;
		BaseElementHolder resultHolder = new BaseElementHolder();
		args[3] = resultHolder;
		conversationInterceptor.invoke(methodToInvoke, this, args);
		return resultHolder.getBaseElement();
	}
	
	private class BaseElementHolder 
	{
		private BaseElement baseElement;

		public BaseElement getBaseElement() {
			return this.baseElement;
		}

		public void setBaseElement(BaseElement be) {
			this.baseElement = be;
		}
	}
	
	public BaseElement cloneBaseElement(BaseElement baseElement, String name) throws Exception 
	{
		Method methodToInvoke = BaseElementConversationUtils.class.getMethod("privateCloneBaseElement", 
				BaseElement.class, String.class, BaseElementHolder.class);
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[3];
		args[0] = baseElement;
		args[1] = name;
		BaseElementHolder resultHolder = new BaseElementHolder();
		args[2] = resultHolder;
		conversationInterceptor.invoke(methodToInvoke, this, args);
		return resultHolder.getBaseElement();
	}
	
	public ConversationTokenProvider privateCloneBaseElement(BaseElement baseElementToClone, String clonedBaseElementName, BaseElementHolder resultHolder)
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		ConfigurationService configService = TmcdbContextFactory.INSTANCE.getConfigurationService();
		BaseElement clonedBaseElement = configService.cloneBaseElement(baseElementToClone, clonedBaseElementName);
		resultHolder.setBaseElement(clonedBaseElement);
		return retVal;	
	}
	
	public void copySwItemsForBaseElement(BaseElement baseElement, HwConfiguration addToConfiguration) 
	 throws Exception 
	 {
		Method methodToInvoke = BaseElementConversationUtils.class.getMethod("privateCopySwItemsForBaseElement", 
				BaseElement.class, HwConfiguration.class);
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[2];
		args[0] = baseElement;
		args[1] = addToConfiguration;
		conversationInterceptor.invokeWithDeferredConstraints(methodToInvoke, this, args);
	 }
	
	public ConversationTokenProvider privateCopySwItemsForBaseElement(BaseElement baseElementToCopy,
			HwConfiguration addToConfiguration)
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		ConfigurationService configService = TmcdbContextFactory.INSTANCE.getConfigurationService();
		configService.copySwItemsForBaseElement(baseElementToCopy, addToConfiguration);
		return retVal;	
	}
	
	public BaseElement copyBaseElement(BaseElement baseElement, String name, HwConfiguration addToConfiguration) 
	   throws Exception 
	{
		Method methodToInvoke = BaseElementConversationUtils.class.getMethod("privateCopyBaseElement", 
				BaseElement.class, String.class, HwConfiguration.class, BaseElementHolder.class);
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[4];
		args[0] = baseElement;
		args[1] = name;
		args[2] = addToConfiguration;
		BaseElementHolder resultHolder = new BaseElementHolder();
		args[3] = resultHolder;
		conversationInterceptor.invoke(methodToInvoke, this, args);
		return resultHolder.getBaseElement();
	}
	
	public ConversationTokenProvider privateCopyBaseElement(BaseElement baseElementToCopy, String copiedBaseElementName, 
			HwConfiguration addToConfiguration, BaseElementHolder resultHolder)
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		ConfigurationService configService = TmcdbContextFactory.INSTANCE.getConfigurationService();
		BaseElement copiedBaseElement = configService.copyBaseElement(baseElementToCopy, copiedBaseElementName, addToConfiguration);
		resultHolder.setBaseElement(copiedBaseElement);
		return retVal;	
	}
	
	/**
	 * @param antenna the antenna for which we want to find a2p assignments
	 * @return an array of a2p assignments related to the given antenna
	 * @throws Exception if there's a problem
	 */
	public AntennaToPad findOpenAntennaToPadAssignmentForAntennaInGlobal(Antenna antenna) throws Exception 
	{
		Method methodToInvoke = BaseElementConversationUtils.class.getMethod("privateFindOpenAntennaToPadAssignmentForAntennaInGlobal", 
				Antenna.class, AntennaToPadHolder.class);
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[2];
		args[0] = antenna;
		AntennaToPadHolder resultHolder = new AntennaToPadHolder();
		args[1] = resultHolder;
		conversationInterceptor.invoke(methodToInvoke, this, args);
		return resultHolder.getAntennaToPad();
	}
	
	public ConversationTokenProvider privateFindOpenAntennaToPadAssignmentForAntennaInGlobal(Antenna antenna, AntennaToPadHolder resultHolder)
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		if(null == antenna.getConfiguration().getGlobalConfiguration()) {
			resultHolder.setAntennaToPad(null);
		} else {
			AntennaToPadService service = TmcdbContextFactory.INSTANCE.getAntennaToPadService();
			List<AntennaToPad> a2ps = service.findCurrentAntennaToPadAssignmentsForAntennaInGlobalConfiguration(antenna);
			if(a2ps.size() > 1) {
				throw new RuntimeException("Too many open a2p assignments!");
			}
			if(null != a2ps && a2ps.size() == 1) {
				resultHolder.setAntennaToPad(a2ps.get(0));
			} else {
				resultHolder.setAntennaToPad(null);
			}
		}
		return retVal;	
	}
	
	private class AntennaToPadHolder 
	{
		private AntennaToPad antennaToPad;
		
		public AntennaToPad getAntennaToPad() {
			return this.antennaToPad;
		}
		
		public void setAntennaToPad(AntennaToPad a2p) {
			this.antennaToPad = a2p;
		}
	}
	
	private class AntennaToPadArrayHolder 
	{
		private AntennaToPad[] antennaToPads;
		
		public AntennaToPad[] getAntennaToPads() {
			return this.antennaToPads;
		}
		
		public void setAntennaToPads(AntennaToPad[] a2ps) {
			this.antennaToPads = a2ps;
		}
	}
	
	/**
	 * @param pad the pad for which we want to find a2p assignments
	 * @return an array of a2p assignments related to the given pad
	 * @throws Exception if there's a problem
	 */
	public AntennaToPad[] findOpenAntennaToPadAssignmentsForPad(Pad pad, HwConfiguration config) throws Exception 
	{
		Method methodToInvoke = BaseElementConversationUtils.class.getMethod("privateFindOpenAntennaToPadAssignmentForPad", 
				Pad.class, HwConfiguration.class, AntennaToPadArrayHolder.class);
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[3];
		args[0] = pad;
		args[1] = config;
		AntennaToPadArrayHolder resultHolder = new AntennaToPadArrayHolder();
		args[2] = resultHolder;
		conversationInterceptor.invoke(methodToInvoke, this, args);
		return resultHolder.getAntennaToPads();
	}
	
	public ConversationTokenProvider privateFindOpenAntennaToPadAssignmentForPad(Pad pad, HwConfiguration config, AntennaToPadArrayHolder resultHolder)
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		AntennaToPadService service = TmcdbContextFactory.INSTANCE.getAntennaToPadService();
		List<AntennaToPad> a2ps = service.findCurrentAntennaToPadAssignmentsForPad(pad, config);
		if(null != a2ps) {
			resultHolder.setAntennaToPads(a2ps.toArray(new AntennaToPad[0]));			
		}
		return retVal;	
	}
	
	/**
	 * This is not actually a conversational method; it's a bit of a hack, but avoids duplicate objects and tricky bugs related to antennatopad assignments...
	 * @param antenna the antenna for which we want to find a2p assignments
	 * @return an array of a2p assignments related to the given antenna
	 * @throws Exception if there's a problem
	 */
	public AntennaToPad[] findOpenAntennaToPadAssignmentsForAntenna(Antenna antenna) throws Exception 
	{
		AntennaToPad[] retVal = new AntennaToPad[0];
		List<AntennaToPad> a2ps = new ArrayList<AntennaToPad>();
		for(BaseElement be: antenna.getConfiguration().getBaseElements()) {
			if(be.getType().equals(BaseElementType.Pad)) {
				Pad padIterated = (Pad)be;
				BaseElementConversationUtils.getInstance().hydratePad(padIterated);
				for(AntennaToPad a2p: padIterated.getScheduledAntennas()) {
					if(a2p.getAntenna().getId().equals(antenna.getId()) && (a2p.getEndTime() == null || a2p.getEndTime().equals(Long.valueOf(0))) ) 
					{
						a2ps.add(a2p);
						break;
					}
				}
			}
		}
		
		retVal = a2ps.toArray(retVal);
		return retVal;
	}
	
	/**
	 * This is not actually a conversational method; it's a bit of a hack, but avoids duplicate objects and tricky bugs related to antennatopad assignments...
	 * @param antenna the antenna for which we want to find a2p assignments
	 * @return an array of a2p assignments related to the given antenna
	 * @throws Exception if there's a problem
	 */
	public AntennaToPad[] findAllAntennaToPadAssignmentsForAntenna(Antenna antenna, HwConfiguration globalConfig) throws Exception 
	{
		// first the local config
		AntennaToPad[] retVal = new AntennaToPad[0];
		List<AntennaToPad> a2ps = new ArrayList<AntennaToPad>();
		for(BaseElement be: antenna.getConfiguration().getBaseElements()) {
			if(be.getType().equals(BaseElementType.Pad)) {
				Pad padIterated = (Pad)be;
				BaseElementConversationUtils.getInstance().hydratePad(padIterated);
				for(AntennaToPad a2p: padIterated.getScheduledAntennas()) {
					if(a2p.getAntenna().getId().equals(antenna.getId())) {
						a2ps.add(a2p);
					}
				}
			}
		}
		
		// now for the global config as well
		if(globalConfig != null) 
		{
			HwConfigurationConversationUtils.getInstance().hydrateBaseElements(globalConfig);
			for(BaseElement be: globalConfig.getBaseElements()) {
				if(be.getType().equals(BaseElementType.Pad)) {
					Pad padIterated = (Pad)be;
					BaseElementConversationUtils.getInstance().hydratePad(padIterated);
					for(AntennaToPad a2p: padIterated.getScheduledAntennas()) {
						if(a2p.getAntenna().getName().equals(antenna.getName())) {
							a2ps.add(a2p);
						}
					}
				}
			}
		}
		
		retVal = a2ps.toArray(retVal);
		return retVal;
	}
	
	public void saveOrUpdateAntenna(Antenna antenna) throws Exception {
		Method methodToInvoke = BaseElementConversationUtils.class.getMethod("privateSaveOrUpdateAntenna", Antenna.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[1];
		args[0] = antenna;
		conversationInterceptor.invoke(methodToInvoke, this, args);
	}
	
	
	public ConversationTokenProvider privateSaveOrUpdateAntenna(Antenna antenna) {
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		AntennaService antennaService = TmcdbContextFactory.INSTANCE.getAntennaService();
		antennaService.update(antenna);
		return retVal;
	}
	
	public Pad getCurrentlyAssignedPad(Antenna antenna) throws Exception
	{
		Method methodToInvoke = BaseElementConversationUtils.class.getMethod("privateGetCurrentlyAssignedPad", Antenna.class, PadHolder.class);
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[2];
		args[0] = antenna;
		PadHolder resultHolder = new PadHolder();
		args[1] = resultHolder;
		conversationInterceptor.invoke(methodToInvoke, this, args);
		return resultHolder.getPad();
	}
	
	public ConversationTokenProvider privateGetCurrentlyAssignedPad(Antenna antenna, PadHolder result)
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		
		AntennaToPadService a2pService = TmcdbContextFactory.INSTANCE.getAntennaToPadService();
		List<AntennaToPad> preexistingAssignments = a2pService.findCurrentAntennaToPadAssignmentForAntenna(antenna);
		if(null == preexistingAssignments || preexistingAssignments.size() == 0)
		{
			result.setPad(null);
		}
		else if(preexistingAssignments.size() == 1)
		{
			result.setPad(preexistingAssignments.get(0).getPad());
		} else {
			throw new IllegalStateException("Antenna is assigned to more than one pad; database is in an invalid state");
		}
		return retVal;
	}
	
	private class PadHolder
	{
		private Pad pad;
		
		public Pad getPad() {
			return pad;
		}
		
		public void setPad(Pad pad) {
			this.pad = pad;
		}
	}
	
	public Pad findPadByName(Long configId, String name) 
	   throws Exception 
	{
		Method methodToInvoke = BaseElementConversationUtils.class.getMethod("privateFindPadByName", Long.class, String.class, PadHolder.class);
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[3];
		args[0] = configId;
		args[1] = name;
		PadHolder result = new PadHolder();
		args[2] = result;
		conversationInterceptor.invoke(methodToInvoke, this, args);
		Pad retVal = result.getPad();
		return retVal;
	}
	
	public ConversationTokenProvider privateFindPadByName(Long configId, String name, PadHolder resultHolder) 
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		ConfigurationService configService =  TmcdbContextFactory.INSTANCE.getConfigurationService();
		HwConfiguration config = (HwConfiguration)configService.read(configId);
		Set<BaseElement> baseElements = config.getBaseElements();
		for(BaseElement baseElement: baseElements ) {
			if(baseElement.getType().equals(BaseElementType.Pad) && baseElement.getName().equals(name)) {
				resultHolder.setPad((Pad)baseElement);
				break;
			}
		}
		// TODO: add a service method to do the search, so we don't have to do a linear brute-force search? 
		// But, this is only used in testing at present, so probably doesn't matter (yet).
		return retVal;
	}
	
	public Pad findPadById(Long id) throws Exception 
	{
		Method methodToInvoke = BaseElementConversationUtils.class.getMethod("privateFindPadById", Long.class, PadHolder.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[2];
		args[0] = id;
		PadHolder resultHolder = new PadHolder();
		args[1] = resultHolder;
		conversationInterceptor.invoke(methodToInvoke, this, args);
		return resultHolder.getPad();
	}
	
	public ConversationTokenProvider privateFindPadById(Long id, PadHolder resultHolder)
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		PadService service = TmcdbContextFactory.INSTANCE.getPadService();
		resultHolder.setPad((Pad)service.read(id));
		service.hydrate(resultHolder.getPad());
		ConfigurationService service2 = TmcdbContextFactory.INSTANCE.getConfigurationService();
		service2.hydrate(resultHolder.getPad().getConfiguration());
		return retVal;
	}
	
	public List<HistoryRecord> getPadHistory(Pad pad) throws Exception
	{
		Method methodToInvoke = BaseElementConversationUtils.class.getMethod("privateGetPadHistory", Pad.class, HistoryRecordListHolder.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		HistoryRecordListHolder holder = new HistoryRecordListHolder();
		Object[] args = new Object[2];
		args[0] = pad;
		args[1] = holder;
		conversationInterceptor.invoke(methodToInvoke, this, args);
		return holder.getHistoryRecords();

	}

	public ConversationTokenProvider privateGetPadHistory(Pad pad, HistoryRecordListHolder resultHolder) 
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		PadService service = TmcdbContextFactory.INSTANCE.getPadService();
		List<HistoryRecord> results = service.getHistory(pad);
		resultHolder.setHistoryRecords(results);
		return retVal;
	}
	
	public Pad getHistoricalPad(Pad pad, HistoryRecord clickedRecord) throws Exception 
	{
		Method methodToInvoke = BaseElementConversationUtils.class.getMethod("privateGetHistoricalPad", Pad.class, HistoryRecord.class, PadHolder.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		PadHolder holder = new PadHolder();
		Object[] args = new Object[3];
		args[0] = pad;
		args[1] = clickedRecord;
		args[2] = holder;
		conversationInterceptor.invoke(methodToInvoke, this, args);
		Pad retVal = holder.getPad();
		return retVal;
	}
	
	public ConversationTokenProvider privateGetHistoricalPad(Pad pad, HistoryRecord record, PadHolder resultHolder) 
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		PadService service = TmcdbContextFactory.INSTANCE.getPadService();
		Pad historicalPad = service.getHistoricalPad(pad, record.getVersion());
		resultHolder.setPad(historicalPad);
		return retVal;
	}
	
	/**
	 * @param antenna the antenna for which we want the historical version
	 * @param clickedRecord the history record in question
	 * @return a 'historical' incarnation of the antenna, as it existed in the past
	 * @throws exception if there is a problem
	 */
	public Antenna getHistoricalAntenna(Antenna antenna,
			HistoryRecord clickedRecord) throws Exception 
	{
		Method methodToInvoke = BaseElementConversationUtils.class.getMethod("privateGetHistoricalAntenna", Antenna.class, HistoryRecord.class, AntennaHolder.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		AntennaHolder holder = new AntennaHolder();
		Object[] args = new Object[3];
		args[0] = antenna;
		args[1] = clickedRecord;
		args[2] = holder;
		conversationInterceptor.invoke(methodToInvoke, this, args);
		Antenna retVal = holder.getAntenna();
		return retVal;
	}
	
	public ConversationTokenProvider privateGetHistoricalAntenna(Antenna antenna, HistoryRecord record, AntennaHolder resultHolder) 
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		AntennaService service = TmcdbContextFactory.INSTANCE.getAntennaService();
		Antenna historicalAntenna = service.getHistoricalAntenna(antenna, record.getVersion());
		resultHolder.setAntenna(historicalAntenna);
		return retVal;
	}

	/**
	 * @param antenna the antenna for which we want the history
	 * @return a list of history records for the given antenna
	 * @throws exception if there is a problem
	 */
	public List<HistoryRecord> getAntennaHistory(Antenna antenna) throws Exception  
	{
		Method methodToInvoke = BaseElementConversationUtils.class.getMethod("privateGetAntennaHistory", Antenna.class, HistoryRecordListHolder.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		HistoryRecordListHolder holder = new HistoryRecordListHolder();
		Object[] args = new Object[2];
		args[0] = antenna;
		args[1] = holder;
		conversationInterceptor.invoke(methodToInvoke, this, args);
		return holder.getHistoryRecords();

	}

	public ConversationTokenProvider privateGetAntennaHistory(Antenna antenna, HistoryRecordListHolder resultHolder) 
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		AntennaService service = TmcdbContextFactory.INSTANCE.getAntennaService();
		List<HistoryRecord> results = service.getAntennaHistory(antenna);
		resultHolder.setHistoryRecords(results);
		return retVal;
	}
	
	public void saveOrUpdateBaseElement(BaseElement be) throws Exception 
	{
		if(be instanceof Antenna) {
			BaseElementConversationUtils.getInstance().saveOrUpdateAntenna((Antenna) be);
		} 
		else if(be instanceof Pad) {
			BaseElementConversationUtils.getInstance().saveOrUpdatePad((Pad) be);
		}
		else if(be instanceof FrontEnd) {
			BaseElementConversationUtils.getInstance().saveOrUpdateFrontEnd((FrontEnd) be);
		}
		else if(be instanceof WeatherStationController) {
			BaseElementConversationUtils.getInstance().saveOrUpdateWeatherStation((WeatherStationController) be);
		} 
		else if(be instanceof HolographyTower) {
			// saveOrUpdateHolographyTower((HolographyTower) be);
		} 
		else if(be instanceof AOSTiming) {
			// saveOrUpdateAOSTiming((AOSTiming) be);
		}
		else if(be instanceof CentralLO) {
			// saveOrUpdateCentralLO((CentralLO) be);
		}
	}
	
	public void saveOrUpdateHolographyTower(HolographyTower ht) throws Exception 
	{
		Method methodToInvoke = BaseElementConversationUtils.class.getMethod("privateSaveOrUpdateHolographyTower", HolographyTower.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[1];
		args[0] = ht;
		conversationInterceptor.invoke(methodToInvoke, this, args);
	}

	public ConversationTokenProvider privateSaveOrUpdateHolographyTower(HolographyTower ht) 
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		HolographyTowerService service = TmcdbContextFactory.INSTANCE.getHolographyTowerService();
		if(ht.getId() != null)
		{
			service.update(ht);
		}
		else {
			service.create(ht);
		}
		return retVal;
	}
	
	public void saveOrUpdatePad(Pad pad) throws Exception {
		saveOrUpdatePad(pad, ConversationToken.CONVERSATION_COMPLETED);
	}
	
	public void saveOrUpdatePad(Pad pad, ConversationToken conversationToken) throws Exception 
	{
		Method methodToInvoke = BaseElementConversationUtils.class.getMethod("privateSaveOrUpdatePad", Pad.class, ConversationToken.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[2];
		args[0] = pad;
		args[1] = conversationToken;
		conversationInterceptor.invoke(methodToInvoke, this, args);
	}
	
	public ConversationTokenProvider privateSaveOrUpdatePad(Pad pad, ConversationToken conversationToken) 
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(conversationToken);
		PadService padService = TmcdbContextFactory.INSTANCE.getPadService();
		padService.update(pad);
		return retVal;
	}
	
	public void saveOrUpdateWeatherStation(WeatherStationController weatherstation)
	throws Exception 
	{
		Method methodToInvoke = BaseElementConversationUtils.class.getMethod("privateSaveOrUpdateWeatherStation", WeatherStationController.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[1];
		args[0] = weatherstation;
		conversationInterceptor.invoke(methodToInvoke, this, args);
	}

	public ConversationTokenProvider privateSaveOrUpdateWeatherStation(WeatherStationController weatherStation) 
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		WeatherStationControllerService wsService = TmcdbContextFactory.INSTANCE.getWeatherStationControllerService();
		wsService.update(weatherStation);
		return retVal;
	}
	
	public void saveOrUpdateFrontEnd(FrontEnd frontEnd)
	throws Exception {
		Method methodToInvoke = BaseElementConversationUtils.class.getMethod("privateSaveOrUpdateFrontEnd", FrontEnd.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[1];
		args[0] = frontEnd;
		conversationInterceptor.invoke(methodToInvoke, this, args);
	}

	public ConversationTokenProvider privateSaveOrUpdateFrontEnd(FrontEnd frontEnd) {
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		FrontEndService frontEndService = TmcdbContextFactory.INSTANCE.getFrontEndService();
		frontEndService.update(frontEnd);
		return retVal;
	}

	public ConversationTokenProvider privateDeleteFrontend(FrontEnd fe, ConversationToken token) {
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(token);
		FrontEndService feService = TmcdbContextFactory.INSTANCE.getFrontEndService();
		feService.delete(fe);
		return retVal;
	}
	
	public ConversationTokenProvider privateDeleteWeatherStation(WeatherStationController ws, ConversationToken token) {
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(token);
		WeatherStationControllerService service = TmcdbContextFactory.INSTANCE.getWeatherStationControllerService();
		service.delete(ws);
		return retVal;
	}
	
	public HolographyTowerToPad[] findHolographyTowerToPadAssignmentsForPad(Pad pad) throws Exception
	{
		Method methodToInvoke = BaseElementConversationUtils.class.getMethod("privateFindHolographyTowerToPadForPad", Pad.class, HolographyTowerToPadsHolder.class);
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[2];
		args[0] = pad;
		HolographyTowerToPadsHolder result = new HolographyTowerToPadsHolder();
		args[1] = result;
		conversationInterceptor.invoke(methodToInvoke, this, args);
		HolographyTowerToPad[]  retVal = result.getHolographyTowerToPads();
		return retVal;
	}
	
	public ConversationTokenProvider privateFindHolographyTowerToPadForPad(Pad pad, HolographyTowerToPadsHolder resultHolder) 
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		HolographyTowerToPadService service =  TmcdbContextFactory.INSTANCE.getHolographyTowerToPadService();
		List<HolographyTowerToPad> a2ps = service.findCurrentHolographyTowerToPadAssignmentForPad(pad);
		resultHolder.setHolographyTowerToPads(a2ps.toArray(new HolographyTowerToPad[0]));
		
		return retVal;
	}
	
	private class HolographyTowerToPadsHolder
	{
		private HolographyTowerToPad[] holographyTowerToPads;
		
		public HolographyTowerToPad[] getHolographyTowerToPads() {
			return holographyTowerToPads;
		}
		
		public void setHolographyTowerToPads(HolographyTowerToPad[] holographyTowerToPads) {
			this.holographyTowerToPads = holographyTowerToPads;
		}
	}
	
	public HolographyTowerToPad[] findHolographyTowerToPadAssignmentsForHolographyTower(HolographyTower holoTower) throws Exception 
	{
		Method methodToInvoke = BaseElementConversationUtils.class.getMethod("privateFindHolographyTowerToPadForHolographyTower", HolographyTower.class, HolographyTowerToPadsHolder.class);
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[2];
		args[0] = holoTower;
		HolographyTowerToPadsHolder result = new HolographyTowerToPadsHolder();
		args[1] = result;
		conversationInterceptor.invoke(methodToInvoke, this, args);
		HolographyTowerToPad[]  retVal = result.getHolographyTowerToPads();
		return retVal;
	}
	
	public ConversationTokenProvider privateFindHolographyTowerToPadForHolographyTower(HolographyTower holoTower, HolographyTowerToPadsHolder resultHolder) 
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);

		HolographyTowerToPadService service =  TmcdbContextFactory.INSTANCE.getHolographyTowerToPadService();
		List<HolographyTowerToPad> h2ps = service.findCurrentHolographyTowerToPadAssignmentForHolographyTower(holoTower);
		resultHolder.setHolographyTowerToPads(h2ps.toArray(new HolographyTowerToPad[0]));
		
		return retVal;
	}
	
	public void hydratePad(Pad pad) throws Exception
	{
		Method methodToInvoke = BaseElementConversationUtils.class.getMethod("privateHydratePad", Pad.class);
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[1];
		args[0] = pad;
		conversationInterceptor.invoke(methodToInvoke, this, args);
	}
	
	public ConversationTokenProvider privateHydratePad(Pad pad)
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		PadService padService =  TmcdbContextFactory.INSTANCE.getPadService();	   
		padService.hydrate(pad);
		return retVal;
	}
	
	public boolean preparePadSave(Pad pad, String userId, String description) throws Exception
	{
		Method methodToInvoke = BaseElementConversationUtils.class.getMethod("privatePreparePadSave", 
				Pad.class, String.class, String.class, BooleanHolder.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		BooleanHolder resultholder = new BooleanHolder();
		Object[] args = new Object[4];
		args[0] = pad;
		args[1] = userId;
		args[2] = description;
		args[3] = resultholder;
		conversationInterceptor.invoke(methodToInvoke, this, args);
		boolean retVal = resultholder.getBooleanValue();
		return retVal;
	}
	
	public ConversationTokenProvider privatePreparePadSave(Pad pad, String who, String description, BooleanHolder resultHolder) 
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		PadService service = TmcdbContextFactory.INSTANCE.getPadService();
		boolean successful = service.prepareSave(pad, who, description);
		resultHolder.setBooleanValue(successful);
		return retVal;
	}
	
	public void endPadSave(Pad pad) throws Exception 
	{
		Method methodToInvoke = BaseElementConversationUtils.class.getMethod("privateEndPadSave", 
				Pad.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[1];
		args[0] = pad;
		conversationInterceptor.invoke(methodToInvoke, this, args);
	}
	
	public ConversationTokenProvider privateEndPadSave(Pad pad) 
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		PadService service = TmcdbContextFactory.INSTANCE.getPadService();
		service.endSave(pad);
		return retVal;
	}
	
	/**
	 * @param antenna the antenna which we wish to save
	 * @param userId the userid of the person making the change
	 * @param description a description of the change
	 * @return boolean indicating if the save can be performed or not
	 * @throws exception if there's a problem
	 */
	public boolean prepareAntennaSave(Antenna antenna, String userId,
			String description) throws Exception
	{
		Method methodToInvoke = BaseElementConversationUtils.class.getMethod("privatePrepareAntennaSave", 
				Antenna.class, String.class, String.class, BooleanHolder.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		BooleanHolder resultholder = new BooleanHolder();
		Object[] args = new Object[4];
		args[0] = antenna;
		args[1] = userId;
		args[2] = description;
		args[3] = resultholder;
		conversationInterceptor.invoke(methodToInvoke, this, args);
		boolean retVal = resultholder.getBooleanValue();
		return retVal;
	}
	
	public ConversationTokenProvider privatePrepareAntennaSave(Antenna antenna, String who, String description, BooleanHolder resultHolder) 
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		AntennaService service = TmcdbContextFactory.INSTANCE.getAntennaService();
		boolean successful = service.prepareAntennaSave(antenna, who, description);
		resultHolder.setBooleanValue(successful);
		return retVal;
	}
	
	public void endAntennaSave(Antenna antenna) throws Exception 
	{
		Method methodToInvoke = BaseElementConversationUtils.class.getMethod("privateEndAntennaSave", 
				Antenna.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[1];
		args[0] = antenna;
		conversationInterceptor.invoke(methodToInvoke, this, args);
	}
	
	public ConversationTokenProvider privateEndAntennaSave(Antenna antenna) 
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		AntennaService service = TmcdbContextFactory.INSTANCE.getAntennaService();
		service.endAntennaSave(antenna);
		return retVal;
	}

	public void hydrateHolographyTower(HolographyTower tower) throws Exception
	{
		Method methodToInvoke = BaseElementConversationUtils.class.getMethod("privateHydrateHolographyTower", HolographyTower.class);
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[1];
		args[0] = tower;
		conversationInterceptor.invoke(methodToInvoke, this, args);
	}
	
	public ConversationTokenProvider privateHydrateHolographyTower(HolographyTower tower)
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		HolographyTowerService towerService =  TmcdbContextFactory.INSTANCE.getHolographyTowerService();	   
		towerService.hydrate(tower);
		return retVal;
	}
}
