/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdbgui.utils.conversation;

import java.lang.reflect.Method;

import alma.acs.tmcdb.AcsService;
import alma.acs.tmcdb.AlarmCategory;
import alma.acs.tmcdb.AlarmDefinition;
import alma.acs.tmcdb.BACIProperty;
import alma.acs.tmcdb.ChannelMapping;
import alma.acs.tmcdb.Component;
import alma.acs.tmcdb.Computer;
import alma.acs.tmcdb.Configuration;
import alma.acs.tmcdb.Container;
import alma.acs.tmcdb.ContainerStartupOption;
import alma.acs.tmcdb.DefaultMember;
import alma.acs.tmcdb.DomainsMapping;
import alma.acs.tmcdb.FaultCode;
import alma.acs.tmcdb.FaultFamily;
import alma.acs.tmcdb.FaultMember;
import alma.acs.tmcdb.ReductionLink;
import alma.acs.tmcdb.ReductionThreshold;
import alma.acs.tmcdb.Schemas;
import alma.obops.dam.config.TmcdbContextFactory;
import alma.obops.dam.tmcdb.service.ConfigurationService;
import alma.obops.dam.utils.ConversationInterceptor;
import alma.obops.dam.utils.ConversationTokenProvider;
import alma.obops.dam.utils.ConversationTokenProviderAdapter;
import alma.obops.dam.utils.ConversationTokenProvider.ConversationToken;
import alma.tmcdb.domain.Antenna;
import alma.tmcdb.domain.AntennaToPad;
import alma.tmcdb.domain.AssemblyStartup;
import alma.tmcdb.domain.BaseElementStartup;
import alma.tmcdb.domain.FocusModelCoeff;
import alma.tmcdb.domain.FrontEnd;
import alma.tmcdb.domain.HolographyTowerToPad;
import alma.tmcdb.domain.HwConfiguration;
import alma.tmcdb.domain.PointingModelCoeff;
import alma.tmcdb.domain.StartupScenario;
import alma.tmcdb.domain.WeatherStationController;

public class BackendConversationUtils 
{
	private static BackendConversationUtils singletonInstance;
	
	private BackendConversationUtils() 
	{
	}
	
	public static synchronized BackendConversationUtils getInstance()
	{
		if(null == singletonInstance)
		{
			singletonInstance = new BackendConversationUtils();
		}
		
		return singletonInstance;
	}

	/******************************** GENERIC METHODS ************************************/
	public void delete(Object o, ConversationToken token) throws Exception {
		delete(o, token, false);
	}

	/**
	 * Deletes an object.
	 * @param o the object to delete
	 * @param token the conversationtoken which designates how the conversation will be handled
	 * @param deferredConstraints boolean indicating if db constraints should be deferred (true) or not (false)
	 * @throws Exception if a problem occurs
	 */
	public void delete(Object o, ConversationToken token, boolean deferredConstraints) throws Exception 
	{
		Method methodToInvoke = null;
		Object[] args = null;
		Object objForInvocation = null;

		if( o instanceof AcsService ) {
			methodToInvoke = AcsServiceConversationUtils.class.getMethod("privateDeleteAcsService", AcsService.class, ConversationToken.class);	
			objForInvocation = AcsServiceConversationUtils.getInstance();
			args = new Object[2];
			args[0] = o;
			args[1] = token;
		}
		else if( o instanceof ChannelMapping ) {
			methodToInvoke = ChannelMappingConversationUtils.class.getMethod("privateDeleteChannelMapping", ChannelMapping.class, ConversationToken.class);
			objForInvocation = ChannelMappingConversationUtils.getInstance();
			args = new Object[2];
			args[0] = o;
			args[1] = token;
		}
		else if( o instanceof DomainsMapping ) {
			methodToInvoke = DomainsMappingConversationUtils.class.getMethod("privateDeleteDomainsMapping", DomainsMapping.class, ConversationToken.class);
			objForInvocation = DomainsMappingConversationUtils.getInstance();
			args = new Object[2];
			args[0] = o;
			args[1] = token;
		}
		else if( o instanceof Antenna ) {
			methodToInvoke = BaseElementConversationUtils.class.getMethod("privateDeleteAntenna", Antenna.class, ConversationToken.class);
			objForInvocation = BaseElementConversationUtils.getInstance();
			args = new Object[2];
			args[0] = o;
			args[1] = token;
		}
		else if( o instanceof FrontEnd ) {
			methodToInvoke = BaseElementConversationUtils.class.getMethod("privateDeleteFrontend", FrontEnd.class, ConversationToken.class);
			objForInvocation = BaseElementConversationUtils.getInstance();
			args = new Object[2];
			args[0] = o;
			args[1] = token;
		}
		else if( o instanceof WeatherStationController ) {
			methodToInvoke = BaseElementConversationUtils.class.getMethod("privateDeleteWeatherStation", WeatherStationController.class, ConversationToken.class);
			objForInvocation = BaseElementConversationUtils.getInstance();
			args = new Object[2];
			args[0] = o;
			args[1] = token;
		}
		else if( o instanceof Component ) {
			methodToInvoke = ComponentConversationUtils.class.getMethod("privateDeleteComponent", Component.class, ConversationToken.class);	
			objForInvocation = ComponentConversationUtils.getInstance();
			args = new Object[2];
			args[0] = o;
			args[1] = token;
		}
		else if( o instanceof Container ) 
		{
			// delete all the container's startup options before deleting the container
			Container cont = (Container) o;
			ContainerConversationUtils.getInstance().hydrateContainerStartupOptions(cont);
			for(ContainerStartupOption opt : cont.getContainerStartupOptions() ) {
				BackendConversationUtils.getInstance().delete(opt, ConversationToken.CONVERSATION_PENDING, false);
			}
			if(cont.getContainerStartupOptions().size() > 0) {
				cont.getContainerStartupOptions().clear();
				ContainerConversationUtils.getInstance().saveOrUpdateContainer(cont, ConversationToken.CONVERSATION_PENDING);
			}
			
			// now delete the container
			methodToInvoke = ContainerConversationUtils.class.getMethod("privateDeleteContainer", Container.class, ConversationToken.class);
			objForInvocation = ContainerConversationUtils.getInstance();
			args = new Object[2];
			args[0] = cont;
			args[1] = token;
		}
		else if( o instanceof ContainerStartupOption ) {
			methodToInvoke = ContainerConversationUtils.class.getMethod("privateDeleteContainerStartupOption", ContainerStartupOption.class, ConversationToken.class);	
			objForInvocation = ContainerConversationUtils.getInstance();
			args = new Object[2];
			args[0] = o;
			args[1] = token;
		}
		else if( o instanceof AntennaToPad ) {
			methodToInvoke = AntennaToPadConversationUtils.class.getMethod("privateDeleteAntennaToPad", AntennaToPad.class, ConversationToken.class);	
			objForInvocation = AntennaToPadConversationUtils.getInstance();
			args = new Object[2];
			args[0] = o;
			args[1] = token;
		}
		else if( o instanceof HolographyTowerToPad ) {
			methodToInvoke = HolographyTowerToPadConversationUtils.class.getMethod("privateDeleteHolographyTowerToPad", HolographyTowerToPad.class, ConversationToken.class);	
			objForInvocation = HolographyTowerToPadConversationUtils.getInstance();
			args = new Object[2];
			args[0] = o;
			args[1] = token;
		}
		else if( o instanceof Computer ) {
			methodToInvoke = ComputerConversationUtils.class.getMethod("privateDeleteComputer", Computer.class, ConversationToken.class);
			objForInvocation = ComputerConversationUtils.getInstance();
			args = new Object[2];
			args[0] = o;
			args[1] = token;
		}
		else if( o instanceof BACIProperty ) {
			methodToInvoke = BaciConversationUtils.class.getMethod("privateDeleteBACIProperty", BACIProperty.class, ConversationToken.class);
			objForInvocation = BaciConversationUtils.getInstance();
			args = new Object[2];
			args[0] = o;
			args[1] = token;
		}
		else if( o instanceof HwConfiguration ) {
			methodToInvoke = HwConfigurationConversationUtils.class.getMethod("privateDeleteHwConfiguration", HwConfiguration.class, ConversationToken.class);
			objForInvocation = HwConfigurationConversationUtils.getInstance();
			args = new Object[2];
			args[0] = o;
			args[1] = token;
		}
		else if( o instanceof BaseElementStartup && token.equals(ConversationToken.CONVERSATION_COMPLETED)) {
			methodToInvoke = StartupScenarioConversationUtils.class.getMethod("privateDeleteBaseElementStartup", BaseElementStartup.class);
			objForInvocation = StartupScenarioConversationUtils.getInstance();
			args = new Object[1];
			args[0] = o;
		}
		else if( o instanceof BaseElementStartup && !token.equals(ConversationToken.CONVERSATION_COMPLETED)) {
			methodToInvoke = StartupScenarioConversationUtils.class.getMethod("privateDeleteBaseElementStartup", BaseElementStartup.class, ConversationToken.class);
			objForInvocation = StartupScenarioConversationUtils.getInstance();
			args = new Object[2];
			args[0] = o;
			args[1] = token;
		}
		else if( o instanceof AssemblyStartup ) {
			methodToInvoke = StartupScenarioConversationUtils.class.getMethod("privateDeleteAssemblyStartup", AssemblyStartup.class, ConversationToken.class);
			objForInvocation = StartupScenarioConversationUtils.getInstance();
			args = new Object[2];
			args[0] = o;
			args[1] = token;
		}
		else if( o instanceof FocusModelCoeff ) {
			methodToInvoke = FocusModelConversationUtils.class.getMethod("privateDeleteFocusModelCoeff", FocusModelCoeff.class, ConversationToken.class);
			objForInvocation = FocusModelConversationUtils.getInstance();
			args = new Object[2];
			args[0] = o;
			args[1] = token;
		}
		else if( o instanceof PointingModelCoeff ) {
			methodToInvoke = PointingModelConversationUtils.class.getMethod("privateDeletePointingModelCoeff", PointingModelCoeff.class, ConversationToken.class);
			objForInvocation = PointingModelConversationUtils.getInstance();
			args = new Object[2];
			args[0] = o;
			args[1] = token;
		}
		else if( o instanceof StartupScenario ) {
			methodToInvoke = StartupScenarioConversationUtils.class.getMethod("privateDeleteStartupScenario", StartupScenario.class, ConversationToken.class);
			objForInvocation = StartupScenarioConversationUtils.getInstance();
			args = new Object[2];
			args[0] = o;
			args[1] = token;
		} else if( o instanceof Configuration ) {
			methodToInvoke = SwConfigurationConversationUtils.class.getMethod("privateDeleteConfiguration", Configuration.class, ConversationToken.class);
			objForInvocation = SwConfigurationConversationUtils.getInstance();
			args = new Object[2];
			args[0] = o;
			args[1] = token;
		}
		else if( o instanceof Schemas ) {
			methodToInvoke = HwConfigurationConversationUtils.class.getMethod("privateDeleteSchemas", Schemas.class, ConversationToken.class);
			objForInvocation = HwConfigurationConversationUtils.getInstance();
			args = new Object[2];
			args[0] = o;
			args[1] = token;
		}
		else if( o instanceof AlarmCategory ) {
			methodToInvoke = AlarmConversationUtils.class.getMethod("privateDeleteAlarmCategory", AlarmCategory.class, ConversationToken.class);
			objForInvocation = AlarmConversationUtils.getInstance();
			args = new Object[2];
			args[0] = o;
			args[1] = token;
		}
		else if( o instanceof DefaultMember ) {
			methodToInvoke = AlarmConversationUtils.class.getMethod("privateDeleteDefaultMember", DefaultMember.class, ConversationToken.class);
			objForInvocation = AlarmConversationUtils.getInstance();
			args = new Object[2];
			args[0] = o;
			args[1] = token;
		}
		else if( o instanceof FaultCode ) {
			methodToInvoke = AlarmConversationUtils.class.getMethod("privateDeleteFaultCode", FaultCode.class, ConversationToken.class);
			objForInvocation = AlarmConversationUtils.getInstance();
			args = new Object[2];
			args[0] = o;
			args[1] = token;
		}
		else if( o instanceof FaultFamily ) {
			methodToInvoke = AlarmConversationUtils.class.getMethod("privateDeleteFaultFamily", FaultFamily.class, ConversationToken.class);
			objForInvocation = AlarmConversationUtils.getInstance();
			args = new Object[2];
			args[0] = o;
			args[1] = token;
		}
		else if( o instanceof FaultMember ) {
			methodToInvoke = AlarmConversationUtils.class.getMethod("privateDeleteFaultMember", FaultMember.class, ConversationToken.class);
			objForInvocation = AlarmConversationUtils.getInstance();
			args = new Object[2];
			args[0] = o;
			args[1] = token;
		}
		else if( o instanceof ReductionLink ) {
			methodToInvoke = AlarmConversationUtils.class.getMethod("privateDeleteReductionLink", ReductionLink.class, ConversationToken.class);
			objForInvocation = AlarmConversationUtils.getInstance();
			args = new Object[2];
			args[0] = o;
			args[1] = token;
		}
		else if( o instanceof ReductionThreshold ) {
			methodToInvoke = AlarmConversationUtils.class.getMethod("privateDeleteReductionThreshold", ReductionThreshold.class, ConversationToken.class);
			objForInvocation = AlarmConversationUtils.getInstance();
			args = new Object[2];
			args[0] = o;
			args[1] = token;
		}
		else if( o instanceof AlarmDefinition ) {
			methodToInvoke = AlarmConversationUtils.class.getMethod("privateDeleteAlarmDefinition", AlarmDefinition.class, ConversationToken.class);
			objForInvocation = AlarmConversationUtils.getInstance();
			args = new Object[2];
			args[0] = o;
			args[1] = token;
		}
		else {
			throw new UnsupportedOperationException("Cannot delete this type of entity");
		}
		// else if... Fill with the rest -- Configuration, HwConfiguration, BaseElement, etc...

		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		
		if(!deferredConstraints || !TmcdbContextFactory.INSTANCE.connectionIsOracle()) {
			// either: a) we don't want deferred constraints OR b) we're using hsqldb (which does not support them);
			// in either case, we will invoke the method using DB constraints in their "normal" (i.e. immediate) configuration
			conversationInterceptor.invoke(methodToInvoke, objForInvocation, args);
		} else {
			// we both: a) want deferred constraints AND b) we're using oracle (which supports them);
			// so, we invoke our method using deferred DB constraints (which means the constraints
			// are checked when the entire transaction is flushed to the DB but not at each intermediate 
			// step along the way).
			conversationInterceptor.invokeWithDeferredConstraints(methodToInvoke, objForInvocation, args);
		}
	}

	public boolean exists(Object o) throws Exception 
	{
		Method methodToInvoke = null;
		Object[] args = null;
		Object objForInvocation = null;
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();

		if( o instanceof Computer ) {
			methodToInvoke = ComputerConversationUtils.class.getMethod("privateFindComputer", Computer.class, ComputerHolder.class);
			objForInvocation = ComputerConversationUtils.getInstance();
			ComputerHolder holder = new ComputerHolder();
			args = new Object[2];
			args[0] = o;
			args[1] = holder;
			conversationInterceptor.invoke(methodToInvoke, objForInvocation, args);

			// If the given computer is transient and the returned computer exists, then the object already exists
			// If the given computer is persisted, and the returned computer exists, then check the IDs: if they
			// differ, then the object already exists; if not, then we obtained the same object
			if( holder.getComputer() != null ) {
				Computer c = (Computer)o;
				if( c.getNetworkDeviceId() == null || c.getNetworkDeviceId() < 0 )
					return true;
				if( c.getNetworkDeviceId().intValue() == holder.getComputer().getNetworkDeviceId().intValue() )
					return false;
				return true;
			}
		}
		else if( o instanceof Component ) {
			methodToInvoke = ComponentConversationUtils.class.getMethod("privateFindComponent", Component.class, ComponentHolder.class);
			objForInvocation = ComponentConversationUtils.getInstance();
			ComponentHolder holder = new ComponentHolder();
			args = new Object[2];
			args[0] = o;
			args[1] = holder;
			conversationInterceptor.invoke(methodToInvoke, objForInvocation, args);

			// If the given component is transient and the returned component exists, then the object already exists
			// If the given component is persisted, and the returned component exists, then check the IDs: if they
			// differ, then the object already exists; if not, then we obtained the same object
			if( holder.getComponent() != null ) {
				Component c = (Component)o;
				if( c.getComponentId() == null || c.getComponentId() < 0 )
					return true;
				if( c.getComponentId().intValue() == holder.getComponent().getComponentId().intValue() )
					return false;
				return true;
			}
		}
		else if( o instanceof Container ) {
			methodToInvoke = ContainerConversationUtils.class.getMethod("privateFindContainer", Container.class, ContainerHolder.class);
			objForInvocation = ContainerConversationUtils.getInstance();
			ContainerHolder holder = new ContainerHolder();
			args = new Object[2];
			args[0] = o;
			args[1] = holder;
			conversationInterceptor.invoke(methodToInvoke, objForInvocation, args);

			// If the given container is transient and the returned container exists, then the object already exists
			// If the given container is persisted, and the returned container exists, then check the IDs: if they
			// differ, then the object already exists; if not, then we obtained the same object
			if( holder.getContainer() != null ) {
				Container c = (Container)o;
				if( c.getContainerId() == null || c.getContainerId() < 0 )
					return true;
				if( c.getContainerId().intValue() == holder.getContainer().getContainerId().intValue() )
					return false;
				return true;
			}
		}
		// else if... Fill with the rest -- Configuration, HwConfiguration, BaseElement, etc...		

		return false;
	}
	
	public void reAttach(Object obj) throws Exception
	{
		Method methodToInvoke = BackendConversationUtils.class.getMethod("privateReAttach", Object.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[1];
		args[0] = obj;
		conversationInterceptor.invoke(methodToInvoke, this, args);
	}
	
	public ConversationTokenProvider privateReAttach(Object obj) 
	{
		// TODO: try to find a way to use CONVERSATION_PENDING, otherwise this reattachment
		// doesn't really do much of anything (since the object is reattached and then immediately
		// detached again when the conversation/session is completed). CONVERSATION_PENDING seems to
		// mostly work, but for a few places in the code where we get a hibernate exception regarding trying
		// to associate an object w/ 2 open sessions; need to understand where the additional session is
		// getting opened. For now, this reattachment doesn't accomplish much, but it's here for future use.
		// The idea was that editors would lock (reattach) the item they're editing, then proceed to edit, 
		// then we wouldn't have to worry about hydration, etc. during the editing process because we'd be
		// dealing with an attached object rather than a detached object. good goal in the long run, i think, 
		// but for now we're not yet there.
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		ConfigurationService service = TmcdbContextFactory.INSTANCE.getConfigurationService();
		try {
			service.reAttach(obj);			
		}
		catch(org.springframework.dao.InvalidDataAccessApiUsageException ex) {
			if(ex.getCause() instanceof org.hibernate.TransientObjectException) {
				// noop, we're dealing with a transient instance and it's safe to ignore this...
			} else {
				throw ex;
			}
		}
		return retVal;
	}
}
