/*******************************************************************************
  * ALMA - Atacama Large Millimeter Array
  * Copyright (c) NRAO - National Radio Astronomy Observatory, 2012
  * (in the framework of the ALMA collaboration).
  * All rights reserved.
  *
  * This library is free software; you can redistribute it and/or
  * modify it under the terms of the GNU Lesser General Public
  * License as published by the Free Software Foundation; either
  * version 2.1 of the License, or (at your option) any later version.
  *
  * This library is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  * Lesser General Public License for more details.
  *
  * You should have received a copy of the GNU Lesser General Public
  * License along with this library; if not, write to the Free Software
  * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 
*******************************************************************************/
package alma.obops.tmcdbgui.utils.conversation;

import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import alma.acs.tmcdb.BACIProperty;
import alma.acs.tmcdb.Component;
import alma.acs.tmcdb.Configuration;
import alma.obops.dam.config.TmcdbContextFactory;
import alma.obops.dam.tmcdb.service.BACIPropertyService;
import alma.obops.dam.tmcdb.service.ComponentService;
import alma.obops.dam.tmcdb.service.SwConfigurationService;
import alma.obops.dam.utils.ConversationInterceptor;
import alma.obops.dam.utils.ConversationTokenProvider;
import alma.obops.dam.utils.ConversationTokenProviderAdapter;
import alma.obops.dam.utils.ProgressMonitor;
import alma.obops.dam.utils.ConversationTokenProvider.ConversationToken;

/**
 * @author sharring
 *
 */
public class BaciConversationUtils 
{
	private static BaciConversationUtils singletonInstance;
	
	private BaciConversationUtils() 
	{
	}
	
	public static synchronized BaciConversationUtils getInstance()
	{
		if(null == singletonInstance)
		{
			singletonInstance = new BaciConversationUtils();
		}
		
		return singletonInstance;
	}
	
	public ConversationTokenProvider privateDeleteBACIProperty(BACIProperty bp, ConversationToken token)
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(token);
		BACIPropertyService bpService = TmcdbContextFactory.INSTANCE.getBACIPropertyService();
		bpService.delete(bp);
		return retVal;
	}

	public List<String> findBACIPropertyNamesForComponents(Component[] components, ProgressMonitor monitor) throws Exception {
		Method methodToInvoke = BaciConversationUtils.class.getMethod("privateFindBACIPropertyNamesForComponents", Component[].class, StringListHolder.class, ProgressMonitor.class);
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		StringListHolder holder = new StringListHolder();
		Object[] args = new Object[3];
		args[0] = components;
		args[1] = holder;
		args[2] = monitor;
		conversationInterceptor.invoke(methodToInvoke, this, args);
		return holder.getStringList();
	}

	public ConversationTokenProvider privateFindBACIPropertyNamesForComponents(Component[] components, StringListHolder holder, ProgressMonitor monitor) {

		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		ComponentService compService = TmcdbContextFactory.INSTANCE.getComponentService();

		// We get all the properties for all the components
		Set<BACIProperty> baciProps;
		Set<String> tmp;
		List<Set<String>> baciPropsList = new ArrayList<Set<String>>();
		monitor.beginTask("Getting common BACI properties", components.length);
		for (Component component : components) {
			compService.hydrateBACIProperties(component);
			baciProps = component.getBACIProperties();
			tmp = new HashSet<String>();
			for (BACIProperty prop: baciProps) {
				tmp.add(prop.getPropertyName());
			}
			baciPropsList.add(tmp);
			monitor.worked(1);
		}

		// Then we intersect them
		// If none, return empty list
		if( baciPropsList.isEmpty() )
			holder.setStringList( new ArrayList<String>() );
		else {
			Iterator<Set<String>> iterator = baciPropsList.iterator();
			tmp = baciPropsList.get(0);
			iterator.next();
			while( iterator.hasNext() ) {
				tmp.retainAll( iterator.next() );
			}
			holder.setStringList( new ArrayList<String>(tmp) );
		}

		return retVal;
	}

	

	public void bulkUpdateBACIProperties(BACIProperty[] propsToUpdate, String[] objectProperties, Object[] values, ProgressMonitor monitor) throws Exception {
		Method methodToInvoke = BaciConversationUtils.class.getMethod("privateBulkUpdateBACIProperties", BACIProperty[].class, String[].class, Object[].class, ProgressMonitor.class);
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[4];
		args[0] = propsToUpdate;
		args[1] = objectProperties;
		args[2] = values;
		args[3] = monitor;
		conversationInterceptor.invoke(methodToInvoke, this, args);
	}

	public ConversationTokenProvider privateBulkUpdateBACIProperties(BACIProperty[] propsToUpdate, String[] objectProperties, Object[] values, ProgressMonitor eclipseMonitor) {
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		BACIPropertyService baciService = TmcdbContextFactory.INSTANCE.getBACIPropertyService();
		baciService.bulkUpdateBACIProperties(propsToUpdate, objectProperties, values, eclipseMonitor);
		return retVal;
	}

	
    private class StringListHolder
	{
		private List<String> str;

		public List<String> getStringList() {
			return str;
		}

		public void setStringList(List<String> str) {
			this.str = str;
		}
	}
    
    public void hydrateBACIProperties(Component component) throws Exception 
	{		
		Method methodToInvoke = BaciConversationUtils.class.getMethod("privateHydrateBACIProperties", Component.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[1];
		args[0] = component;
		conversationInterceptor.invoke(methodToInvoke, this, args);
	}
	
	public ConversationTokenProvider privateHydrateBACIProperties(Component comp)
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		ComponentService compService = TmcdbContextFactory.INSTANCE.getComponentService();
		compService.hydrateBACIProperties(comp);
		return retVal;
	}

	public void hydrateBACIProperties(Configuration configuration) 
	   throws Exception 
	{
		Method methodToInvoke = BaciConversationUtils.class.getMethod("privateHydrateBACIProperties", Configuration.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[1];
		args[0] = configuration;
		conversationInterceptor.invoke(methodToInvoke, this, args);
	}
	
	public ConversationTokenProvider privateHydrateBACIProperties(Configuration config)
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		SwConfigurationService configService = TmcdbContextFactory.INSTANCE.getSwConfigurationService();
		configService.hydrateBACIProperties(config);
		return retVal;
	}
}
