/*******************************************************************************
  * ALMA - Atacama Large Millimeter Array
  * Copyright (c) NRAO - National Radio Astronomy Observatory, 2012
  * (in the framework of the ALMA collaboration).
  * All rights reserved.
  *
  * This library is free software; you can redistribute it and/or
  * modify it under the terms of the GNU Lesser General Public
  * License as published by the Free Software Foundation; either
  * version 2.1 of the License, or (at your option) any later version.
  *
  * This library is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  * Lesser General Public License for more details.
  *
  * You should have received a copy of the GNU Lesser General Public
  * License along with this library; if not, write to the Free Software
  * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 
*******************************************************************************/
package alma.obops.tmcdbgui.utils.conversation;

import java.lang.reflect.Method;
import java.util.List;

import alma.acs.tmcdb.BACIProperty;
import alma.acs.tmcdb.Component;
import alma.obops.dam.config.TmcdbContextFactory;
import alma.obops.dam.tmcdb.service.BACIPropertyService;
import alma.obops.dam.utils.ConversationInterceptor;
import alma.obops.dam.utils.ConversationTokenProvider;
import alma.obops.dam.utils.ConversationTokenProviderAdapter;
import alma.obops.dam.utils.ProgressMonitor;
import alma.obops.dam.utils.ConversationTokenProvider.ConversationToken;

/**
 * @author sharring
 *
 */
public class BACIPropertyConversationUtils 
{
	private static BACIPropertyConversationUtils singletonInstance;

	private BACIPropertyConversationUtils() 
	{
	}

	public static synchronized BACIPropertyConversationUtils getInstance()
	{
		if(null == singletonInstance)
		{
			singletonInstance = new BACIPropertyConversationUtils();
		}

		return singletonInstance;
	}
	
	public void saveOrUpdate(BACIProperty bp) throws Exception 
	{
		Method methodToInvoke = BACIPropertyConversationUtils.class.getMethod("privateSaveOrUpdate", BACIProperty.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[1];
		args[0] = bp;
		conversationInterceptor.invoke(methodToInvoke, this, args);
	}

	public ConversationTokenProvider privateSaveOrUpdate(BACIProperty bp) 
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		BACIPropertyService service = TmcdbContextFactory.INSTANCE.getBACIPropertyService();
		service.update(bp);
		return retVal;
	}

	/**
	 * @param element the element for which we want to run in an attached mode
	 * @param runnable the runnable to run
	 */
	public void runWithAttachedObject(Object element, Runnable runnable) throws Exception 
	{
		Method methodToInvoke = BACIPropertyConversationUtils.class.getMethod("privateRunWithAttachedObject", Object.class, Runnable.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[2];
		args[0] = element;
		args[1] = runnable;
		conversationInterceptor.invoke(methodToInvoke, this, args);
	}

	public ConversationTokenProvider privateRunWithAttachedObject(Object obj, Runnable runnable)
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		BACIPropertyService service = TmcdbContextFactory.INSTANCE.getBACIPropertyService();
		service.runWithAttachedObject(obj, runnable);
		return retVal;
	}

	/**
	 * @param criteria
	 * @param object
	 * @return
	 */
	public List<?> find(List<Object> searchCriteria, List<Object> orderCriteria) throws Exception 
	{
		Method methodToInvoke = BACIPropertyConversationUtils.class.getMethod("privateFind", List.class, List.class, BACIPropertyListHolder.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[3];
		args[0] = searchCriteria;
		args[1] = orderCriteria;
		BACIPropertyListHolder resultHolder = new BACIPropertyListHolder();
		args[2] = resultHolder;
		conversationInterceptor.invoke(methodToInvoke, this, args);
		return resultHolder.getBACIProperties();
	}

	@SuppressWarnings("unchecked")
	public ConversationTokenProvider privateFind(List searchCriteria, List orderCriteria, BACIPropertyListHolder resultHolder)
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		BACIPropertyService service = TmcdbContextFactory.INSTANCE.getBACIPropertyService();
		resultHolder.setBACIProperties((List<BACIProperty>) service.find(searchCriteria, orderCriteria));
		return retVal;
	}
	
	private class BACIPropertyListHolder 
	{
		private List<BACIProperty> baciprops;
		public List<BACIProperty> getBACIProperties() { return this.baciprops; }
		public void setBACIProperties(List<BACIProperty> props) { this.baciprops = props; }
	}

	/**
	 * @param component
	 * @param newName
	 * @return
	 */
	public boolean componentHasProperty(Component component, String newName) throws Exception 
	{
		Method methodToInvoke = BACIPropertyConversationUtils.class.getMethod("privateComponentHasProperty", Component.class, String.class, BooleanHolder.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[3];
		args[0] = component;
		args[1] = newName;
		BooleanHolder resultHolder = new BooleanHolder();
		args[2] = resultHolder;
		conversationInterceptor.invoke(methodToInvoke, this, args);
		return resultHolder.getBooleanValue();
	}

	public ConversationTokenProvider privateComponentHasProperty(Component component, String newName, BooleanHolder resultHolder) 
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		BACIPropertyService service = TmcdbContextFactory.INSTANCE.getBACIPropertyService();
		resultHolder.setBooleanValue(service.componentHasProperty(component, newName));
		return retVal;
	}

	/**
	 * @param array
	 * @param objProps
	 * @param objVals
	 * @param eclipseMonitor
	 */
	public void bulkUpdateBACIProperties(BACIProperty[] array,
			String[] objProps, Object[] objVals, ProgressMonitor eclipseMonitor) throws Exception 
	{
		Method methodToInvoke = BACIPropertyConversationUtils.class.getMethod("privateBulkUpdateBACIProperties", 
				BACIProperty[].class, String[].class, Object[].class, ProgressMonitor.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[4];
		args[0] = array;
		args[1] = objProps;
		args[2] = objVals;
		args[3] = eclipseMonitor;
		conversationInterceptor.invoke(methodToInvoke, this, args);
	}

	public ConversationTokenProvider privateBulkUpdateBACIProperties(BACIProperty[] array,
			String[] objProps, Object[] objVals, ProgressMonitor eclipseMonitor) 
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		BACIPropertyService service = TmcdbContextFactory.INSTANCE.getBACIPropertyService();
		service.bulkUpdateBACIProperties(array, objProps, objVals, eclipseMonitor);
		return retVal;
	}

	/**
	 * @param props
	 * @return
	 */
	public boolean baciPropertiesHaveMonitorData(BACIProperty[] props) throws Exception 
	{
		Method methodToInvoke = BACIPropertyConversationUtils.class.getMethod("privateBaciPropertiesHaveMonitorData", BACIProperty[].class, BooleanHolder.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[2];
		args[0] = props;
		BooleanHolder resultHolder = new BooleanHolder();
		args[1] = resultHolder;
		conversationInterceptor.invoke(methodToInvoke, this, args);
		return resultHolder.getBooleanValue();
	}

	public ConversationTokenProvider privateBaciPropertiesHaveMonitorData(BACIProperty[] props, BooleanHolder resultHolder) 
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		BACIPropertyService service = TmcdbContextFactory.INSTANCE.getBACIPropertyService();
		resultHolder.setBooleanValue(service.baciPropertiesHaveMonitorData(props));
		return retVal;
	}

	/**
	 * @param props
	 * @param eclipseMonitor
	 */
	public void bulkDeleteBACIProperties(BACIProperty[] props, ProgressMonitor eclipseMonitor) throws Exception 
	{
		Method methodToInvoke = BACIPropertyConversationUtils.class.getMethod("privateBulkDeleteBACIProperties", 
				BACIProperty[].class, ProgressMonitor.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[2];
		args[0] = props;
		args[1] = eclipseMonitor;
		conversationInterceptor.invoke(methodToInvoke, this, args);
	}

	public ConversationTokenProvider privateBulkDeleteBACIProperties(BACIProperty[] props, ProgressMonitor eclipseMonitor) 
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		BACIPropertyService service = TmcdbContextFactory.INSTANCE.getBACIPropertyService();
		service.bulkDeleteBACIProperties(props, eclipseMonitor);
		return retVal;
	}

	/**
	 * @param components
	 * @param objectProperties
	 * @param values
	 * @param eclipseMonitor
	 */
	public void bulkCreateBACIProperties(Component[] components, String[] objectProperties, Object[] values, ProgressMonitor eclipseMonitor) throws Exception 
	{
		Method methodToInvoke = BACIPropertyConversationUtils.class.getMethod("privateBulkCreateBACIProperties", 
				Component[].class, String[].class, Object[].class, ProgressMonitor.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[4];
		args[0] = components;
		args[1] = objectProperties;
		args[2] = values;
		args[3] = eclipseMonitor;
		conversationInterceptor.invoke(methodToInvoke, this, args);
	}

	public ConversationTokenProvider privateBulkCreateBACIProperties(Component[] array,
			String[] objProps, Object[] objVals, ProgressMonitor eclipseMonitor) 
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		BACIPropertyService service = TmcdbContextFactory.INSTANCE.getBACIPropertyService();
		service.bulkCreateBACIProperties(array, objProps, objVals, eclipseMonitor);
		return retVal;
	}

	/**
	 * @param prop
	 */
	public void hydrateComponent(BACIProperty prop) throws Exception 
	{
		Method methodToInvoke = BACIPropertyConversationUtils.class.getMethod("privateHydrateComponent", BACIProperty.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[1];
		args[0] = prop;
		conversationInterceptor.invoke(methodToInvoke, this, args);
	}

	public ConversationTokenProvider privateHydrateComponent(BACIProperty bp) 
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		BACIPropertyService service = TmcdbContextFactory.INSTANCE.getBACIPropertyService();
		service.hydrateComponent(bp);
		return retVal;
	}

}
