/*******************************************************************************
  * ALMA - Atacama Large Millimeter Array
  * Copyright (c) NRAO - National Radio Astronomy Observatory, 2012
  * (in the framework of the ALMA collaboration).
  * All rights reserved.
  *
  * This library is free software; you can redistribute it and/or
  * modify it under the terms of the GNU Lesser General Public
  * License as published by the Free Software Foundation; either
  * version 2.1 of the License, or (at your option) any later version.
  *
  * This library is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  * Lesser General Public License for more details.
  *
  * You should have received a copy of the GNU Lesser General Public
  * License along with this library; if not, write to the Free Software
  * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 
*******************************************************************************/
package alma.obops.tmcdbgui.utils.conversation;

import java.lang.reflect.Method;
import java.util.List;

import alma.obops.dam.config.TmcdbContextFactory;
import alma.obops.dam.tmcdb.service.AssemblyTypeService;
import alma.obops.dam.tmcdb.service.HwSchemaService;
import alma.obops.dam.utils.ConversationInterceptor;
import alma.obops.dam.utils.ConversationTokenProvider;
import alma.obops.dam.utils.ConversationTokenProviderAdapter;
import alma.obops.dam.utils.ConversationTokenProvider.ConversationToken;
import alma.tmcdb.domain.AntennaType;
import alma.tmcdb.domain.AssemblyType;
import alma.tmcdb.domain.HwSchema;

/**
 * @author sharring
 *
 */
public class AssemblyTypeConversationUtils 
{
	private static AssemblyTypeConversationUtils singletonInstance;

	private AssemblyTypeConversationUtils() 
	{
	}

	public static synchronized AssemblyTypeConversationUtils getInstance()
	{
		if(null == singletonInstance)
		{
			singletonInstance = new AssemblyTypeConversationUtils();
		}

		return singletonInstance;
	}
	
	private class HwSchemasHolder {
		private List<HwSchema> hwSchemas;
		public void setSchemas(List<HwSchema> hwSchemas) { this.hwSchemas = hwSchemas; }
		public List<HwSchema> getSchemas() { return this.hwSchemas; }
	}
	
	public List<HwSchema> findHwSchemasForAssemblyType(AssemblyType assemblyType) throws Exception {
		Method methodToInvoke = AssemblyTypeConversationUtils.class.getMethod("privateFindHwSchemasForAssemblyType", AssemblyType.class, HwSchemasHolder.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		HwSchemasHolder holder = new HwSchemasHolder();
		Object[] args = new Object[2];
		args[0] = assemblyType;
		args[1] = holder;
		conversationInterceptor.invoke(methodToInvoke, this, args);
		return holder.getSchemas();
	}

	public ConversationTokenProvider privateFindHwSchemasForAssemblyType(AssemblyType assemblyType, HwSchemasHolder holder) {
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		HwSchemaService service = TmcdbContextFactory.INSTANCE.getHwSchemaService();
		holder.setSchemas( service.findForAssemblyType(assemblyType) );
		return retVal;
	}
	
	public void hydrateComponentType(AssemblyType assemblyType) throws Exception
	{
		Method methodToInvoke = AssemblyTypeConversationUtils.class.getMethod("privateHydrateComponentType", AssemblyType.class);
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[1];
		args[0] = assemblyType;
		conversationInterceptor.invoke(methodToInvoke, this, args);
	}

	public ConversationTokenProvider privateHydrateComponentType(AssemblyType assemblyType)
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		AssemblyTypeService service = TmcdbContextFactory.INSTANCE.getAssemblyTypeService();
		service.hydrateComponentType(assemblyType);
		return retVal;
	}
	
	private class AssemblyTypeListHolder {
		private List<AssemblyType> assemblyTypes;
		public List<AssemblyType> getAssemblyTypes() {return assemblyTypes;}
		public void setAssemblyTypes(List<AssemblyType> assemblyTypes) {this.assemblyTypes = assemblyTypes;}
	}

	public List<AssemblyType> findAllAssemblyTypes() throws Exception {
		Method methodToInvoke = AssemblyTypeConversationUtils.class.getMethod("privateFindAllAssemblyTypes", AssemblyTypeListHolder.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		AssemblyTypeListHolder holder = new AssemblyTypeListHolder();
		Object[] args = new Object[1];
		args[0] = holder;
		conversationInterceptor.invoke(methodToInvoke, this, args);
		return holder.getAssemblyTypes();
	}

	public ConversationTokenProvider privateFindAllAssemblyTypes(AssemblyTypeListHolder holder) {
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		AssemblyTypeService service = TmcdbContextFactory.INSTANCE.getAssemblyTypeService();
		holder.setAssemblyTypes( service.findAll() );
		return retVal;
	}

	public void updateAssemblyType(AssemblyType at) throws Exception {
		Method methodToInvoke = AssemblyTypeConversationUtils.class.getMethod("privateUpdateAssemblyType", AssemblyType.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[1];
		args[0] = at;
		conversationInterceptor.invoke(methodToInvoke, this, args);
	}

	public ConversationTokenProvider privateUpdateAssemblyType(AssemblyType at) {
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		AssemblyTypeService service = TmcdbContextFactory.INSTANCE.getAssemblyTypeService();
		service.update(at);
		return retVal;
	}

	private class AssemblyTypeHolder
	{
		private AssemblyType assemblyType;
		public AssemblyType getAssemblyType() { return assemblyType; }
		public void setAssemblyType(AssemblyType at) { this.assemblyType = at; }
	}
	
	public AssemblyType findAssemblyTypeForMountOfGivenAntennaType(AntennaType antennaType) 
	throws Exception
	{
		AssemblyType retVal = null;
		Method methodToInvoke = AssemblyTypeConversationUtils.class.getMethod("privateFindAssemblyTypeForMountGivenAntennaType", 
				AntennaType.class, AssemblyTypeHolder.class);	
		
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		AssemblyTypeHolder resultHolder = new AssemblyTypeHolder();
		Object[] args = new Object[2];
		args[0] = antennaType;
		args[1] = resultHolder;
		conversationInterceptor.invoke(methodToInvoke, this, args);
		
		retVal = resultHolder.getAssemblyType();
		return retVal;
	}
	
	@SuppressWarnings("unchecked")
	public ConversationTokenProvider privateFindAssemblyTypeForMountGivenAntennaType(AntennaType antType, AssemblyTypeHolder resultHolder) 
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		String searchString = "Mount";
		switch(antType) { 
		case AEC:
			searchString += "AEM";
			break;
		case VA:
			searchString += "Vertex";
			break;
		case ACA:
			searchString += "ACA";
			break;
		}
		
		AssemblyTypeService service = TmcdbContextFactory.INSTANCE.getAssemblyTypeService();
		List<AssemblyType> results = (List<AssemblyType>)service.findByName(searchString);
		if(results.size() == 0) {
			throw new IllegalStateException("Database does not have an assemblytype for: " + searchString);
		}
		else if(results.size() > 1) {
			throw new IllegalStateException("Database has more than one assemblytype for: " + searchString);
		}
		resultHolder.setAssemblyType(results.get(0));
		return retVal;
	}
}
