/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) NRAO - National Radio Astronomy Observatory, 2012
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA

 *******************************************************************************/
package alma.obops.tmcdbgui.utils.conversation;

import java.lang.reflect.Method;

import alma.obops.dam.config.TmcdbContextFactory;
import alma.obops.dam.tmcdb.service.AssemblyService;
import alma.obops.dam.tmcdb.service.AssemblyTypeService;
import alma.obops.dam.utils.ConversationInterceptor;
import alma.obops.dam.utils.ConversationTokenProvider;
import alma.obops.dam.utils.ConversationTokenProviderAdapter;
import alma.obops.dam.utils.ConversationTokenProvider.ConversationToken;
import alma.tmcdb.domain.Assembly;
import alma.tmcdb.domain.AssemblyType;
import alma.tmcdb.domain.HwConfiguration;

/**
 * @author sharring
 *
 */
public class AssemblyConversationUtils 
{
	private static AssemblyConversationUtils singletonInstance;

	private AssemblyConversationUtils() 
	{
	}

	public static synchronized AssemblyConversationUtils getInstance()
	{
		if(null == singletonInstance)
		{
			singletonInstance = new AssemblyConversationUtils();
		}

		return singletonInstance;
	}

	/************************** Assembly related methods ********************/
	public void hydrateAssemblyType(AssemblyType a) throws Exception {
		Method methodToInvoke = AssemblyConversationUtils.class.getMethod("privateHydrateAssemblyType", AssemblyType.class);
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[1];
		args[0] = a;
		conversationInterceptor.invoke(methodToInvoke, this, args);
	}

	public ConversationTokenProvider privateHydrateAssemblyType(AssemblyType a)
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		AssemblyTypeService assemblyTypeService =  TmcdbContextFactory.INSTANCE.getAssemblyTypeService();	   
		assemblyTypeService.hydrate(a);
		return retVal;
	}

	public void hydrateAssembly(Assembly assembly) throws Exception
	{
		Method methodToInvoke = AssemblyConversationUtils.class.getMethod("privateHydrateAssembly", Assembly.class);
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[1];
		args[0] = assembly;
		conversationInterceptor.invoke(methodToInvoke, this, args);
	}

	public ConversationTokenProvider privateHydrateAssembly(Assembly assembly)
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		AssemblyService service = TmcdbContextFactory.INSTANCE.getAssemblyService();
		service.hydrate(assembly);
		return retVal;
	}

	public void hydrateAssemblies(HwConfiguration config) 
	throws Exception 
	{
		Method methodToInvoke = AssemblyConversationUtils.class.getMethod("privateHydrateAssemblies", HwConfiguration.class);
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[1];
		args[0] = config;
		conversationInterceptor.invoke(methodToInvoke, this, args);
	}

	public ConversationTokenProvider privateHydrateAssemblies(HwConfiguration config) 
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		AssemblyService assemblyService =  TmcdbContextFactory.INSTANCE.getAssemblyService();
		for(Assembly assembly: config.getAssemblies()) {
			assemblyService.hydrate(assembly);	
		}
		return retVal;
	}
	
	public void saveOrUpdateAssembly(Assembly assembly) throws Exception {
		Method methodToInvoke = AssemblyConversationUtils.class.getMethod("privateSaveOrUpdateAssembly", Assembly.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[1];
		args[0] = assembly;
		conversationInterceptor.invoke(methodToInvoke, this, args);
	}

	public ConversationTokenProvider privateSaveOrUpdateAssembly(Assembly assembly) {
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		AssemblyService assemblyService = TmcdbContextFactory.INSTANCE.getAssemblyService();
		assemblyService.update(assembly);
		return retVal;
	}

	/**
	 * @param element the element for which we want to run in an attached mode
	 * @param runnable the runnable to run
	 */
	public void runWithAttachedObject(Object element, Runnable runnable) throws Exception 
	{
		Method methodToInvoke = AssemblyConversationUtils.class.getMethod("privateRunWithAttachedObject", Object.class, Runnable.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[2];
		args[0] = element;
		args[1] = runnable;
		conversationInterceptor.invoke(methodToInvoke, this, args);
	}

	public ConversationTokenProvider privateRunWithAttachedObject(Object obj, Runnable runnable)
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_PENDING);
		AssemblyService service = TmcdbContextFactory.INSTANCE.getAssemblyService();
		service.runWithAttachedObject(obj, runnable);
		return retVal;
	}
}
