/*******************************************************************************
  * ALMA - Atacama Large Millimeter Array
  * Copyright (c) NRAO - National Radio Astronomy Observatory, 2012
  * (in the framework of the ALMA collaboration).
  * All rights reserved.
  *
  * This library is free software; you can redistribute it and/or
  * modify it under the terms of the GNU Lesser General Public
  * License as published by the Free Software Foundation; either
  * version 2.1 of the License, or (at your option) any later version.
  *
  * This library is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  * Lesser General Public License for more details.
  *
  * You should have received a copy of the GNU Lesser General Public
  * License along with this library; if not, write to the Free Software
  * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 
*******************************************************************************/
package alma.obops.tmcdbgui.utils.conversation;

import java.lang.reflect.Method;
import java.util.List;

import alma.obops.dam.config.TmcdbContextFactory;
import alma.obops.dam.tmcdb.service.AntennaToPadService;
import alma.obops.dam.utils.ConversationInterceptor;
import alma.obops.dam.utils.ConversationTokenProvider;
import alma.obops.dam.utils.ConversationTokenProviderAdapter;
import alma.obops.dam.utils.ConversationTokenProvider.ConversationToken;
import alma.tmcdb.domain.AntennaToPad;
import alma.tmcdb.history.HistoryRecord;

/**
 * @author sharring
 *
 */
public class AntennaToPadConversationUtils 
{
	private static AntennaToPadConversationUtils singletonInstance;

	private AntennaToPadConversationUtils() 
	{
	}

	public static synchronized AntennaToPadConversationUtils getInstance()
	{
		if(null == singletonInstance)
		{
			singletonInstance = new AntennaToPadConversationUtils();
		}

		return singletonInstance;
	}
	
	public ConversationTokenProvider privateDeleteAntennaToPad(AntennaToPad a2p, ConversationToken token)
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(token);
		AntennaToPadService a2pService = TmcdbContextFactory.INSTANCE.getAntennaToPadService();
		a2pService.delete(a2p);
		return retVal;
	}
	
	public void hydrateAntennaToPad(AntennaToPad a2p) throws Exception
	{
		Method methodToInvoke = AntennaToPadConversationUtils.class.getMethod("privateHydrateAntennaToPad", AntennaToPad.class);
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[1];
		args[0] = a2p;
		conversationInterceptor.invoke(methodToInvoke, this, args);
	}
	
	public ConversationTokenProvider privateHydrateAntennaToPad(AntennaToPad a2p)
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		AntennaToPadService a2pService =  TmcdbContextFactory.INSTANCE.getAntennaToPadService();	   
		a2pService.hydrate(a2p);
		return retVal;
	}
	
	public void saveOrUpdateAntennaToPad(AntennaToPad a2p) throws Exception {
		saveOrUpdateAntennaToPad(a2p, ConversationToken.CONVERSATION_COMPLETED);
	}

	public void saveOrUpdateAntennaToPad(AntennaToPad a2p, ConversationToken conversationToken) throws Exception 
	{
		Method methodToInvoke = AntennaToPadConversationUtils.class.getMethod("privateSaveOrUpdateAntennaToPad", AntennaToPad.class, ConversationToken.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[2];
		args[0] = a2p;
		args[1] = conversationToken;
		conversationInterceptor.invoke(methodToInvoke, this, args);
	}
	
	public ConversationTokenProvider privateSaveOrUpdateAntennaToPad(AntennaToPad a2p, ConversationToken token) 
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(token);
		AntennaToPadService a2ps = TmcdbContextFactory.INSTANCE.getAntennaToPadService();
		a2ps.update(a2p);
		return retVal;
	}

	public List<HistoryRecord> getAntennaToPadHistory(AntennaToPad a2p) throws Exception
	{
		Method methodToInvoke = AntennaToPadConversationUtils.class.getMethod("privateGetAntennaToPadHistory", AntennaToPad.class, HistoryRecordListHolder.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		HistoryRecordListHolder holder = new HistoryRecordListHolder();
		Object[] args = new Object[2];
		args[0] = a2p;
		args[1] = holder;
		conversationInterceptor.invoke(methodToInvoke, this, args);
		return holder.getHistoryRecords();

	}

	public ConversationTokenProvider privateGetAntennaToPadHistory(AntennaToPad a2p, HistoryRecordListHolder resultHolder) 
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		AntennaToPadService service = TmcdbContextFactory.INSTANCE.getAntennaToPadService();
		List<HistoryRecord> results = service.getHistory(a2p);
		resultHolder.setHistoryRecords(results);
		return retVal;
	}

	private class AntennaToPadHolder
	{
		private AntennaToPad antennaToPad;
		
		public AntennaToPad getAntennaToPad() {
			return antennaToPad;
		}
		
		public void setAntennaToPad(AntennaToPad antennaToPad) {
			this.antennaToPad = antennaToPad;
		}
	}
	
	/**
	 * @param antennaToPad the a2p for which we want the historical version
	 * @param clickedRecord the history record indicating which version of the a2p we wish to retrieve
	 * @throws exception if there's a problem
	 * @return the a2p as it existed for the chosen moment in its history
	 */
	public AntennaToPad getHistoricalAntennaToPad(AntennaToPad antennaToPad,
			HistoryRecord clickedRecord) throws Exception 
	{
		Method methodToInvoke = AntennaToPadConversationUtils.class.getMethod("privateGetHistoricalAntennaToPad", AntennaToPad.class, HistoryRecord.class, AntennaToPadHolder.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		AntennaToPadHolder holder = new AntennaToPadHolder();
		Object[] args = new Object[3];
		args[0] = antennaToPad;
		args[1] = clickedRecord;
		args[2] = holder;
		conversationInterceptor.invoke(methodToInvoke, this, args);
		AntennaToPad retVal = holder.getAntennaToPad();
		return retVal;
	}
	
	public ConversationTokenProvider privateGetHistoricalAntennaToPad(AntennaToPad a2p, HistoryRecord record, AntennaToPadHolder resultHolder) 
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		AntennaToPadService service = TmcdbContextFactory.INSTANCE.getAntennaToPadService();
		AntennaToPad historicalAntennaToPad = service.getHistoricalAntennaToPad(a2p, record.getVersion());
		resultHolder.setAntennaToPad(historicalAntennaToPad);
		return retVal;
	}

	/**
	 * @param antennaToPad the a2p which we wish to save
	 * @param userId the userid of the person making the change
	 * @param description a description of the change
	 * @return a boolean indicating the a2p can be saved (true) or not (false)
	 * @throws exception if there's a problem
	 */
	public boolean prepareAntennaToPadSave(AntennaToPad antennaToPad,
			String userId, String description) throws Exception
	{
		Method methodToInvoke = AntennaToPadConversationUtils.class.getMethod("privatePrepareAntennaToPadSave", 
				AntennaToPad.class, String.class, String.class, BooleanHolder.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		BooleanHolder resultholder = new BooleanHolder();
		Object[] args = new Object[4];
		args[0] = antennaToPad;
		args[1] = userId;
		args[2] = description;
		args[3] = resultholder;
		conversationInterceptor.invoke(methodToInvoke, this, args);
		boolean retVal = resultholder.getBooleanValue();
		return retVal;
	}
	
	public ConversationTokenProvider privatePrepareAntennaToPadSave(AntennaToPad a2p, String who, String description, BooleanHolder resultHolder) 
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		AntennaToPadService service = TmcdbContextFactory.INSTANCE.getAntennaToPadService();
		boolean successful = service.prepareSave(a2p, who, description);
		resultHolder.setBooleanValue(successful);
		return retVal;
	}

	/**
	 * @param antennaToPad
	 */
	public void endAntennaToPadSave(AntennaToPad antennaToPad) throws Exception
	{
		Method methodToInvoke = AntennaToPadConversationUtils.class.getMethod("privateEndAntennaToPadSave", 
				AntennaToPad.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[1];
		args[0] = antennaToPad;
		conversationInterceptor.invoke(methodToInvoke, this, args);
	}
	
	public ConversationTokenProvider privateEndAntennaToPadSave(AntennaToPad a2p) 
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		AntennaToPadService service = TmcdbContextFactory.INSTANCE.getAntennaToPadService();
		service.endSave(a2p);
		return retVal;
	}
	
}
