/*******************************************************************************
  * ALMA - Atacama Large Millimeter Array
  * Copyright (c) NRAO - National Radio Astronomy Observatory, 2012
  * (in the framework of the ALMA collaboration).
  * All rights reserved.
  *
  * This library is free software; you can redistribute it and/or
  * modify it under the terms of the GNU Lesser General Public
  * License as published by the Free Software Foundation; either
  * version 2.1 of the License, or (at your option) any later version.
  *
  * This library is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  * Lesser General Public License for more details.
  *
  * You should have received a copy of the GNU Lesser General Public
  * License along with this library; if not, write to the Free Software
  * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 
*******************************************************************************/
package alma.obops.tmcdbgui.utils.conversation;

import java.lang.reflect.Method;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import alma.acs.tmcdb.AlarmCategory;
import alma.acs.tmcdb.AlarmDefinition;
import alma.acs.tmcdb.Configuration;
import alma.acs.tmcdb.Contact;
import alma.acs.tmcdb.DefaultMember;
import alma.acs.tmcdb.FaultCode;
import alma.acs.tmcdb.FaultFamily;
import alma.acs.tmcdb.FaultMember;
import alma.acs.tmcdb.Location;
import alma.acs.tmcdb.ReductionLink;
import alma.acs.tmcdb.ReductionLinkType;
import alma.acs.tmcdb.ReductionThreshold;
import alma.obops.dam.config.TmcdbContextFactory;
import alma.obops.dam.tmcdb.service.AlarmCategoryService;
import alma.obops.dam.tmcdb.service.AlarmDefinitionService;
import alma.obops.dam.tmcdb.service.ContactService;
import alma.obops.dam.tmcdb.service.DefaultMemberService;
import alma.obops.dam.tmcdb.service.FaultCodeService;
import alma.obops.dam.tmcdb.service.FaultFamilyService;
import alma.obops.dam.tmcdb.service.FaultMemberService;
import alma.obops.dam.tmcdb.service.LocationService;
import alma.obops.dam.tmcdb.service.ReductionLinkService;
import alma.obops.dam.tmcdb.service.ReductionThresholdService;
import alma.obops.dam.tmcdb.service.SwConfigurationService;
import alma.obops.dam.utils.ConversationInterceptor;
import alma.obops.dam.utils.ConversationTokenProvider;
import alma.obops.dam.utils.ConversationTokenProviderAdapter;
import alma.obops.dam.utils.ConversationTokenProvider.ConversationToken;

/**
 * @author sharring
 *
 */
public class AlarmConversationUtils 
{
	private static AlarmConversationUtils singletonInstance;

	private AlarmConversationUtils() 
	{
	}

	public static synchronized AlarmConversationUtils getInstance()
	{
		if(null == singletonInstance)
		{
			singletonInstance = new AlarmConversationUtils();
		}

		return singletonInstance;
	}
	
	public void saveOrUpdateAlarmDefinition(AlarmDefinition alarmDefinition, boolean conversationCompleted) throws Exception 
	{
		Method methodToInvoke = AlarmConversationUtils.class.getMethod("privateSaveOrUpdateAlarmDefinition", AlarmDefinition.class, Boolean.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[2];
		args[0] = alarmDefinition;
		args[1] = conversationCompleted;
		conversationInterceptor.invoke(methodToInvoke, this, args);	
	}
	
	public ConversationTokenProvider privateSaveOrUpdateAlarmDefinition(AlarmDefinition alarmDefinition, Boolean conversationCompleted) 
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_PENDING);
		if(conversationCompleted) {
			retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		}
		AlarmDefinitionService service = TmcdbContextFactory.INSTANCE.getAlarmDefinitionService();
		service.update(alarmDefinition);
		return retVal;
	}
	
	public void saveOrUpdateAlarmCategory(AlarmCategory category, boolean conversationCompleted) throws Exception 
	{
		Method methodToInvoke = AlarmConversationUtils.class.getMethod("privateSaveOrUpdateAlarmCategory", AlarmCategory.class, Boolean.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[2];
		args[0] = category;
		args[1] = conversationCompleted;
		conversationInterceptor.invoke(methodToInvoke, this, args);	
	}
	
	public ConversationTokenProvider privateSaveOrUpdateAlarmCategory(AlarmCategory category, Boolean conversationCompleted) {
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_PENDING);
		if(conversationCompleted) {
			retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		}
		AlarmCategoryService service = TmcdbContextFactory.INSTANCE.getAlarmCategoryService();
		service.update(category);
		return retVal;
	}
	
	public ConversationTokenProvider privateDeleteAlarmCategory(AlarmCategory category, ConversationToken token) {
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(token);
		AlarmCategoryService service = TmcdbContextFactory.INSTANCE.getAlarmCategoryService();
		
		// delete all the fault families, and their members, codes, default members
		for(FaultFamily family : category.getFaultFamilies()) 
		{
			FaultFamilyService ffs = TmcdbContextFactory.INSTANCE.getFaultFamilyService();
			for(FaultCode code : family.getFaultCodes()) {
				FaultCodeService fcs = TmcdbContextFactory.INSTANCE.getFaultCodeService();
				fcs.delete(code);
			}
			for(FaultMember member : family.getFaultMembers()) {
				FaultMemberService fms = TmcdbContextFactory.INSTANCE.getFaultMemberService();
				fms.delete(member);
			}
			for(DefaultMember member : family.getDefaultMembers()) {
				DefaultMemberService dms = TmcdbContextFactory.INSTANCE.getDefaultMemberService();
				dms.delete(member);
			}
			if(family.getAlarmCategories().size() == 1) {
				ffs.delete(family);
			}
		}
		
		service.delete(category);
		return retVal;
	}
	
	public ConversationTokenProvider privateDeleteDefaultMember(DefaultMember member, ConversationToken token)
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(token);
		DefaultMemberService service = TmcdbContextFactory.INSTANCE.getDefaultMemberService();
		service.delete(member);
		return retVal;
	}
	
	public ConversationTokenProvider privateDeleteFaultCode(FaultCode code, ConversationToken token)
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(token);
		FaultCodeService service = TmcdbContextFactory.INSTANCE.getFaultCodeService();
		service.delete(code);
		return retVal;
	}
	
	public ConversationTokenProvider privateDeleteFaultMember(FaultMember member, ConversationToken token)
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(token);
		FaultMemberService service = TmcdbContextFactory.INSTANCE.getFaultMemberService();
		service.delete(member);
		return retVal;
	}
	
	public ConversationTokenProvider privateDeleteReductionLink(ReductionLink link, ConversationToken token)
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(token);
		ReductionLinkService service = TmcdbContextFactory.INSTANCE.getReductionLinkService();
		service.delete(link);
		return retVal;
	}
	
	public ConversationTokenProvider privateDeleteReductionThreshold(ReductionThreshold threshold, ConversationToken token)
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(token);
		ReductionThresholdService service = TmcdbContextFactory.INSTANCE.getReductionThresholdService();
		service.delete(threshold);
		return retVal;
	}
	
	public ConversationTokenProvider privateDeleteAlarmDefinition(AlarmDefinition def, ConversationToken token)
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(token);
		AlarmDefinitionService service = TmcdbContextFactory.INSTANCE.getAlarmDefinitionService();
		service.delete(def);
		return retVal;
	}
	
	public ConversationTokenProvider privateDeleteFaultFamily(FaultFamily family, ConversationToken token)
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(token);
		FaultFamilyService service = TmcdbContextFactory.INSTANCE.getFaultFamilyService();
		service.delete(family);
		return retVal;
	}
	
	private class AlarmDefinitionsHolder 
	{
		private List< AlarmDefinition> alarmDefinitions;
		private List< AlarmDefinition> getAlarmDefinitions() { return alarmDefinitions; }
		private void setAlarmDefinitions(List< AlarmDefinition> alarmDefinitions ) { this.alarmDefinitions = alarmDefinitions; }
	}
	
	public List<AlarmDefinition> findAlarmDefinitionsWithReductionLinksForFaultFamily(FaultFamily faultFamily) throws Exception 
	{
		Method methodToInvoke = AlarmConversationUtils.class.getMethod("privateFindAlarmDefinitionsWithReductionLinksForFaultFamily", FaultFamily.class, AlarmDefinitionsHolder.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		AlarmDefinitionsHolder holder = new AlarmDefinitionsHolder();
		Object[] args = new Object[2];
		args[0] = faultFamily;
		args[1] = holder;
		conversationInterceptor.invoke(methodToInvoke, this, args);
		return holder.getAlarmDefinitions();
	}
	
	public ConversationTokenProvider privateFindAlarmDefinitionsWithReductionLinksForFaultFamily(FaultFamily family, AlarmDefinitionsHolder holder)
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_PENDING);
		FaultFamilyService service = TmcdbContextFactory.INSTANCE.getFaultFamilyService();
		holder.setAlarmDefinitions( service.findAlarmDefinitionsWithReductionLinksByFaultFamily(family) );
		return retVal;
	}
	
	public List<AlarmDefinition> findAlarmDefinitionsForFaultFamily(FaultFamily faultFamily) throws Exception 
	{
		Method methodToInvoke = AlarmConversationUtils.class.getMethod("privateFindAlarmDefinitionsForFaultFamily", FaultFamily.class, AlarmDefinitionsHolder.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		AlarmDefinitionsHolder holder = new AlarmDefinitionsHolder();
		Object[] args = new Object[2];
		args[0] = faultFamily;
		args[1] = holder;
		conversationInterceptor.invoke(methodToInvoke, this, args);
		return holder.getAlarmDefinitions();
	}
	
	public ConversationTokenProvider privateFindAlarmDefinitionsForFaultFamily(FaultFamily family, AlarmDefinitionsHolder holder)
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_PENDING);
		FaultFamilyService service = TmcdbContextFactory.INSTANCE.getFaultFamilyService();
		holder.setAlarmDefinitions( service.findAlarmDefinitionsByFaultFamily(family) );
		return retVal;
	}
	
	public List<AlarmDefinition> findAlarmDefinitionsByConfiguration(Configuration configuration) throws Exception 
	{
		Method methodToInvoke = AlarmConversationUtils.class.getMethod("privateFindAlarmDefinitionsByConfiguration", Configuration.class, AlarmDefinitionsHolder.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		AlarmDefinitionsHolder holder = new AlarmDefinitionsHolder();
		Object[] args = new Object[2];
		args[0] = configuration;
		args[1] = holder;
		conversationInterceptor.invoke(methodToInvoke, this, args);
		return holder.getAlarmDefinitions();
	}
	
	public ConversationTokenProvider privateFindAlarmDefinitionsByConfiguration(Configuration config, AlarmDefinitionsHolder holder)
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_PENDING);
		AlarmDefinitionService service = TmcdbContextFactory.INSTANCE.getAlarmDefinitionService();
		holder.setAlarmDefinitions( service.findAllInConfiguration(config) );
		return retVal;
	}
	
	public AlarmDefinition findMatchingAlarmDefinition(String familyName, String memberName, String faultCodeStr, Configuration config)
	   throws Exception
	{
		Method methodToInvoke = AlarmConversationUtils.class.getMethod("privateFindMatchingAlarmDefinition", 
				String.class, String.class, String.class, Configuration.class, AlarmDefinitionsHolder.class);	
		
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		AlarmDefinitionsHolder holder = new AlarmDefinitionsHolder();
		Object[] args = new Object[5];
		args[0] = familyName;
		args[1] = memberName;
		args[2] = faultCodeStr;
		args[3] = config;
		args[4] = holder;
		conversationInterceptor.invoke(methodToInvoke, this, args);
		AlarmDefinition retVal = null;
		if(holder.getAlarmDefinitions().size() > 1) {
			throw new IllegalStateException("Located more than one alarm definition within the configuration for the same FF, FM, FC values!");
		} else if (holder.getAlarmDefinitions().size() == 1) {
			retVal = holder.getAlarmDefinitions().get(0);
		} else {
			retVal = null;
		}
		return retVal;
	}
	
	public ConversationTokenProvider privateFindMatchingAlarmDefinition(String ff, String fm, String fc, Configuration config, AlarmDefinitionsHolder holder)
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_PENDING);
		AlarmDefinitionService service = TmcdbContextFactory.INSTANCE.getAlarmDefinitionService();
		holder.setAlarmDefinitions( service.findByValuesInConfiguration(ff, fm, fc, config) );
		return retVal;
	}
	
	private class FaultMembersHolder 
	{
		private List<FaultMember> faultMembers;
		private List<FaultMember> getFaultMembers() { return faultMembers; }
		private void setFaultMembers(List<FaultMember> faultMembers ) { this.faultMembers = faultMembers; }
	}
	
	public List<FaultMember> findFaultMembersByConfiguration(Configuration config) throws Exception
	{
		Method methodToInvoke = AlarmConversationUtils.class.getMethod("privateFindFaultMembersByConfiguration", Configuration.class, FaultMembersHolder.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		FaultMembersHolder holder = new FaultMembersHolder();
		Object[] args = new Object[2];
		args[0] = config;
		args[1] = holder;
		conversationInterceptor.invoke(methodToInvoke, this, args);
		return holder.getFaultMembers();
	}
	
	public ConversationTokenProvider privateFindFaultMembersByConfiguration(Configuration config, FaultMembersHolder holder)
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_PENDING);
		FaultMemberService service = TmcdbContextFactory.INSTANCE.getFaultMemberService();
		holder.setFaultMembers( service.findAllInConfiguration(config) );
		return retVal;
	}
	
	private class FaultCodesHolder 
	{
		private List<FaultCode> faultCodes;
		private List<FaultCode> getFaultCodes() { return faultCodes; }
		private void setFaultCodes(List<FaultCode> faultCodes ) { this.faultCodes = faultCodes; }
	}
	
	public List<FaultCode> findFaultCodesByConfiguration(Configuration config) throws Exception
	{
		Method methodToInvoke = AlarmConversationUtils.class.getMethod("privateFindFaultCodesByConfiguration", Configuration.class, FaultCodesHolder.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		FaultCodesHolder holder = new FaultCodesHolder();
		Object[] args = new Object[2];
		args[0] = config;
		args[1] = holder;
		conversationInterceptor.invoke(methodToInvoke, this, args);
		return holder.getFaultCodes();
	}
	
	public ConversationTokenProvider privateFindFaultCodesByConfiguration(Configuration config, FaultCodesHolder holder)
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_PENDING);
		FaultCodeService service = TmcdbContextFactory.INSTANCE.getFaultCodeService();
		holder.setFaultCodes( service.findAllInConfiguration(config) );
		return retVal;
	}
	
	private class LocationsHolder 
	{
		private List<Location> locations;
		private List<Location> getLocations() { return locations; }
		private void setLocations(List<Location> locations ) { this.locations = locations; }
	}
	
	public List<Location> findLocationsByName(String name) throws Exception
	{
		Method methodToInvoke = AlarmConversationUtils.class.getMethod("privateFindLocationsByName", String.class, LocationsHolder.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		LocationsHolder holder = new LocationsHolder();
		Object[] args = new Object[2];
		args[0] = name;
		args[1] = holder;
		conversationInterceptor.invoke(methodToInvoke, this, args);
		return holder.getLocations();
	}
	
	@SuppressWarnings("unchecked")
	public ConversationTokenProvider privateFindLocationsByName(String name, LocationsHolder holder)
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_PENDING);
		LocationService service = TmcdbContextFactory.INSTANCE.getLocationService();
		holder.setLocations( (List<Location>)service.findByName(name) );
		return retVal;
	}
	
	private class FaultFamiliesHolder 
	{
		private List<FaultFamily> faultFamilies;
		private List<FaultFamily> getFaultFamilies() { return faultFamilies; }
		private void setFaultFamilies(List<FaultFamily> faultFamilies ) { this.faultFamilies = faultFamilies; }
	}
	
	public List<FaultFamily> findFaultFamiliesByRegexp(String faultFamilyRegEx, Configuration config) throws Exception 
	{
		Method methodToInvoke = AlarmConversationUtils.class.getMethod("privateFindFaultFamilesByRegexp", String.class, Configuration.class, FaultFamiliesHolder.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		FaultFamiliesHolder holder = new FaultFamiliesHolder();
		Object[] args = new Object[3];
		args[0] = faultFamilyRegEx;
		args[1] = config;
		args[2] = holder;
		conversationInterceptor.invoke(methodToInvoke, this, args);
		return holder.getFaultFamilies();
	}
	
	public ConversationTokenProvider privateFindFaultFamilesByRegexp(String faultFamilyRegEx, Configuration config, FaultFamiliesHolder holder)
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_PENDING);
		FaultFamilyService service = TmcdbContextFactory.INSTANCE.getFaultFamilyService();
		holder.setFaultFamilies( service.findFaultFamiliesByRegExInConfig(faultFamilyRegEx, config));
		return retVal;
	}
	
	public List<FaultMember> findFaultMembersByRegexp(String faultMemberRegEx, String faultFamilyRegEx, Configuration config) throws Exception 
	{
		Method methodToInvoke = AlarmConversationUtils.class.getMethod("privateFindFaultMembersByRegexp", String.class, String.class, Configuration.class, FaultMembersHolder.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		FaultMembersHolder holder = new FaultMembersHolder();
		Object[] args = new Object[4];
		args[0] = faultMemberRegEx;
		args[1] = faultFamilyRegEx;
		args[2] = config;
		args[3] = holder;
		conversationInterceptor.invoke(methodToInvoke, this, args);
		return holder.getFaultMembers();
	}
	
	public ConversationTokenProvider privateFindFaultMembersByRegexp(String faultMemberRegEx, String faultFamilyRegEx, Configuration config, FaultMembersHolder holder)
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_PENDING);
		FaultMemberService service = TmcdbContextFactory.INSTANCE.getFaultMemberService();
		holder.setFaultMembers( service.findFaultMembersByRegExInConfig(faultMemberRegEx, faultFamilyRegEx, config));
		return retVal;
	}
	
	public List<FaultMember> findFaultMembersForFaultFamilyRegEx(String faultFamilyRegEx, Configuration config) throws Exception 
	{
		Method methodToInvoke = AlarmConversationUtils.class.getMethod("privateFindFaultMembersForFaultFamilyRegEx", String.class, Configuration.class, FaultMembersHolder.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		FaultMembersHolder holder = new FaultMembersHolder();
		Object[] args = new Object[3];
		args[0] = faultFamilyRegEx;
		args[1] = config;
		args[2] = holder;
		conversationInterceptor.invoke(methodToInvoke, this, args);
		return holder.getFaultMembers();
	}
	
	public ConversationTokenProvider privateFindFaultMembersForFaultFamilyRegEx(String faultFamilyRegEx, Configuration config, FaultMembersHolder holder)
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_PENDING);
		FaultFamilyService service = TmcdbContextFactory.INSTANCE.getFaultFamilyService();
		holder.setFaultMembers( service.findFaultMembersByFaultFamilyRegExInConfig(faultFamilyRegEx, config));
		return retVal;
	}
	
	public List<FaultCode> findFaultCodesByRegexp(String faultCodeRegEx, String faultFamilyRegEx, Configuration config) throws Exception 
	{
		Method methodToInvoke = AlarmConversationUtils.class.getMethod("privateFindFaultCodesByRegEx", String.class, String.class, Configuration.class, FaultCodesHolder.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		FaultCodesHolder holder = new FaultCodesHolder();
		Object[] args = new Object[4];
		args[0] = faultCodeRegEx;
		args[1] = faultFamilyRegEx;
		args[2] = config;
		args[3] = holder;
		conversationInterceptor.invoke(methodToInvoke, this, args);
		return holder.getFaultCodes();
	}
	
	public ConversationTokenProvider privateFindFaultCodesByRegEx(String faultCodeRegEx, String faultFamilyRegEx, Configuration config, FaultCodesHolder holder)
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_PENDING);
		FaultCodeService service = TmcdbContextFactory.INSTANCE.getFaultCodeService();
		holder.setFaultCodes( service.findFaultCodesByRegExInConfig(faultCodeRegEx, faultFamilyRegEx, config));
		return retVal;
	}
	
	public Collection<FaultCode> findFaultCodesForFaultFamilyRegEx(String faultFamilyRegEx, Configuration config) throws Exception
	{
		Method methodToInvoke = AlarmConversationUtils.class.getMethod("privateFindFaultCodesForFaultFamilyRegEx", String.class, Configuration.class, FaultCodesHolder.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		FaultCodesHolder holder = new FaultCodesHolder();
		Object[] args = new Object[3];
		args[0] = faultFamilyRegEx;
		args[1] = config;
		args[2] = holder;
		conversationInterceptor.invoke(methodToInvoke, this, args);
		return holder.getFaultCodes();
	}
	
	public ConversationTokenProvider privateFindFaultCodesForFaultFamilyRegEx(String faultFamilyRegEx, Configuration config, FaultCodesHolder holder)
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_PENDING);
		FaultFamilyService service = TmcdbContextFactory.INSTANCE.getFaultFamilyService();
		holder.setFaultCodes( service.findFaultCodesByFaultFamilyRegExInConfig(faultFamilyRegEx, config));
		return retVal;
	}
	
	private class ContactsHolder 
	{
		private List<Contact> contacts;
		private List<Contact> getContacts() { return contacts; }
		private void setContacts(List<Contact> contacts ) { this.contacts = contacts; }
	}

	public List<Contact> findContactsByName(String name) throws Exception 
	{
		Method methodToInvoke = AlarmConversationUtils.class.getMethod("privateFindContactsByName", String.class, ContactsHolder.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		ContactsHolder holder = new ContactsHolder();
		Object[] args = new Object[2];
		args[0] = name;
		args[1] = holder;
		conversationInterceptor.invoke(methodToInvoke, this, args);
		return holder.getContacts();
	}
	
	@SuppressWarnings("unchecked")
	public ConversationTokenProvider privateFindContactsByName(String name, ContactsHolder holder)
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_PENDING);
		ContactService contactService = TmcdbContextFactory.INSTANCE.getContactService();
		holder.setContacts( (List<Contact>)contactService.findByName(name) );
		return retVal;
	}
	
	public AlarmCategory findAlarmCategoryById(Integer alarmCategoryId) throws Exception 
	{
		Method methodToInvoke = AlarmConversationUtils.class.getMethod("privateFindAlarmCategoryById", Integer.class, AlarmCategoryHolder.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[2];
		args[0] = alarmCategoryId;
		AlarmCategoryHolder resultHolder = new AlarmCategoryHolder();
		args[1] = resultHolder;
		conversationInterceptor.invoke(methodToInvoke, this, args);
		return resultHolder.getAlarmCategory();
	}
	
	public ConversationTokenProvider privateFindAlarmCategoryById(Integer reductionLinkId, AlarmCategoryHolder resultHolder)
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_PENDING);
		AlarmCategoryService service = TmcdbContextFactory.INSTANCE.getAlarmCategoryService();
		resultHolder.setAlarmCategory((AlarmCategory)service.read(reductionLinkId));
		return retVal;
	}

	private class ReductionThresholdHolder
	{
		private ReductionThreshold rt;
		
		public ReductionThreshold getReductionThreshold() {
			return rt;
		}
		
		public void setReductionThreshold(ReductionThreshold rt) {
			this.rt = rt;
		}
	}
	
	public ReductionThreshold findReductionThresholdById(Integer alarmDefinitionId) throws Exception 
	{
		Method methodToInvoke = AlarmConversationUtils.class.getMethod("privateFindReductionThresholdById", Integer.class, ReductionThresholdHolder.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[2];
		args[0] = alarmDefinitionId;
		ReductionThresholdHolder resultHolder = new ReductionThresholdHolder();
		args[1] = resultHolder;
		conversationInterceptor.invoke(methodToInvoke, this, args);
		return resultHolder.getReductionThreshold();
	}
	
	public ConversationTokenProvider privateFindReductionThresholdById(Integer id, ReductionThresholdHolder resultHolder)
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_PENDING);
		ReductionThresholdService service = TmcdbContextFactory.INSTANCE.getReductionThresholdService();
		resultHolder.setReductionThreshold((ReductionThreshold)service.read(id));
		return retVal;
	}	
	public FaultFamily findFaultFamilyById(Integer faultFamilyId) throws Exception 
	{
		Method methodToInvoke = AlarmConversationUtils.class.getMethod("privateFindFaultFamilyById", Integer.class, FaultFamilyHolder.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[2];
		args[0] = faultFamilyId;
		FaultFamilyHolder resultHolder = new FaultFamilyHolder();
		args[1] = resultHolder;
		conversationInterceptor.invoke(methodToInvoke, this, args);
		return resultHolder.getFaultFamily();
	}
	
	public ConversationTokenProvider privateFindFaultFamilyById(Integer id, FaultFamilyHolder resultHolder)
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_PENDING);
		FaultFamilyService service = TmcdbContextFactory.INSTANCE.getFaultFamilyService();
		resultHolder.setFaultFamily((FaultFamily)service.read(id));
		return retVal;
	}
	
	public AlarmDefinition findAlarmDefinitionById(Integer alarmDefinitionId) throws Exception 
	{
		Method methodToInvoke = AlarmConversationUtils.class.getMethod("privateFindAlarmDefinitionById", Integer.class, AlarmDefinitionHolder.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[2];
		args[0] = alarmDefinitionId;
		AlarmDefinitionHolder resultHolder = new AlarmDefinitionHolder();
		args[1] = resultHolder;
		conversationInterceptor.invoke(methodToInvoke, this, args);
		return resultHolder.getAlarmDefinition();
	}
	
	public ConversationTokenProvider privateFindAlarmDefinitionById(Integer id, AlarmDefinitionHolder resultHolder)
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_PENDING);
		AlarmDefinitionService service = TmcdbContextFactory.INSTANCE.getAlarmDefinitionService();
		resultHolder.setAlarmDefinition((AlarmDefinition)service.read(id));
		return retVal;
	}
	
	private class ReductionLinkHolder
	{
		private ReductionLink redLink;
		
		public ReductionLink getReductionLink() {
			return redLink;
		}
		
		public void setReductionLink(ReductionLink rl) {
			this.redLink = rl;
		}
	}
	
	public ReductionLink findReductionLinkById(Integer reductionLinkId) throws Exception 
	{
		Method methodToInvoke = AlarmConversationUtils.class.getMethod("privateFindReductionLinkById", Integer.class, ReductionLinkHolder.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[2];
		args[0] = reductionLinkId;
		ReductionLinkHolder resultHolder = new ReductionLinkHolder();
		args[1] = resultHolder;
		conversationInterceptor.invoke(methodToInvoke, this, args);
		return resultHolder.getReductionLink();
	}
	
	public ConversationTokenProvider privateFindReductionLinkById(Integer reductionLinkId, ReductionLinkHolder resultHolder)
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_PENDING);
		ReductionLinkService service = TmcdbContextFactory.INSTANCE.getReductionLinkService();
		resultHolder.setReductionLink((ReductionLink)service.read(reductionLinkId));
		return retVal;
	}
	
	public ReductionLink hydrateReductionLink(ReductionLink reductionLink) throws Exception 
	{
		Method methodToInvoke = AlarmConversationUtils.class.getMethod("privateHydrateReductionLink", ReductionLink.class, ReductionLinkHolder.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[2];
		args[0] = reductionLink;
		ReductionLinkHolder resultHolder = new ReductionLinkHolder();
		args[1] = resultHolder;
		conversationInterceptor.invoke(methodToInvoke, this, args);
		return resultHolder.getReductionLink();
	}
	
	public ConversationTokenProvider privateHydrateReductionLink(ReductionLink reductionLink, ReductionLinkHolder resultHolder)
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_PENDING);
		ReductionLinkService service = TmcdbContextFactory.INSTANCE.getReductionLinkService();
//		resultHolder.setReductionLink(service.hydrateAndMerge(reductionLink));
		service.hydrate(reductionLink);
		resultHolder.setReductionLink(reductionLink);
		return retVal;
	}
	
	private class FaultMemberHolder
	{
		private FaultMember faultMem;
		
		public FaultMember getFaultMember() {
			return faultMem;
		}
		
		public void setFaultMember(FaultMember fm) {
			this.faultMem = fm;
		}
	}
	
	public FaultMember hydrateFaultMember(FaultMember faultMember) throws Exception {
		Method methodToInvoke = AlarmConversationUtils.class.getMethod("privateHydrateFaultMember", FaultMember.class, FaultMemberHolder.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[2];
		args[0] = faultMember;
		FaultMemberHolder resultHolder = new FaultMemberHolder();
		args[1] = resultHolder;
		conversationInterceptor.invoke(methodToInvoke, this, args);
		return resultHolder.getFaultMember();
	}
	
	public ConversationTokenProvider privateHydrateFaultMember(FaultMember faultMember, FaultMemberHolder resultHolder)
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_PENDING);
		FaultMemberService fmService = TmcdbContextFactory.INSTANCE.getFaultMemberService();
//		resultHolder.setFaultMember(fmService.hydrateAndMerge(faultMember));
		resultHolder.setFaultMember(faultMember);
		fmService.hydrate(faultMember);
		return retVal;
	}
	
	private class FaultCodeHolder
	{
		private FaultCode faultCode;
		
		public FaultCode getFaultCode() {
			return faultCode;
		}
		
		public void setFaultCode(FaultCode fc) {
			this.faultCode = fc;
		}
	}
	
	public FaultCode hydrateFaultCode(FaultCode faultCode) throws Exception 
	{
		Method methodToInvoke = AlarmConversationUtils.class.getMethod("privateHydrateFaultCode", FaultCode.class, FaultCodeHolder.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[2];
		args[0] = faultCode;
		FaultCodeHolder resultHolder = new FaultCodeHolder();
		args[1] = resultHolder;
		conversationInterceptor.invoke(methodToInvoke, this, args);
		return resultHolder.getFaultCode();
	}
	
	public ConversationTokenProvider privateHydrateFaultCode(FaultCode faultCode, FaultCodeHolder resultHolder)
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_PENDING);
		FaultCodeService service = TmcdbContextFactory.INSTANCE.getFaultCodeService();
//		resultHolder.setFaultCode(service.hydrateAndMerge(faultCode));
		resultHolder.setFaultCode(faultCode);
		service.hydrate(faultCode);
		return retVal;
	}

	private class AlarmCategoryHolder
	{
		private AlarmCategory alarmCat;
		
		public AlarmCategory getAlarmCategory() {
			return alarmCat;
		}
		
		public void setAlarmCategory(AlarmCategory ac) {
			this.alarmCat = ac;
		}
	}
	
	public AlarmCategory hydrateAlarmCategory(AlarmCategory alarmCategory) throws Exception 
	{
		Method methodToInvoke = AlarmConversationUtils.class.getMethod("privateHydrateAlarmCategory", AlarmCategory.class, AlarmCategoryHolder.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[2];
		args[0] = alarmCategory;
		AlarmCategoryHolder resultHolder = new AlarmCategoryHolder();
		args[1] = resultHolder;
		conversationInterceptor.invoke(methodToInvoke, this, args);
		return resultHolder.getAlarmCategory();
	}
	
	public ConversationTokenProvider privateHydrateAlarmCategory(AlarmCategory alarmCategory, AlarmCategoryHolder resultHolder)
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_PENDING);
		AlarmCategoryService service = TmcdbContextFactory.INSTANCE.getAlarmCategoryService();
//		resultHolder.setAlarmCategory(service.hydrateAndMerge(alarmCategory));
		resultHolder.setAlarmCategory(alarmCategory);
		service.hydrate(alarmCategory);
		return retVal;
	}
	
	private class AlarmDefinitionHolder
	{
		private AlarmDefinition alarmDef;
		
		public AlarmDefinition getAlarmDefinition() {
			return alarmDef;
		}
		
		public void setAlarmDefinition(AlarmDefinition ad) {
			this.alarmDef = ad;
		}
	}
	
	public AlarmDefinition hydrateAlarmDefinition(AlarmDefinition alarmDefinition) throws Exception 
	{
		Method methodToInvoke = AlarmConversationUtils.class.getMethod("privateHydrateAlarmDefinition", AlarmDefinition.class, AlarmDefinitionHolder.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[2];
		args[0] = alarmDefinition;
		AlarmDefinitionHolder resultHolder = new AlarmDefinitionHolder();
		args[1] = resultHolder;
		conversationInterceptor.invoke(methodToInvoke, this, args);
		return resultHolder.getAlarmDefinition();
	}
	
	public ConversationTokenProvider privateHydrateAlarmDefinition(AlarmDefinition alarmDefinition, AlarmDefinitionHolder resultHolder)
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_PENDING);
		AlarmDefinitionService service = TmcdbContextFactory.INSTANCE.getAlarmDefinitionService();
//		resultHolder.setAlarmDefinition(service.hydrateAndMerge(alarmDefinition));
		service.hydrate(alarmDefinition);
		resultHolder.setAlarmDefinition(alarmDefinition);
		service.hydrate(alarmDefinition);
		return retVal;
	}
	
	private class FaultFamilyHolder
	{
		private FaultFamily faultFam;
		
		public FaultFamily getFaultFamily() {
			return faultFam;
		}
		
		public void setFaultFamily(FaultFamily ff) {
			this.faultFam = ff;
		}
	}
	
	public FaultFamily hydrateFaultFamily(FaultFamily faultFamily) throws Exception {
		Method methodToInvoke = AlarmConversationUtils.class.getMethod("privateHydrateFaultFamily", FaultFamily.class, FaultFamilyHolder.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[2];
		args[0] = faultFamily;
		FaultFamilyHolder resultHolder = new FaultFamilyHolder();
		args[1] = resultHolder;
		conversationInterceptor.invoke(methodToInvoke, this, args);
		return resultHolder.getFaultFamily();
	}
	
	public ConversationTokenProvider privateHydrateFaultFamily(FaultFamily faultFamily, FaultFamilyHolder resultHolder)
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_PENDING);
		FaultFamilyService ffService = TmcdbContextFactory.INSTANCE.getFaultFamilyService();
//		resultHolder.setFaultFamily(ffService.hydrateAndMerge(faultFamily));
		resultHolder.setFaultFamily(faultFamily);
		ffService.hydrate(faultFamily);
		return retVal;
	}
	

	public void hydrateReductionThresholds(Configuration conf) throws Exception 
	{
		Method methodToInvoke = AlarmConversationUtils.class.getMethod("privateHydrateReductionThresholds", Configuration.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[1];
		args[0] = conf;
		conversationInterceptor.invoke(methodToInvoke, this, args);
	}
	
	public ConversationTokenProvider privateHydrateReductionThresholds(Configuration config)
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_PENDING);
		SwConfigurationService configService = TmcdbContextFactory.INSTANCE.getSwConfigurationService();
		configService.hydrateReductionThresholds(config);
		return retVal;
	}
	
	public void hydrateReductionLinks(Configuration conf) throws Exception 
	{
		Method methodToInvoke = AlarmConversationUtils.class.getMethod("privateHydrateReductionLinks", Configuration.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[1];
		args[0] = conf;
		conversationInterceptor.invoke(methodToInvoke, this, args);
	}
	
	public ConversationTokenProvider privateHydrateReductionLinks(Configuration config)
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_PENDING);
		SwConfigurationService configService = TmcdbContextFactory.INSTANCE.getSwConfigurationService();
		configService.hydrateReductionLinks(config);
		return retVal;
	}
	
	public void hydrateAlarmCategories(Configuration configuration) throws Exception 
	{
		Method methodToInvoke = AlarmConversationUtils.class.getMethod("privateHydrateAlarmCategories", Configuration.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[1];
		args[0] = configuration;
		conversationInterceptor.invoke(methodToInvoke, this, args);
	}
	
	public ConversationTokenProvider privateHydrateAlarmCategories(Configuration config)
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_PENDING);
		SwConfigurationService configService = TmcdbContextFactory.INSTANCE.getSwConfigurationService();
		configService.hydrateAlarmCategories(config);
		return retVal;
	}
	

	public void hydrateAlarmDefinitions(Configuration conf) throws Exception 
	{
		Method methodToInvoke = AlarmConversationUtils.class.getMethod("privateHydrateAlarmDefinitions", Configuration.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[1];
		args[0] = conf;
		conversationInterceptor.invoke(methodToInvoke, this, args);
	}
	
	public ConversationTokenProvider privateHydrateAlarmDefinitions(Configuration config)
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_PENDING);
		SwConfigurationService configService = TmcdbContextFactory.INSTANCE.getSwConfigurationService();
		configService.hydrateAlarmDefinitions(config);
		return retVal;
	}
	
	public void hydrateFaultFamilies(Configuration configuration) throws Exception 
	{
		Method methodToInvoke = AlarmConversationUtils.class.getMethod("privateHydrateFaultFamilies", Configuration.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[1];
		args[0] = configuration;
		conversationInterceptor.invoke(methodToInvoke, this, args);
	}
	
	public ConversationTokenProvider privateHydrateFaultFamilies(Configuration config)
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_PENDING);
		SwConfigurationService configService = TmcdbContextFactory.INSTANCE.getSwConfigurationService();
		configService.hydrateFaultFamilies(config);
		return retVal;
	}
	
	private class DefaultMemberHolder
	{
		private DefaultMember defMem;
		
		public DefaultMember getDefaultMember() {
			return defMem;
		}
		
		public void setDefaultMember(DefaultMember dm) {
			this.defMem = dm;
		}
	}
	
	public DefaultMember hydrateDefaultMember(DefaultMember defaultMember) throws Exception
	{
		Method methodToInvoke = AlarmConversationUtils.class.getMethod("privateHydrateDefaultMember", DefaultMember.class, DefaultMemberHolder.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[2];
		args[0] = defaultMember;
		DefaultMemberHolder resultHolder = new DefaultMemberHolder();
		args[1] = resultHolder;
		conversationInterceptor.invoke(methodToInvoke, this, args);
		return resultHolder.getDefaultMember();
	}
	
	public ConversationTokenProvider privateHydrateDefaultMember(DefaultMember defaultMember, DefaultMemberHolder resultHolder)
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_PENDING);
		DefaultMemberService dfmService = TmcdbContextFactory.INSTANCE.getDefaultMemberService();
//		resultHolder.setDefaultMember(dfmService.hydrateAndMerge(defaultMember));
		dfmService.hydrate(defaultMember);
		resultHolder.setDefaultMember(defaultMember);
		return retVal;
	}
	
	public void hydrateReductionThreshold(ReductionThreshold redThreshold) throws Exception 
	{
		Method methodToInvoke = AlarmConversationUtils.class.getMethod("privateHydrateReductionThreshold", ReductionThreshold.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[1];
		args[0] = redThreshold;
		conversationInterceptor.invoke(methodToInvoke, this, args);
	}
	
	public ConversationTokenProvider privateHydrateReductionThreshold(ReductionThreshold reductionThreshold)
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_PENDING);
		ReductionThresholdService service = TmcdbContextFactory.INSTANCE.getReductionThresholdService();
		service.hydrate(reductionThreshold);
		return retVal;
	}
	

	public FaultCode findFaultCodeById(Integer faultCodeId) throws Exception 
	{
		Method methodToInvoke = AlarmConversationUtils.class.getMethod("privateFindFaultCodeById", Integer.class, FaultCodeHolder.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[2];
		args[0] = faultCodeId;
		FaultCodeHolder resultHolder = new FaultCodeHolder();
		args[1] = resultHolder;
		conversationInterceptor.invoke(methodToInvoke, this, args);
		return resultHolder.getFaultCode();
	}
	
	public ConversationTokenProvider privateFindFaultCodeById(Integer id, FaultCodeHolder resultHolder)
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_PENDING);
		FaultCodeService service = TmcdbContextFactory.INSTANCE.getFaultCodeService();
		resultHolder.setFaultCode((FaultCode)service.read(id));
		return retVal;
	}

	public DefaultMember findDefaultMemberById(Integer defaultMemberId) throws Exception 
	{
		Method methodToInvoke = AlarmConversationUtils.class.getMethod("privateFindDefaultMemberById", Integer.class, DefaultMemberHolder.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[2];
		args[0] = defaultMemberId;
		DefaultMemberHolder resultHolder = new DefaultMemberHolder();
		args[1] = resultHolder;
		conversationInterceptor.invoke(methodToInvoke, this, args);
		return resultHolder.getDefaultMember();
	}
	
	public ConversationTokenProvider privateFindDefaultMemberById(Integer id, DefaultMemberHolder resultHolder)
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_PENDING);
		DefaultMemberService service = TmcdbContextFactory.INSTANCE.getDefaultMemberService();
		resultHolder.setDefaultMember((DefaultMember)service.read(id));
		return retVal;
	}
	
	public FaultMember findFaultMemberById(Integer faultMemberId) throws Exception 
	{
		Method methodToInvoke = AlarmConversationUtils.class.getMethod("privateFindFaultMemberById", Integer.class, FaultMemberHolder.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[2];
		args[0] = faultMemberId;
		FaultMemberHolder resultHolder = new FaultMemberHolder();
		args[1] = resultHolder;
		conversationInterceptor.invoke(methodToInvoke, this, args);
		return resultHolder.getFaultMember();
	}
	
	public ConversationTokenProvider privateFindFaultMemberById(Integer id, FaultMemberHolder resultHolder)
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_PENDING);
		FaultMemberService service = TmcdbContextFactory.INSTANCE.getFaultMemberService();
		resultHolder.setFaultMember((FaultMember)service.read(id));
		return retVal;
	}
	
	public void removeFaultFamily(FaultFamily faultFamily, Configuration config, boolean commit) throws Exception 
	{
		boolean removed = false;
		
		AlarmConversationUtils.getInstance().hydrateFaultFamily(faultFamily);
		for(AlarmCategory cat: faultFamily.getAlarmCategories()) 
		{
			removed = cat.getFaultFamilies().remove(faultFamily);
			if(!removed) {
				removeFaultFamilyFromAlarmCategoryHack(cat, faultFamily);
			}
			AlarmConversationUtils.getInstance().saveOrUpdateAlarmCategory(cat, false);
		}
		
		Contact contact = faultFamily.getContact();
		removed = contact.getFaultFamilies().remove(faultFamily);
		if(!removed) {
			removeFaultFamilyFromContactHack(contact, faultFamily);
		}
		AlarmConversationUtils.getInstance().saveOrUpdateContact(contact, false);
		
		for(DefaultMember member : faultFamily.getDefaultMembers()) {
			BackendConversationUtils.getInstance().delete(member, ConversationToken.CONVERSATION_PENDING);
		}
		for(FaultMember member : faultFamily.getFaultMembers()) {
			BackendConversationUtils.getInstance().delete(member, ConversationToken.CONVERSATION_PENDING);
		}
		for(FaultCode code : faultFamily.getFaultCodes()) {
			BackendConversationUtils.getInstance().delete(code, ConversationToken.CONVERSATION_PENDING);
		}
		
		BackendConversationUtils.getInstance().delete(faultFamily, ConversationToken.CONVERSATION_PENDING);
		
		removed = config.getFaultFamilies().remove(faultFamily);
		if(!removed) {
			removeFaultFamilyFromConfigHack(config, faultFamily);
		}
		
		SwConfigurationConversationUtils.getInstance().saveOrUpdateSwConfiguration(config, commit);
	}
	
	private void removeFaultFamilyFromContactHack(Contact contact, FaultFamily ff) 
	{
		Set<FaultFamily> newSet = new HashSet<FaultFamily>();
		Set<FaultFamily> oldSet = contact.getFaultFamilies();
		for(FaultFamily ffamily: oldSet) 
		{
			if( (ffamily.hashCode() != ff.hashCode() ||  !ffamily.equals(ff))) 
			{
				if(!ffamily.getFaultFamilyId().equals(ff.getFaultFamilyId())) {
					newSet.add(ffamily);					
				}
			}
		}
		assert (newSet.size() != oldSet.size());
		contact.setFaultFamilies(newSet);
	}

	private void removeFaultFamilyFromAlarmCategoryHack(AlarmCategory cat, FaultFamily ff) 
	{
		Set<FaultFamily> newSet = new HashSet<FaultFamily>();
		Set<FaultFamily> oldSet = cat.getFaultFamilies();
		
		for(FaultFamily ffamily: oldSet) 
		{
			if( (ffamily.hashCode() != ff.hashCode() ||  !ffamily.equals(ff))) 
			{
				if(!ffamily.getFaultFamilyId().equals(ff.getFaultFamilyId())) {
					newSet.add(ffamily);					
				}
			}
		}
		assert (newSet.size() != oldSet.size());
		cat.setFaultFamilies(newSet);
	}

	private void removeFaultFamilyFromConfigHack(Configuration config, FaultFamily faultFamilyToRemove) 
	{
		Set<FaultFamily> newSet = new HashSet<FaultFamily>();
		Set<FaultFamily> oldSet = config.getFaultFamilies();
		for(FaultFamily ffamily: oldSet) 
		{
			if( (ffamily.hashCode() != faultFamilyToRemove.hashCode() ||  !ffamily.equals(faultFamilyToRemove))) 
			{
				if(!ffamily.getFaultFamilyId().equals(faultFamilyToRemove.getFaultFamilyId())) 
				{
					newSet.add(ffamily);
				}
			}
		}
		assert (newSet.size() != oldSet.size());
		config.setFaultFamilies(newSet);
	}

	public void removeAlarmCategory(Configuration config, AlarmCategory category) throws Exception 
	{
		for(FaultFamily family : category.getFaultFamilies())
		{
			AlarmConversationUtils.getInstance().removeFaultFamily(family, config, false);
		}
		BackendConversationUtils.getInstance().delete(category, ConversationToken.CONVERSATION_PENDING);
		boolean removed = config.getAlarmCategories().remove(category);
		if(!removed) {
			removeAlarmCategoryFromConfigurationHack(config, category);
		}
		SwConfigurationConversationUtils.getInstance().saveOrUpdateSwConfiguration(config, true);
	}
	
	private void removeAlarmCategoryFromConfigurationHack(Configuration config, AlarmCategory categoryToRemove) 
	{
		Set<AlarmCategory> newSet = new HashSet<AlarmCategory>();
		Set<AlarmCategory> oldSet = config.getAlarmCategories();
		
		for(AlarmCategory cat: oldSet) 
		{
			if( (categoryToRemove.hashCode() != cat.hashCode() ||  !cat.equals(categoryToRemove))) 
			{
				if(!cat.getAlarmCategoryId().equals(categoryToRemove.getAlarmCategoryId())) {
					newSet.add(cat);					
				}
			}
		}
		assert (newSet.size() != oldSet.size());
		config.setAlarmCategories(newSet);	
	}

	public void removeFaultCode(FaultFamily ff, FaultCode faultCodeToRemove) throws Exception 
	{
		boolean removed = ff.getFaultCodes().remove(faultCodeToRemove);
		if(!removed) {
			removeFaultCodeFromFaultFamilyHack(ff, faultCodeToRemove);
		}
		BackendConversationUtils.getInstance().delete(faultCodeToRemove, ConversationToken.CONVERSATION_PENDING);
		AlarmConversationUtils.getInstance().saveOrUpdateFaultFamily(ff, true);
	}

	public void removeDefaultMember(FaultFamily ff, Location location, DefaultMember member) throws Exception 
	{
		boolean removed = ff.getDefaultMembers().remove(member);
		if(!removed) {
			removeDefaultMemberFromFaultFamilyHack(ff, member);
		}
		
		BackendConversationUtils.getInstance().delete(member, ConversationToken.CONVERSATION_PENDING);
		
		if(null != location)
		{
			removed = location.getDefaultMembers().remove(member);
			if(!removed) {
				removeDefaultMemberFromLocationHack(location, member);
			}
			AlarmConversationUtils.getInstance().saveOrUpdateLocation(location, false);
		}
		
		AlarmConversationUtils.getInstance().saveOrUpdateFaultFamily(ff, true);
	}
	
	private void removeFaultCodeFromFaultFamilyHack(FaultFamily ff, FaultCode faultCodeToRemove) 
	{
		Set<FaultCode> newSet = new HashSet<FaultCode>();
		Set<FaultCode> oldSet = ff.getFaultCodes();
		for(FaultCode fcode: oldSet) 
		{
			if( (fcode.hashCode() != faultCodeToRemove.hashCode() ||  !fcode.equals(faultCodeToRemove))) 
			{
				if(!fcode.getFaultCodeId().equals(faultCodeToRemove.getFaultCodeId())) {
					newSet.add(fcode);					
				}
			}
		}
		assert (newSet.size() != oldSet.size());
		ff.setFaultCodes(newSet);
	}

	public void removeFaultMember(FaultFamily ff, Location location, FaultMember member) throws Exception 
	{
		boolean removed = ff.getFaultMembers().remove(member);
		if(!removed) {
			removeFaultMemberFromFaultFamilyHack(ff, member);
		}

		BackendConversationUtils.getInstance().delete(member, ConversationToken.CONVERSATION_PENDING);
		
		if(null != location)
		{
			removed = location.getFaultMembers().remove(member);
			if(!removed) {
				removeFaultMemberFromLocationHack(location, member);
			}
			AlarmConversationUtils.getInstance().saveOrUpdateLocation(location, false);
		}

		AlarmConversationUtils.getInstance().saveOrUpdateFaultFamily(ff, true);
	}

	private void removeFaultMemberFromLocationHack(Location location, FaultMember memberToRemove) 
	{
		Set<FaultMember> newSet = new HashSet<FaultMember>();
		Set<FaultMember> oldSet = location.getFaultMembers();
		for(FaultMember fmember: oldSet) 
		{
			if( (fmember.hashCode() != memberToRemove.hashCode() ||  !fmember.equals(memberToRemove))) 
			{
				if(!fmember.getFaultMemberId().equals(memberToRemove.getFaultMemberId()))
				{
					newSet.add(fmember);
				}
			}
		}
		assert (newSet.size() != oldSet.size());
		location.setFaultMembers(newSet);
	}

	private void removeFaultMemberFromFaultFamilyHack(FaultFamily family, FaultMember memberToRemove) 
	{
		Set<FaultMember> newSet = new HashSet<FaultMember>();
		Set<FaultMember> oldSet = family.getFaultMembers();
		
		for(FaultMember fmember: oldSet) 
		{
			if( (fmember.hashCode() != memberToRemove.hashCode() ||  !fmember.equals(memberToRemove))) 
			{
				if(!(fmember.getFaultMemberId().equals(memberToRemove.getFaultMemberId())))
				{
					newSet.add(fmember);
				}
			}
		}
		assert (newSet.size() != oldSet.size());
		family.setFaultMembers(newSet);
	}
	
	private void removeDefaultMemberFromLocationHack(Location location, DefaultMember memberToRemove) 
	{
		Set<DefaultMember> newSet = new HashSet<DefaultMember>();
		Set<DefaultMember> oldSet = location.getDefaultMembers();
		for(DefaultMember fmember: oldSet) 
		{
			if( (fmember.hashCode() != memberToRemove.hashCode() ||  !fmember.equals(memberToRemove))) 
			{
				if(!fmember.getDefaultMemberId().equals(memberToRemove.getDefaultMemberId()))
				{
					newSet.add(fmember);
				}
			}
		}
		assert (newSet.size() != oldSet.size());
		location.setDefaultMembers(newSet);
	}

	private void removeDefaultMemberFromFaultFamilyHack(FaultFamily family, DefaultMember memberToRemove) 
	{
		Set<DefaultMember> newSet = new HashSet<DefaultMember>();
		Set<DefaultMember> oldSet = family.getDefaultMembers();
		for(DefaultMember fmember: oldSet) 
		{
			if( (fmember.hashCode() != memberToRemove.hashCode() ||  !fmember.equals(memberToRemove))) 
			{
				if(!fmember.getDefaultMemberId().equals(memberToRemove.getDefaultMemberId()))
				{
					newSet.add(fmember);
				}
			}
		}
		assert (newSet.size() != oldSet.size());
		family.setDefaultMembers(newSet);
	}

	public void removeReductionLink(Configuration config, AlarmDefinition parentDef, AlarmDefinition childDef, ReductionLink reductionLinkToRemove) 
	   throws Exception
	{
		BackendConversationUtils.getInstance().delete(reductionLinkToRemove, ConversationToken.CONVERSATION_PENDING);
		
		boolean removed = config.getReductionLinks().remove(reductionLinkToRemove);
		if(!removed) {
			removeReductionLinkFromConfigurationHack(config, reductionLinkToRemove);
		}
		
		if(null != parentDef) {
			removed = parentDef.getReductionLinksForParentalarmdefid().remove(reductionLinkToRemove);
			if(!removed) {
				removeReductionLinkForParentAlarmDefinitionHack(parentDef, reductionLinkToRemove);
			}
		}
		
		if(null != childDef) {
			removed = childDef.getReductionLinksForChildalarmdefid().remove(reductionLinkToRemove);
			if(!removed) {
				removeReductionLinkForChildAlarmDefinitionHack(childDef, reductionLinkToRemove);
			}
		}
		
		boolean foundMultiplicityReduction = false;
		boolean deletedParentDef = false;
		
		if(null != parentDef)
		{
			for(ReductionLink link : parentDef.getReductionLinksForParentalarmdefid()) {
				if(link.getType().equals(ReductionLinkType.MULTIPLICITY)) {
					foundMultiplicityReduction = true; 
					break;
				}
			}

			if(!foundMultiplicityReduction && null != parentDef.getReductionThreshold()) {
				BackendConversationUtils.getInstance().delete(parentDef.getReductionThreshold(), ConversationToken.CONVERSATION_PENDING);
				parentDef.setReductionThreshold(null);
			}

			if(parentDef.getReductionLinksForChildalarmdefid().size() == 0 && 
					parentDef.getReductionLinksForParentalarmdefid().size() == 0)
			{
				BackendConversationUtils.getInstance().delete(parentDef, ConversationToken.CONVERSATION_PENDING);
				deletedParentDef = true;
			}
		}

		boolean deletedChildDef = false;
		if(null != childDef)
		{
			if(childDef.getReductionLinksForChildalarmdefid().size() == 0 && 
					childDef.getReductionLinksForParentalarmdefid().size() == 0)
			{
				BackendConversationUtils.getInstance().delete(childDef, ConversationToken.CONVERSATION_PENDING);
				deletedChildDef = true;
			}
		}
			
		if(!deletedParentDef) {
			AlarmConversationUtils.getInstance().saveOrUpdateAlarmDefinition(parentDef, false);			
		}
		if(!deletedChildDef) {
			AlarmConversationUtils.getInstance().saveOrUpdateAlarmDefinition(childDef, false);			
		}
		
		SwConfigurationConversationUtils.getInstance().saveOrUpdateSwConfiguration(config, true);
	}

	private void removeReductionLinkForChildAlarmDefinitionHack(AlarmDefinition childDef, ReductionLink linkToRemove) 
	{
		Set<ReductionLink> newSet = new HashSet<ReductionLink>();
		Set<ReductionLink> oldSet = childDef.getReductionLinksForChildalarmdefid();
		for(ReductionLink link: oldSet) 
		{
			if( (link.hashCode() != link.hashCode() ||  !link.equals(linkToRemove))) 
			{
				if(!link.getReductionLinkId().equals(linkToRemove.getReductionLinkId()))
				{
					newSet.add(link);
				}
			}
		}
		assert (newSet.size() != oldSet.size());
		childDef.setReductionLinksForChildalarmdefid(newSet);
		
	}

	private void removeReductionLinkForParentAlarmDefinitionHack(AlarmDefinition parentDef, ReductionLink linkToRemove) 
	{
		Set<ReductionLink> newSet = new HashSet<ReductionLink>();
		Set<ReductionLink> oldSet = parentDef.getReductionLinksForParentalarmdefid();
		for(ReductionLink link: oldSet) 
		{
			if( (link.hashCode() != link.hashCode() ||  !link.equals(linkToRemove))) 
			{
				if(!link.getReductionLinkId().equals(linkToRemove.getReductionLinkId()))
				{
					newSet.add(link);
				}
			}
		}
		assert (newSet.size() != oldSet.size());
		parentDef.setReductionLinksForParentalarmdefid(newSet);
	}

	private void removeReductionLinkFromConfigurationHack(Configuration config, ReductionLink linkToRemove) 
	{
		Set<ReductionLink> newSet = new HashSet<ReductionLink>();
		Set<ReductionLink> oldSet = config.getReductionLinks();
		for(ReductionLink link: oldSet) 
		{
			if( (link.hashCode() != link.hashCode() ||  !link.equals(linkToRemove))) 
			{
				if(!link.getReductionLinkId().equals(linkToRemove.getReductionLinkId()))
				{
					newSet.add(link);
				}
			}
		}
		assert (newSet.size() != oldSet.size());
		config.setReductionLinks(newSet);
	}
	
	public void saveOrUpdateFaultCode(FaultCode newFaultCode, boolean conversationCompleted) throws Exception 
	{
		Method methodToInvoke = AlarmConversationUtils.class.getMethod("privateSaveOrUpdateFaultCode", FaultCode.class, Boolean.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[2];
		args[0] = newFaultCode;
		args[1] = conversationCompleted;
		conversationInterceptor.invoke(methodToInvoke, this, args);
	}

	public ConversationTokenProvider privateSaveOrUpdateFaultCode(FaultCode faultCode, Boolean conversationCompleted) 
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_PENDING);
		if(conversationCompleted) {
			retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		}
		FaultCodeService fcService = TmcdbContextFactory.INSTANCE.getFaultCodeService();
//		FaultCode fc = fcService.hydrateAndMerge(faultCode);
//		fcService.update(fc);
		fcService.update(faultCode);
		return retVal;
	}
	
	public void saveOrUpdateFaultMember(FaultMember newFaultMember, boolean conversationCompleted) throws Exception 
	{
		Method methodToInvoke = AlarmConversationUtils.class.getMethod("privateSaveOrUpdateFaultMember", FaultMember.class, Boolean.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[2];
		args[0] = newFaultMember;
		args[1] = conversationCompleted;
		conversationInterceptor.invoke(methodToInvoke, this, args);
	}
	
	public ConversationTokenProvider privateSaveOrUpdateFaultMember(FaultMember faultmember, Boolean conversationCompleted) 
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_PENDING);
		if(conversationCompleted) {
			retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		}
		FaultMemberService fmService = TmcdbContextFactory.INSTANCE.getFaultMemberService();
		fmService.update(faultmember);
		return retVal;
	}
	
	public void saveOrUpdateFaultFamily(FaultFamily newFaultFamily, boolean conversationCompleted) throws Exception 
	{
		Method methodToInvoke = AlarmConversationUtils.class.getMethod("privateSaveOrUpdateFaultFamily", FaultFamily.class, Boolean.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[2];
		args[0] = newFaultFamily;
		args[1] = conversationCompleted;
		conversationInterceptor.invoke(methodToInvoke, this, args);
	}

	public ConversationTokenProvider privateSaveOrUpdateFaultFamily(FaultFamily faultfamily, Boolean conversationCompleted) 
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_PENDING);
		if(conversationCompleted) {
			retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		}
		FaultFamilyService ffService = TmcdbContextFactory.INSTANCE.getFaultFamilyService();
		ffService.update(faultfamily);
		return retVal;
	}
	
	public FaultFamily[] getFaultFamilies(Configuration swConfig, ConversationToken token) throws Exception 
	{
		Method methodToInvoke = AlarmConversationUtils.class.getMethod("privateGetFaultFamilies", Configuration.class, FaultFamilyArrayHolder.class, ConversationToken.class);
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[3];
		args[0] = swConfig;
		FaultFamilyArrayHolder ffArrayHolder = new FaultFamilyArrayHolder();
		args[1] = ffArrayHolder;
		args[2] = token;
		
		conversationInterceptor.invoke(methodToInvoke, this, args);
		return ffArrayHolder.getFaultFamilies();
		
	}
	
	private class FaultFamilyArrayHolder
	{
		private FaultFamily[] faultFamilies;
		public FaultFamily[] getFaultFamilies() { return faultFamilies; }
		public void setFaultFamilies(FaultFamily[] families) { this.faultFamilies = families; }
	}
	
	public ConversationTokenProvider privateGetFaultFamilies(Configuration swConfig, FaultFamilyArrayHolder resultHolder, ConversationToken token)
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(token);
		FaultFamilyService ffService = TmcdbContextFactory.INSTANCE.getFaultFamilyService();
		List<FaultFamily> families = ffService.findAllInConfig(swConfig);
		resultHolder.setFaultFamilies(families.toArray(new FaultFamily[0]));
		return retVal;
	}
	
	private class FaultMemberArrayHolder
	{
		private FaultMember[] faultMembers;
		public FaultMember[] getFaultMembers() { return faultMembers; }
		public void setFaultMembers(FaultMember[] members) { this.faultMembers = members; }
	}

	public FaultMember[] getFaultMembersForFaultFamily(FaultFamily family, ConversationToken token) throws Exception 
	{
		Method methodToInvoke = AlarmConversationUtils.class.getMethod("privateGetFaultMembersForFaultFamily", FaultFamily.class, FaultMemberArrayHolder.class, ConversationToken.class);
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[3];
		args[0] = family;
		FaultMemberArrayHolder ffArrayHolder = new FaultMemberArrayHolder();
		args[1] = ffArrayHolder;
		args[2] = token;
		conversationInterceptor.invoke(methodToInvoke, this, args);
		return ffArrayHolder.getFaultMembers();
		
	}
	
	public ConversationTokenProvider privateGetFaultMembersForFaultFamily(FaultFamily ff, FaultMemberArrayHolder resultHolder, ConversationToken token)
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(token);
		FaultMemberService service = TmcdbContextFactory.INSTANCE.getFaultMemberService();
		List<FaultMember> members = service.findByFaultFamily(ff);
		resultHolder.setFaultMembers(members.toArray(new FaultMember[0]));
		return retVal;
	}
	
	private class FaultCodeArrayHolder
	{
		private FaultCode[] faultCodes;
		public FaultCode[] getFaultCodes() { return faultCodes; }
		public void setFaultCodes(FaultCode[] codes) { this.faultCodes = codes; }
	}

	public FaultCode[] getFaultCodesForFaultFamily(FaultFamily family, ConversationToken token) throws Exception 
	{
		Method methodToInvoke = AlarmConversationUtils.class.getMethod("privateGetFaultCodesForFaultFamily", FaultFamily.class, FaultCodeArrayHolder.class, ConversationToken.class);
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[3];
		args[0] = family;
		FaultCodeArrayHolder ffArrayHolder = new FaultCodeArrayHolder();
		args[1] = ffArrayHolder;
		args[2] = token;
		conversationInterceptor.invoke(methodToInvoke, this, args);
		return ffArrayHolder.getFaultCodes();
		
	}
	
	public ConversationTokenProvider privateGetFaultCodesForFaultFamily(FaultFamily ff, FaultCodeArrayHolder resultHolder, ConversationToken token)
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(token);
		FaultCodeService service = TmcdbContextFactory.INSTANCE.getFaultCodeService();
		List<FaultCode> codes = service.findByFaultFamily(ff);
		resultHolder.setFaultCodes(codes.toArray(new FaultCode[0]));
		return retVal;
	}
	
	public void saveOrUpdateDefaultMember(DefaultMember newDefaultMember, boolean conversationCompleted ) throws Exception 
	{
		Method methodToInvoke = AlarmConversationUtils.class.getMethod("privateSaveOrUpdateDefaultMember", DefaultMember.class, Boolean.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[2];
		args[0] = newDefaultMember;
		args[1] = conversationCompleted;
		conversationInterceptor.invoke(methodToInvoke, this, args);
	}

	public ConversationTokenProvider privateSaveOrUpdateDefaultMember(DefaultMember defaultMember, Boolean conversationCompleted) 
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_PENDING);
		if(conversationCompleted) {
			retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		}
		DefaultMemberService service = TmcdbContextFactory.INSTANCE.getDefaultMemberService();
		service.update(defaultMember);
		return retVal;
	}
	
	public void saveOrUpdateLocation(Location newLocation, boolean conversationCompleted) throws Exception
	{
		Method methodToInvoke = AlarmConversationUtils.class.getMethod("privateSaveOrUpdateLocation", Location.class, Boolean.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[2];
		args[0] = newLocation;
		args[1] = conversationCompleted;
		conversationInterceptor.invoke(methodToInvoke, this, args);
	}
	
	public ConversationTokenProvider privateSaveOrUpdateLocation(Location newLocation, Boolean conversationCompleted) 
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_PENDING);
		if(conversationCompleted) {
			retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		}
		LocationService service = TmcdbContextFactory.INSTANCE.getLocationService();
		service.update(newLocation);
		return retVal;
	}
	
	public void saveOrUpdateContact(Contact newContact, boolean conversationCompleted) throws Exception
	{
		Method methodToInvoke = AlarmConversationUtils.class.getMethod("privateSaveOrUpdateContact", Contact.class, Boolean.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[2];
		args[0] = newContact;
		args[1] = conversationCompleted;
		conversationInterceptor.invoke(methodToInvoke, this, args);
	}
	
	public ConversationTokenProvider privateSaveOrUpdateContact(Contact newContact, Boolean conversationCompleted) 
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_PENDING);
		if(conversationCompleted) {
			retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		}
		ContactService service = TmcdbContextFactory.INSTANCE.getContactService();
		service.update(newContact);
		return retVal;
	}
	
	public void saveOrUpdateReductionThreshold(ReductionThreshold newReductionThreshold, boolean conversationCompleted) throws Exception 
	{
		Method methodToInvoke = AlarmConversationUtils.class.getMethod("privateSaveOrUpdateReductionThreshold", ReductionThreshold.class, Boolean.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[2];
		args[0] = newReductionThreshold;
		args[1] = conversationCompleted;
		conversationInterceptor.invoke(methodToInvoke, this, args);	
	}
	
	public ConversationTokenProvider privateSaveOrUpdateReductionThreshold(ReductionThreshold redThreshold, Boolean conversationCompleted) 
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_PENDING);
		if(conversationCompleted) {
			retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		}
		ReductionThresholdService service = TmcdbContextFactory.INSTANCE.getReductionThresholdService();
		service.update(redThreshold);
		return retVal;
	}
	
	public void saveOrUpdateReductionLink(ReductionLink newReductionLink, boolean conversationCompleted) throws Exception 
	{
		Method methodToInvoke = AlarmConversationUtils.class.getMethod("privateSaveOrUpdateReductionLink", ReductionLink.class, Boolean.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[2];
		args[0] = newReductionLink;
		args[1] = conversationCompleted;
		conversationInterceptor.invoke(methodToInvoke, this, args);	
	}
	
	public ConversationTokenProvider privateSaveOrUpdateReductionLink(ReductionLink redLink, Boolean conversationCompleted) 
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_PENDING);
		if(conversationCompleted) {
			retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		}
		ReductionLinkService service = TmcdbContextFactory.INSTANCE.getReductionLinkService();
		service.update(redLink);
		return retVal;
	}
}
