/*******************************************************************************
  * ALMA - Atacama Large Millimeter Array
  * Copyright (c) NRAO - National Radio Astronomy Observatory, 2012
  * (in the framework of the ALMA collaboration).
  * All rights reserved.
  *
  * This library is free software; you can redistribute it and/or
  * modify it under the terms of the GNU Lesser General Public
  * License as published by the Free Software Foundation; either
  * version 2.1 of the License, or (at your option) any later version.
  *
  * This library is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  * Lesser General Public License for more details.
  *
  * You should have received a copy of the GNU Lesser General Public
  * License along with this library; if not, write to the Free Software
  * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 
*******************************************************************************/
package alma.obops.tmcdbgui.utils.conversation;

import java.lang.reflect.Method;
import java.util.List;

import alma.acs.tmcdb.AcsService;
import alma.acs.tmcdb.Configuration;
import alma.acs.tmcdb.Manager;
import alma.obops.dam.config.TmcdbContextFactory;
import alma.obops.dam.tmcdb.service.AcsServiceService;
import alma.obops.dam.tmcdb.service.ComputerService;
import alma.obops.dam.tmcdb.service.ManagerService;
import alma.obops.dam.tmcdb.service.SwConfigurationService;
import alma.obops.dam.utils.ConversationInterceptor;
import alma.obops.dam.utils.ConversationTokenProvider;
import alma.obops.dam.utils.ConversationTokenProviderAdapter;
import alma.obops.dam.utils.ConversationTokenProvider.ConversationToken;

/**
 * @author sharring
 *
 */
public class AcsServiceConversationUtils
{
	private static AcsServiceConversationUtils singletonInstance;

	private AcsServiceConversationUtils() 
	{
	}

	public static synchronized AcsServiceConversationUtils getInstance()
	{
		if(null == singletonInstance)
		{
			singletonInstance = new AcsServiceConversationUtils();
		}

		return singletonInstance;
	}
	
	public void hydrateAcsServices(Configuration configuration)  throws Exception
	{
		Method methodToInvoke = AcsServiceConversationUtils.class.getMethod("privateHydrateAcsServices", Configuration.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[1];
		args[0] = configuration;
		conversationInterceptor.invoke(methodToInvoke, this, args);
	}

	public ConversationTokenProvider privateHydrateAcsServices(Configuration config)
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		SwConfigurationService service = TmcdbContextFactory.INSTANCE.getSwConfigurationService();
		service.hydrateAcsServices(config);
		return retVal;
	}
	
	public ConversationTokenProvider privateDeleteAcsService(AcsService serviceToDelete, ConversationToken token) 
	{
		AcsServiceService service = TmcdbContextFactory.INSTANCE.getAcsServiceService();
		service.delete(serviceToDelete);
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(token);
		return retVal;
	}
	
	public void updateManagerRecordsAfterAcsServiceDeletion(AcsService serviceDeleted) throws Exception
	{
		Method methodToInvoke = AcsServiceConversationUtils.class.getMethod("privateUpdateManagerRecordsAfterAcsServiceDeletion", AcsService.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[1];
		args[0] = serviceDeleted;
		conversationInterceptor.invoke(methodToInvoke, this, args);
	}
	
	public ConversationTokenProvider privateUpdateManagerRecordsAfterAcsServiceDeletion(AcsService serviceDeleted)
	{
		AcsServiceService service = TmcdbContextFactory.INSTANCE.getAcsServiceService();
		List<AcsService> acsServices = service.findByConfigurationId(serviceDeleted.getConfiguration().getConfigurationId());
		ManagerService mgrService = TmcdbContextFactory.INSTANCE.getManagerService();
		List<Manager> mgrs = mgrService.findByConfigurationId(serviceDeleted.getConfiguration().getConfigurationId());
		updateManagerRecordsForRemovalOfAcsService(mgrs, acsServices, serviceDeleted.getAcsServiceId(), serviceDeleted.getComputer().getNetworkName());
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		return retVal;
	}

	private void updateManagerRecordsForRemovalOfAcsService(List<Manager> managers, 
			List<AcsService> existingServices, Integer idOfDeletedService, 
			String removedComputerNetworkName) 
	{
		ManagerService mgrservice = TmcdbContextFactory.INSTANCE.getManagerService();
		ComputerService computerService = TmcdbContextFactory.INSTANCE.getComputerService();
		for(Manager mgr : managers) 
		{
			// check to ensure that *all* services running on this computer have been deleted; 
			// if not, then leave manager's serviceDaemons field unchanged!
			boolean otherServicesExistOnComputer = false;
			for(AcsService acsService: existingServices) 
			{
				computerService.hydrate(acsService.getComputer());
				if(!acsService.getAcsServiceId().equals(idOfDeletedService) &&
					acsService.getComputer().getNetworkName().equals(removedComputerNetworkName)) 
				{
					otherServicesExistOnComputer = true; 
					break;
				}
			}
			if(!otherServicesExistOnComputer)
			{
				String oldStr = mgr.getServiceDaemons();
				int indexOfRemovedComputer = oldStr.indexOf(removedComputerNetworkName);
				if(indexOfRemovedComputer != -1) {
					int indexOfLeadingComma = indexOfRemovedComputer - 1;
					String newStr = null;
					if(indexOfLeadingComma != -1) {
						newStr = oldStr.replace("," + removedComputerNetworkName, "");
					} else {
						newStr = oldStr.replace(removedComputerNetworkName, "");
					}
					if(newStr.length() == 1 && newStr.equals(",")) {
						newStr = null;
					}
					mgr.setServiceDaemons(newStr);		
					mgrservice.update(mgr);
				}
			}
		}
	}
	
	public ConversationTokenProvider privateMoveAcsService(AcsService acsService, String oldComputerNetworkName) 
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);

		Configuration config = acsService.getConfiguration();
		ManagerService mgrService = TmcdbContextFactory.INSTANCE.getManagerService();
		List<Manager> mgrs = mgrService.findByConfigurationId(config.getConfigurationId());
		
		AcsServiceService service = TmcdbContextFactory.INSTANCE.getAcsServiceService();
		service.update(acsService);
		
		updateManagersForNewAcsService(acsService);
		AcsServiceService acsServiceService = TmcdbContextFactory.INSTANCE.getAcsServiceService();
		List<AcsService> services = acsServiceService.findByConfigurationId(config.getConfigurationId());
		updateManagerRecordsForRemovalOfAcsService(mgrs, services, acsService.getAcsServiceId(), oldComputerNetworkName);

		return retVal;
	}
	
	public void saveOrUpdateAcsService(AcsService service) throws Exception 
	{
		Method methodToInvoke = AcsServiceConversationUtils.class.getMethod("privateSaveOrUpdateAcsService", AcsService.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[1];
		args[0] = service;
		conversationInterceptor.invoke(methodToInvoke, this, args);
	}

	public ConversationTokenProvider privateSaveOrUpdateAcsService(AcsService acsService) 
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		AcsServiceService service = TmcdbContextFactory.INSTANCE.getAcsServiceService();
		updateManagersForNewAcsService(acsService);
		service.update(acsService);
		return retVal;
	}
	
	private void updateManagersForNewAcsService(AcsService acsService) 
	{
		ManagerService mgrservice = TmcdbContextFactory.INSTANCE.getManagerService();
		List<Manager> mgrs = mgrservice.findByConfigurationId(acsService.getConfiguration().getConfigurationId());
		for(Manager mgr : mgrs) {
			String oldStr = mgr.getServiceDaemons();
			if( !oldStr.contains(acsService.getComputer().getNetworkName()) ) {
				String newStr = oldStr + "," + acsService.getComputer().getNetworkName();
				mgr.setServiceDaemons(newStr);
			}
			mgrservice.update(mgr);
		}
	}
	
	/**
	 * @param service the service to be saved/updated
	 * @param oldComputerNetworkName the computer to which the service was previously assigned
	 */
	public void moveAcsService(AcsService service, String oldComputerNetworkName) throws Exception 
	{
		Method methodToInvoke = AcsServiceConversationUtils.class.getMethod("privateMoveAcsService", AcsService.class, String.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[2];
		args[0] = service;
		args[1] = oldComputerNetworkName;
		conversationInterceptor.invoke(methodToInvoke, this, args);
	}
}
